import os
import sys

from basesdk import utils

LOG = utils.get_logger("proxy_install")


class ServiceManager(object):
    def __init__(self, root, is_upgrade=False):
        self.services_path = root
        self.is_upgrade = is_upgrade
        self.service_list = ["base", "zookeeper", "cps-monitor", "omm-ha",
                             "haproxy", "agentproxy"]

    @staticmethod
    def is_file_exist(file_path):
        if os.path.exists(file_path):
            return True
        LOG.warning("The file %s is not exist" % file_path)
        return False

    def get_progress(self, service, step):
        service_list = self.service_list
        num = len(service_list)
        number = service_list.index(service) + 1
        if step in ["install", "uninstall"]:
            return number * 90 / num + 9
        return 100

    def post_start_service(self, service):
        LOG.info("===begin to post_start %s===" % service)
        post_s = os.path.join(self.services_path, service,
                              "post_start.sh")
        if not self.is_file_exist(post_s):
            LOG.info("The Post_start of  %s not exsit" % service)
            return
        if self.is_upgrade:
            ret, _ = utils.run_cmd(['sh', post_s, "upgrade"], timeout=1800)
        else:
            ret, _ = utils.run_cmd(['sh', post_s], timeout=1800)
        utils.check_result(ret, "Post_start %s failed" % service, LOG)
        LOG.info("Post_start %s successfully, " % service)

    def install_service(self, service):
        LOG.info("===begin to install %s===" % service)
        install = os.path.join(self.services_path, service, "install.sh")
        if not self.is_file_exist(install):
            msg = "The install file of  %s not exsit" % service
            utils.check_result(1, msg, LOG)
        if self.is_upgrade:
            ret, _ = utils.run_cmd(['sh', install, "upgrade"], timeout=1800)
        else:
            ret, _ = utils.run_cmd(['sh', install], timeout=1800)
        utils.check_result(ret, "Install %s failed" % service, LOG)
        LOG.info("Install %s successfully, " % service)

    def uninstall_service(self, service):
        LOG.info("===begin to uninstall %s===" % service)
        uninstall = os.path.join(self.services_path, service, "uninstall.sh")
        if not self.is_file_exist(uninstall):
            LOG.info("The uninstall of  %s not exsit" % service)
            return
        if self.is_upgrade:
            ret, _ = utils.run_cmd(['sh', uninstall, "upgrade"], timeout=1800)
        else:
            ret, _ = utils.run_cmd(['sh', uninstall], timeout=1800)
        utils.check_result(ret, "Uninstall %s failed" % service, LOG)
        if not self.is_upgrade:
            uninstall_dir = os.path.join(self.services_path, service)
            ret, _ = utils.run_cmd(['rm', '-rf', uninstall_dir])
            msg = "Delete %s uninstall dir failed" % service
            utils.check_result(ret, msg, LOG)
        LOG.info("Uninstall %s successfully, " % service)

    def config_service(self, service):
        LOG.info("===begin to config %s===" % service)
        config = os.path.join(self.services_path, service, "config.sh")
        if not self.is_file_exist(config):
            LOG.info("The config of  %s not exsit" % service)
            return
        ret, _ = utils.run_cmd(['sh', config], timeout=1800)
        utils.check_result(ret, "Config %s failed" % service, LOG)
        LOG.info("Config %s successfully, " % service)

    def start_service(self, service):
        LOG.info("===begin to start %s===" % service)
        start = os.path.join(self.services_path, service, "start.sh")
        if not self.is_file_exist(start):
            LOG.info("The start of  %s not exsit" % service)
            return
        if self.is_upgrade:
            ret, _ = utils.run_cmd(['sh', start, "upgrade"], timeout=1800)
        else:
            ret, _ = utils.run_cmd(['sh', start], timeout=1800)
        utils.check_result(ret, "Start %s failed" % service, LOG)
        LOG.info("Start %s successfully, " % service)

    def deploy_service(self, service):
        self.install_service(service)
        self.config_service(service)
        self.start_service(service)
        self.post_start_service(service)
        progress = self.get_progress(service, "install")
        msg = "Successfully Install %s" % service
        utils.print_and_judge_progress(progress, True, msg)

    def remove_service(self, service):
        self.uninstall_service(service)
        progress = self.get_progress(service, "uninstall")
        msg = "Successfully Uninstall %s" % service
        utils.print_and_judge_progress(progress, True, msg)

    def deploy(self):
        service_list = self.service_list
        LOG.info("The deploy service list:  %s" % service_list)
        for service in service_list:
            self.deploy_service(service)
        msg = "Successfully Installed all services."
        LOG.info(msg)
        utils.print_and_judge_progress(100, True, msg)

    def remove(self):
        self.service_list.reverse()
        LOG.info("The installed service list:  %s" % self.service_list)
        for service in self.service_list:
            self.remove_service(service)
        msg = "Successfully Uninstall all services."
        LOG.info(msg)
        utils.print_and_judge_progress(100, True, msg)


if __name__ == '__main__':
    if len(sys.argv) == 4:
        srv_man = ServiceManager(sys.argv[1], True)
    else:
        srv_man = ServiceManager(sys.argv[1])
    action = sys.argv[2].lstrip('-')
    if action == "install":
        srv_man.deploy()
    elif action == "uninstall":
        srv_man.remove()
    sys.exit(0)
