#!/bin/bash
export PATH=$PATH:/usr/sbin
LOG_CONFIG="/opt/huawei-data-protection/agentproxy/conf/rsyslog_proxy.conf"

###############################################################
# 功能说明: 公共方法
# 约束: conf配置为[section]下key=value
##############################################################
function read_ini_file()
{
    local version=`awk -F '=' '/\['$2'\]/{a=1}a==1&&$1~/'$1'/{print $2;exit}' $3`
 	  echo "${version}" | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

###############################################################
# 功能说明: 日志打印公共方法
# 约束: 通过source 导入该工具文件时，需要在其脚本中定义 LOG_TAG 变量
# 约束值: LOG_TAG; 日志标记
##############################################################
function LOG()
{
    local log_file=$(read_ini_file "${LOG_TAG}" "LOG_FILE" "${LOG_CONFIG}")
    [[ "$3" -ne 0 ]] && echo "[$1] ${*:4}"
    mkdir -p $(dirname ${log_file})
    echo "$(date "+%Y-%m-%d %H:%M:%S.%N" | cut -b 1-23) $1 ${LOG_TAG} [pid:$$] [$(basename $0):$2] ${*:4}" >> ${log_file}
}
shopt -s expand_aliases
alias log_error='LOG ERROR ${LINENO} 0'
alias log_info='LOG INFO ${LINENO} 0'
alias log_warn='LOG WARNING ${LINENO} 0'
alias log_debug='LOG DEBUG ${LINENO} 0'
alias echo_error='LOG ERROR ${LINENO} 1'
alias echo_info='LOG INFO ${LINENO} 1'
alias echo_warn='LOG WARNING ${LINENO} 1'
alias echo_debug='LOG DEBUG ${LINENO} 1'

function CHECK_RESULT()
{
    [[ "$1" -ne 0 ]] && log_error "$2" && exit 1
}

function ECHO_RETURN()
{
    [[ "$1" -ne 0 ]] && echo_error "$2" && exit 1
}

###############################################################
# 功能说明: 公共方法-->命令重试执行
# 入    参: $1 cmd
#          $2 重试次数
#          $3 重试间隔
##############################################################
function cmd_retry()
{
    if [[ $# -ne 3 ]]; then
        return 2
    fi
    cmd="$1"
    retry_count="$2"
    retry_interval="$3"
    for i in $(seq "${retry_count}")
    do
        if ${cmd}; then
            return 0
        fi
        if [ "$i" -ne "${retry_count}" ]; then
            sleep "${retry_interval}"
        fi
    done
    return 1
}

###############################################################
# 功能说明: 公共方法-->配置或者zookeeper acl
# 入    参: $1 rollback为解除，其他为加权
##############################################################
function config_zookeeper()
{
    ip_list=`get_info.py zookeeper_nodes`
    proxython -c "from basesdk import reset_acl;exit(reset_acl.config_zookeeper(\"${ip_list}\",\"$1\"))"
    return $?
}

######################################################################
# 功能说明  : 创建软连接
# 入    参:    :  path
######################################################################
function create_link()
{
    chmod 500 $1
    name=$(basename $1 .sh)
    rm -f /usr/bin/${name}
    ln -sf $1 /usr/bin/${name}
    chown -h agentproxy:agentproxy /usr/bin/${name}
}

function get_pwd()
{
    pwd=$1
    conf=$2
    if [[ -z "${pwd}" || -z ${conf} ]]; then
        log_error "pwd or conf not exist"
        return 1
    fi
    grep ${pwd} ${conf} | cut -d= -f2 | sed "s/^[ \t]*//g" | sed "s/[ \t]*$//g"
}

function check_password() {
    PARA=$1
    echo "${PARA}" | grep -wE '^[0-9a-zA-Z ~!@#$%^&*()_+=-]{1,}$' 1>/dev/null 2>&1
    if [ $? -ne 0 ]; then
        return 1
    fi
    return 0
}

######################################################################
# 功能说明  : 启动服务
######################################################################
function start_service()
{
    $1Control -A RESTART
    return $?
}

function service_control()
{
    cps-monitorControl -S $1
    omm-haControl -S $1
    haproxyControl -S $1
    agentproxyControl -S $1
    if [[ $1 == "RUN" ]];then
        sleep 15s
        om_status=`omm-haControl -CS CompontState | awk -F ' ' '{print $NF}' | sed -n '2p'`
        if [[ ${om_status} == "standby" ]];then
            zookeeperControl -s $1
            return 0
        fi
    fi
    zookeeperControl -S $1
    v_zookeeperControl -S $1
    log_info "$1 services successfully."
}

function check_services_status()
{
    agentproxyControl -CS CompontState | grep -q "running"
    if [[ $? -ne 0 ]];then
        log_error "agentproxy service abnormal."
        return 1
    fi

    haproxyControl -CS CompontState | grep -q "running"
    if [[ $? -ne 0 ]];then
        log_error "haproxy service abnormal."
        return 1
    fi

    cps-monitorControl -CS CompontState | grep -q "running"
    if [[ $? -ne 0 ]];then
        log_error "cps-monitor service abnormal."
        return 1
    fi

    om_status=`omm-haControl -CS CompontState | grep -Eq "active|standby" | awk -F ' ' '{print $NF}'`
    if [[ $? -ne 0 ]];then
        log_error "omm-ha service abnormal."
        return 1
    fi

    zk_status=`zookeeperControl -CS CompontState | awk -F ' ' '{print $NF}' | sed -n '2p'`
    if [[ ${om_status} == 'active' ]];then
        v_zk_status=`v_zookeeperControl -CS CompontState | awk -F ' ' '{print $NF}' | sed -n '2p'`
        if [[ ${zk_status} == "dead" || ${v_zk_status} == "dead" ]];then
            log_error "zookeeper service abnormal."
            return 1
        fi
    else
        v_zookeeperControl -S STOP
        if [[ ${zk_status} == "dead" ]];then
            log_error "zookeeper service abnormal."
            return 1
        fi
    fi
    log_info "services status normal."
    return 0
}

function show_cmd_help()
{
    echo -e "\nUsage:"
    echo "      -CS CompontState    Check $1 state."
    echo "      -A RESTART          Restart $1."
    echo "      -S RUN              Start $1."
    echo -e "      -S STOP             Stop $1.\n"
}

######################################################################
# 功能说明  : 读写权限 配置文件/日志文件/业务数据文件
######################################################################
function set_rw_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 750 "$file"
        else
            chmod 640 "$file"
        fi
    done
}

######################################################################
# 功能说明  : 只读权限 压缩后的日志
######################################################################
function set_readonly_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 750 "$file"
        else
            chmod 440 "$file"
        fi
    done

}

######################################################################
# 功能说明  : 执行权限 包含可执行文件的目录和文件
######################################################################
function set_exec_permission()
{
    chown -hR agentproxy:agentproxy $1
    chmod -R 550 $1
}

######################################################################
# 功能说明  : 通用权限
######################################################################
function set_common_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 750 "$file"
        else
            chmod 550 "$file"
        fi
    done
}

######################################################################
# 功能说明  : 最小执行权限 包含可执行文件的目录和文件
######################################################################
function set_least_exec_permission()
{
    chown -hR agentproxy:agentproxy $1
    chmod -R 500 $1
}


######################################################################
# 功能说明  : root用户最小执行权限 包含可执行文件的目录和文件
######################################################################
function set_root_exec_permission()
{
    chown -hR root:root $1
    for file in $(find "$1");do
        chmod 500 "$file"
    done
}


######################################################################
# 功能说明  : 最小读写权限 配置文件/日志文件/业务数据文件
######################################################################
function set_least_rw_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 700 "$file"
        else
            chmod 600 "$file"
        fi
    done
}

######################################################################
# 功能说明  : 最小只读权限 压缩后的日志
######################################################################
function set_least_readonly_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 700 "$file"
        else
            chmod 400 "$file"
        fi
    done
}


######################################################################
# 功能说明  : 通用权限
######################################################################
function set_least_common_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ -d "$file" ];then
            chmod 700 "$file"
        else
            chmod 500 "$file"
        fi
    done
}

########################################################################
# 功能说明  : 设置某目录下以***结尾的所有文件权限，参数依次为:目录，文件后缀，权限
########################################################################
function set_permission_for_file()
{
    chown -hR agentproxy:agentproxy $1
    for file in `find $1 -name "*.$2"`
    do
        if [ -f ${file} ]
        then
            chmod $3 ${file}
        fi
    done
}

########################################################################
# 功能说明  : 设置某目录下不可执行文件权限400，其他500（包括目录）
########################################################################
function set_least_permission()
{
    chown -hR agentproxy:agentproxy $1
    for file in $(find "$1");do
        if [ ! -x "${file}" ];then
            chmod 400 "$file"
        else
            chmod 500 "$file"
        fi
    done
}