# -*- coding: utf-8 -*-
import datetime
import getpass
import ipaddress
import logging
import os
import re
import subprocess
import subprocess as cmd
import time

import paramiko
import requests
import six
from FSSecurity import crypt as fscrypt
from FSSysconf import netPolicyParse
from prettytable import PrettyTable
from cinder.backup.ebackupconfig import write_ca_cert_content_to_file

logger = logging.getLogger('__EBACKUP_CONFIG__')
logger.setLevel(level=logging.DEBUG)
logger.propagate = False
logger.handlers = []

log_file_path = os.path.abspath('/var/log/ebackup_install_config.log')
if not os.path.isfile(log_file_path):
    os.mknod(log_file_path)
if os.path.getsize(log_file_path) > 10485760:
    with open(log_file_path) as fp:
        fp.truncate()

handler = logging.FileHandler(log_file_path)
handler.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s [%(levelname)s] at '
                              '%(filename)s,%(lineno)d: %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)
CODING_LIST = ['UTF-8', 'GBK', 'ISO-8859-1']
VALID_VOLUME_DRIVER = ['cinder.volume.drivers.dsware.HuaweiDswareDriver',
                       'cinder.volume.drivers.huawei.vrm.vrm_driver.VRMDriver',
                       'cinder.volume.drivers.huawei.HuaweiISCSIDriver',
                       'cinder.volume.drivers.dsware.HuaweiDswareIscsiDriver']

CPS_CA_CERT = "/etc/FSSecurity/cacert/cps-client.crt"

global G_CPS_BASE_URL
global G_SUPPORTED_TEMPLATES
global G_CLOUD_SCENE
global G_CPS_NODES
global G_CHECK1
global G_CHECK2
global G_CHECK3
global G_CHECK4
global G_AZ_LIST
G_SUPPORTED_TEMPLATES = []
G_AZ_LIST = []
LINE_WORD_NUM = 60
HTTP_TIMEOUT = 60 * 2


def get_cps_server_base_url():
    base_url = ""
    try:
        floating_ip = netPolicyParse. \
            getComponentFloatingIp(compName="cps-server")
        floating_port = netPolicyParse. \
            getComponentFloatingPort(compName="cps-server")
        if floating_ip.find(":") > 0:
            base_url = '[' + floating_ip + ']' + ":" + floating_port
        else:
            base_url = floating_ip + ":" + floating_port
        logger.info("Get base url by interface: %s" % base_url)
        return base_url.replace(' ', '')
    except Exception as e:
        logger.info("not found cps-server maybe it's cascaded %s!" % e)
        file_path = "/etc/huawei/fusionsphere/cfg/serverIp.ini"
        if os.path.exists(file_path):
            fp = open(file_path, 'r')
            logger.info('use config file : %s' % fp)
            while True:
                line = fp.readline().strip('\n')
                if not line:
                    break
                key_value = line.split("=")
                if key_value[0].strip() == 'serverip':
                    base_url = key_value[1].strip()
                    base_url = base_url.replace("https://", "")
                    logger.info("Get base url from serverIP.ini:%s" % base_url)
            fp.close()
            return base_url.replace(' ', '')
        else:
            logger.error(file_path + "is not existed.")
            return ""


def check_ip_version(ip):
    return 4 if -1 == ip.find(':') else 6


def _is_valid_ip(ip):
    try:
        return ipaddress.ip_address(six.text_type(ip).strip())
    except ValueError:
        return False


def ping_ip(ip):
    if not _is_valid_ip(ip):
        return False
    try:
        ip_version = check_ip_version(ip)
        if ip_version not in [4, 6]:
            logger.error("IP {} is not invalid.".format(ip))
            return False
        ping_cmd = 'ping' if ip_version == 4 else 'ping6'
        ret = subprocess.run([ping_cmd, ip, '-c2', '-w', '10'], shell=False).returncode

        if 0 != ret:
            logger.error("Ping {} failed.".format(ip))
            return False
        return True
    except Exception as e:
        logger.error("Exception occurs.The reason is:" + str(e))
        return False


def tostr(args):
    """
    字符串转换
    :param args: 待转换的对象
    :return: 转换后的对象
    """
    if isinstance(args, bytes):
        for encoding_item in CODING_LIST:
            try:
                return args.decode(encoding=encoding_item)
            except UnicodeDecodeError as e:
                logger.error("transfer bytes to str error: " + str(e))
        logger.error("transfer bytes to str error: %s" % args)
        return args
    elif isinstance(args, dict):
        return {tostr(key): tostr(value)
                for key, value in list(args.items())}
    elif isinstance(args, list):
        return [tostr(element) for element in args]
    else:
        return args


class Ssh(object):
    @classmethod
    def ssh_create_client(cls, ip_num, username, passwd,
                          port=22, timeout=5, pub_key=None,
                          status=True, width=80):
        '''创建一个SSH的客户端,可以保持连接,
        支持密匙登录，需要密匙登录时，传入需要的pub_key即可'''
        logger.info("Begin to create ssh client on " + ip_num)
        if (not ip_num) or (not ping_ip(ip_num)):
            err_msg = 'Create ssh client failed, ' \
                      'ip is none or ip cannot be pinged:%s' % ip_num
            logger.error(err_msg)
            raise Exception(err_msg)
        trans = None
        try:
            ssh_client = dict()
            trans = paramiko.Transport((ip_num, port))
            if not pub_key:
                trans.connect(username=username, password=passwd)
            else:
                with open(pub_key) as _fp:
                    key = tostr(paramiko.RSAKey.from_private_key(_fp))
                    trans.connect(username=username, password=passwd, pkey=key)
            trans.set_keepalive(30)
            channel = trans.open_session()
            channel.settimeout(timeout)
            channel.get_pty(width=width)
            channel.invoke_shell()
            stdout = channel.makefile('r', -1)
            ssh_client['ip'] = ip_num
            ssh_client['port'] = port
            ssh_client['username'] = username
            ssh_client['timeout'] = timeout
            ssh_client['width'] = width
            ssh_client['client'] = trans
            ssh_client['channel'] = channel
            ssh_client['stdout'] = stdout
            time.sleep(2)
            if status:
                cls.get_prompt(ssh_client)
            logger.info("Create ssh client on success.".format(ip_num))
            return ssh_client

        except Exception as execpts:
            try:
                if trans:
                    trans.close()
            except Exception as ex:
                logger.error("failed to close connection after"
                             " creating SSH client failed: {}.".format(ex))
            err_msg = 'create ssh client failed. ip:%s, ' \
                      'username:%s, port:%s, pub_key:%s' % (
                          ip_num, username, port, pub_key)
            if not passwd:
                err_msg += ', and passwd is none'
            if str(execpts):
                err_msg += '. the error is:%s' % str(execpts)
            logger.error(err_msg)
            raise Exception(err_msg)

    @classmethod
    def ssh_close(cls, ssh_client):
        '''关闭SSH连接'''
        try:
            ssh_client['client'].close()
        except Exception:
            err_msg = 'close ssh client failed'
            logger.error(err_msg)
            raise Exception(err_msg)

    @classmethod
    def __format_output1(cls, listout, cmds):
        if not listout:
            return listout
        if 0 != len(listout[0]):
            listout = listout[1:]
        if listout and 0 != len(listout[-1]):
            listout = listout[:-1]
        list_out = []
        for out in listout:
            out = out.replace('\n', '').strip()
            if cmds in out or out == '' or \
                    out.__contains__('last cmd result:'):
                continue
            list_out.append(out)
        return list_out

    @classmethod
    def ssh_exec_command_return_list(cls, ssh_client, cmds, timeout=60,
                                     retry_times=0):
        if (not ssh_client) or (not cmds) or \
                (not isinstance(ssh_client, dict)):
            err_msg = 'params is invalid'
            raise Exception(err_msg)
        retry_times = retry_times if retry_times else retry_times + 1
        err_msg = ""
        now_time = datetime.datetime.now()
        for retry_time in range(retry_times):
            try:
                logger.info("execute ssh_exec_command_return_list, "
                            "total times: {}, current times: {}.".
                            format(retry_times, (retry_time + 1)))
                ssh_client['channel'].send(cmds +
                                           ";echo last cmd result: $?\n")
                recv_str = ""
                while not re.search("last cmd result: \d+", recv_str):
                    time.sleep(5)
                    if ssh_client['channel'].closed:
                        logger.error("channel is closed, "
                                     "please check network connection.")
                        raise Exception("channel is closed")
                    if (now_time + datetime.timedelta(minutes=int(timeout))) \
                            < datetime.datetime.now():
                        logger.error("ssh command has been executed "
                                     "exceed {} minutes".format(timeout))
                        raise Exception("execute ssh command timeout")
                    if not ssh_client['channel'].recv_ready():
                        continue
                    recv_str += tostr(ssh_client['channel'].recv(65535))
                    recv_str = recv_str.replace(' \r', '\r')
                    recv_str = recv_str.replace('\r', '')
                    recv_str = recv_str.replace(";echo last cmd result: $?",
                                                "")
                recv_str = recv_str.replace(ssh_client['prompt'], '')
                recv_str = recv_str.replace('\n\n', '\n')
                recv_str = recv_str.replace('\n', '\n\r')
                recv_str = recv_str.replace("last cmd result: 0", "")
                recv_str = recv_str.replace("last cmd result: 1", "")
                listout = recv_str.split('\r')
                list_out = cls.__format_output1(listout, cmds)
                logger.info("execute ssh_exec_command_return_list success.")
                return list_out
            except Exception as ex:
                err_msg = "execute ssh command and " \
                          "get the result on host: {} failed: {}" \
                    .format(ssh_client.get("ip"), ex)
                logger.error(err_msg)
                time.sleep(5)
        else:
            logger.error(err_msg)
            raise Exception(err_msg)

    @classmethod
    def __format_output2(cls, listout):
        if 1 == len(listout) and '' == listout[0]:
            return listout
        if not listout:
            return listout
        if 0 == len(listout[0]):
            listout = listout[1:]
        if 0 == len(listout[-1]):
            listout = listout[:-1]
        return listout

    @classmethod
    def __get_command_out(cls, ssh_client, cmd, expect, timeout):
        try:
            recv_str = ""
            ssh_client['channel'].send(cmd + '\n')
            times = 0
            while not recv_str.__contains__(expect):
                if times >= timeout:
                    break

                time.sleep(0.5)
                times += 0.5
                if ssh_client['channel'].closed:
                    recv_str += tostr(ssh_client['channel'].recv(65535))
                    logger.error("channel is closed")
                    break
                if not ssh_client['channel'].recv_ready():
                    continue
                recv_str += tostr(ssh_client['channel'].recv(65535))
                recv_str = recv_str.replace(' \r', '\r')
                recv_str = recv_str.replace('\r', '')

            return recv_str, None
        except Exception as execpts:
            msg = 'exec ssh command on host:%s failed:%s' % (
                ssh_client['ip'], str(execpts))
            logger.error(msg)
            err_msg = Exception(msg)
            return "", err_msg

    @classmethod
    def ssh_send_command(cls, ssh_client, cmd, expect, timeout, retry_times=0):
        if (not ssh_client) or (not isinstance(ssh_client, dict)) or (not cmd):
            err_msg = 'exec ssh command failed, params is error.'
            logger.error(str(err_msg))
            raise Exception(err_msg)
        retry_times = retry_times if retry_times else retry_times + 1
        err_msg = Exception('execute ssh_send_command failed.')
        for attempNum in range(retry_times):
            logger.info("execute ssh_send_command, total times: "
                        "{}, current times: {}."
                        .format(retry_times, (attempNum + 1)))
            recv_str, err_msg = cls.__get_command_out(ssh_client, cmd,
                                                      expect, timeout)
            if not recv_str.__contains__(expect):
                err = \
                    'exec ssh command on host:' + \
                    ssh_client['ip'] + ' cannot find expect in ' + \
                    str(timeout) + ' seconds for cmd'
                logger.error(err)
                err_msg = Exception(err)
                time.sleep(5)
            else:
                break
        else:
            logger.error(err_msg)
            raise err_msg
        recv_str = recv_str.replace(ssh_client.get('prompt', ''), '')
        recv_str = recv_str.replace('\n\n', '\n')
        recv_str = recv_str.replace('\n', '\n\r')
        listout = recv_str.split('\r')
        list_out = cls.__format_output2(listout)
        logger.info("execute ssh_send_command success.")
        return list_out

    @classmethod
    def ssh_list_to_str(cls, listout):
        '''将列表结果拼接成[一个]字符串'''
        str_out = ""
        # 合并成字符串
        try:
            for line in listout:
                str_out += line
            return str_out
        except Exception:
            err_msg = 'commend return value to str failed'
            logger.error(err_msg)
            raise Exception(err_msg)

    @classmethod
    def get_prompt(cls, ssh_client):
        """
        如下方法只是适用于下发\n
        :param ssh_client:
        :return:
        """
        try:
            ssh_client['channel'].send('\n')
            recv_str = ""
            num = 0
            while not (recv_str.endswith("$ ") or recv_str.endswith("# ")):
                num += 1
                time.sleep(1)
                if num > 10:
                    logger.info("Number of times: {},"
                                " received shell prompt: {}."
                                .format(num, recv_str))
                    break
                if not ssh_client['channel'].recv_ready():
                    continue
                recv_str = recv_str + tostr(ssh_client['channel'].recv(65535))
            if not recv_str:
                logger.error("The obtained shell prompt is empty "
                             "or get the shell prompt timeout")
                ssh_client['prompt'] = ""
            else:
                logger.info("received shell prompt: {}.".format(recv_str))
                listout = recv_str.split('\r\n')
                ssh_client['prompt'] = listout[-1]
        except Exception as ex:
            logger.error("Try to get the shell prompt failed,"
                         " Error Information: {}.".format(ex))
            ssh_client['prompt'] = ""

    @classmethod
    def get_ssh_client(cls, ip, user, user_pwd, root_pwd):
        ssh_client = None
        try:
            ssh_client = cls.ssh_create_client(ip, user, user_pwd, timeout=30)
            cls.ssh_send_command(ssh_client, 'su - root', 'Password:', 60)
            cls.ssh_send_command(ssh_client, root_pwd, '#', 60)
            logger.info("Create ssh client for {} successfully.".format(ip))
            return ssh_client
        except Exception as e:
            cls.close_ssh_clinet(ssh_client)
            logger.error("Create ssh client for {0} failed:{1}".
                         format(ip, str(e)))
            return None

    @classmethod
    def close_ssh_clinet(cls, ssh_client):
        try:
            cls.ssh_close(ssh_client)
        except Exception as e:
            logger.info("Close ssh client failed,the reason is %s" % str(e))


def check_cps_admin_password(cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info(f"Begin to check the password of the {cps_username} user.")
    url = f"https://{G_CPS_BASE_URL}/cps/v1/haproxylocalurl"
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    try:
        rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT, timeout=(HTTP_TIMEOUT, HTTP_TIMEOUT))
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            if -1 != str(rsp_dict).lower().find("failed"):
                logger.error("Authenticate Failed!")
                return False
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")

        logger.info(cps_username + "Authenticate successfully.")
        return True
    except Exception as e:
        logger.error("Authenticate Failed!The reason is: " + str(e))
        return False


def get_cinder_backup_templates(cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info("Begin to query all cinder backup template.")
    cinder_backup_templates = []
    uri = 'cps/v1/services/all/componenttemplates?commit_state=commited'
    url = "https://%s/%s" % (G_CPS_BASE_URL, uri)
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }

    try:
        rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            for service in rsp_dict["templates"]:
                if -1 == service["name"].find("cinder-backup"):
                    continue
                cinder_backup_templates.append(service["name"])
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")
        if 0 == len(cinder_backup_templates):
            logger.debug("There is no cinder backup template.")
            return True, []
        logger.info("Query all cinder backup template successfully.")
        return True, cinder_backup_templates
    except Exception as e:
        logger.error(f"Query all cinder backup template failed. The reason is {str(e)}")
        return False, []


def get_supported_backup_templates(cinder_backup_templates,
                                   cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info("Begin to query supported backup template.")
    supported_templates = []
    m_url = "https://%s/cps/v1/services/cinder/componenttemplates/" \
            "EACH_TEMPLATE/params?commit_state=commited" % G_CPS_BASE_URL
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    try:
        for template in cinder_backup_templates:
            logger.info('start check template %s' % template)
            url = m_url.replace("EACH_TEMPLATE", template)
            logger.debug("The url is " + url)
            rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
            if rsp.status_code == 200:
                template_params = rsp.json()["cfg"]
                volume_driver = template_params.get('volume_driver', '')
                other_storage_cfg = template_params. \
                    get('other_storage_cfg', '')
                logger.debug('volume_driver: %s' % volume_driver)
                logger.debug('other_storage_cfg: %s' % other_storage_cfg)
                valid_volume_driver = VALID_VOLUME_DRIVER
                other_volume_driver = ''
                if other_storage_cfg:
                    for poolid, poollist in other_storage_cfg.items():
                        for cfgkey, poolcfg in poollist.items():
                            if poolcfg.get('volume_driver', ''):
                                other_volume_driver = poolcfg. \
                                    get('volume_driver', '')
                logger.debug('other_volume_driver: %s' % other_volume_driver)
                if volume_driver in valid_volume_driver or \
                        (other_volume_driver in valid_volume_driver):
                    supported_templates.append(template)
                else:
                    logger.debug('There is not valid volume driver %s in'
                                 ' template %s' % (volume_driver, template))
            else:
                logger.error("Response status code is not 200.The url is"
                             + url)
                raise Exception("Response status code is not 200.")
        if 0 == len(supported_templates):
            logger.error("There are no supported templates.")
            return False, []
        logger.info("The supported templates are: " + str(supported_templates))
        return True, supported_templates
    except Exception as e:
        logger.error("Query supported template failed,the reason is " + str(e))
        return False, []


def get_ebackup_driver_node(supported_templates, cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info("Begin to query driver node by supported template.")
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    ebackup_driver_nodes = {}

    try:
        for template in supported_templates:
            url = "https://%s/cps/v1/instances?service=cinder&template=%s" \
                  % (G_CPS_BASE_URL, template)
            rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
            if rsp.status_code == 200:
                instance_list = rsp.json()["instances"]
                for host in instance_list:
                    omip = host["omip"]
                    if omip is not None and omip != '':
                        if omip not in ebackup_driver_nodes:
                            ebackup_driver_nodes.update({omip: [template]})
                        else:
                            ebackup_driver_nodes[omip].append(template)
            else:
                logger.error("Response status code is not 200.Url is" + url)
                raise Exception("Response status code is not 200.")

        if 0 == len(ebackup_driver_nodes):
            logger.error("There is no ebackup driver node.")
            return False, {}
        logger.info("Query eBackup driver node successfully."
                    "The eBackup driver node are:" + str(ebackup_driver_nodes))
        return True, ebackup_driver_nodes
    except Exception as e:
        logger.error("Query eBackup driver node failed,the reason: " + str(e))
        return False, {}


def get_cps_server_node(cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info("Begin to query cps server nodes.")
    uri = 'cps/v1/instances?service=cps-server&template=cps-server'
    url = "https://%s/%s" % (G_CPS_BASE_URL, uri)
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    cps_server = []
    try:
        rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)

        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            for cps in rsp_dict["instances"]:
                omip = cps["omip"]
                if omip is not None and omip != '':
                    cps_server.append(omip)
        else:
            logger.info("The cps-server service name is cps")
            uri = 'cps/v1/instances?service=cps&template=cps-server'
            url = f"https://{G_CPS_BASE_URL}/{uri}"
            rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
            if rsp.status_code == 200:
                rsp_dict = rsp.json()
                for cps in rsp_dict["instances"]:
                    omip = cps["omip"]
                    if omip is not None and omip != '':
                        cps_server.append(omip)
            else:
                logger.error("Response status code is not 200.Url is" + url)
                raise Exception("Response status code is not 200.")
        if 0 == len(cps_server):
            logger.error("There is no cps server.")
            return False, []
        logger.info("Query cps server nodes successfully.The cps server are:"
                    + str(cps_server))
        return True, cps_server
    except Exception as e:
        logger.error("Query cps server nodes failed,the reason is " + str(e))
        return False, []


def get_cps_user_and_password(user_dict):
    logger.info('Begin to get the cps user and password')
    while True:
        while True:
            cps_username = six.moves.input('Enter cps username(default:cps_admin)=')
            if cps_username.strip() == '':
                cps_username = 'cps_admin'
                logger.info("Use default cps user:" + cps_username)
            break
        user_dict['cps_username'] = cps_username

        while True:
            cps_password = getpass.getpass('Enter cps passowrd=')
            if cps_password.strip() == '':
                msg = "Cps passowrd can not be empty,please input again."
                logger.info(msg)
            else:
                break
        user_dict['cps_password'] = cps_password
        if check_cps_admin_password(cps_username, cps_password):
            break
        else:
            msg = "Cps username or password is wrong, please input again."
            print(add_color(msg))
            logger.error(add_color(msg))
    logger.info('Get the cps user and password successfully.')
    return user_dict


def get_fsp_user_and_password(user_dict):
    logger.info('Begin to get the fsp user and password')
    while True:
        while True:
            fsp_password = getpass.getpass('Enter fsp password=')
            if fsp_password.strip() == '':
                msg = "Fsp password can not be empty,please input again."
                logger.info(msg)
            else:
                break
        user_dict['fsp_password'] = fsp_password
        while True:
            root_password = getpass.getpass('Enter fsp root password=')
            if root_password.strip() == '':
                msg = "Fsp root password can not be empty,please input again."
                logger.info(msg)
            else:
                break
        user_dict['root_password'] = root_password
        ret_code, msg = check_ssh_login(user_dict['local_omip'],
                                        user_dict['fsp_username'],
                                        fsp_password, root_password, 7)
        if 0 == ret_code:
            break
        elif 3 == ret_code:
            warning_msg = "Warning: The password of the fsp user " \
                          "is about to expire in {} days.".format(msg)
            print(add_color(warning_msg, '33'))
            logger.info(warning_msg)
            break
        elif 1 == ret_code:
            msg = "The fsp password is wrong, please input again."
            print(add_color(msg))
            logger.error(msg)
        elif 2 == ret_code:
            msg = "The fsp password is expired, " \
                  "please modify fsp password and input again."
            print(add_color(msg))
            logger.error(msg)
        elif 4 == ret_code:
            msg = "The root password is wrong, please input again."
            print(add_color(msg))
            logger.error(msg)

    logger.info('Get the cps fsp and password successfully.')
    return user_dict


def get_user_and_password():
    logger.info('Begin to get the user and password')
    user_dict = {
        'cps_username': '',
        'cps_password': '',
        'fsp_username': 'fsp',
        'fsp_password': '',
        'root_password': '',
        'local_omip': ''
    }
    get_cps_user_and_password(user_dict)
    cps_username = user_dict['cps_username']
    cps_password = user_dict['cps_password']
    user_dict['local_omip'] = get_local_omip(cps_username, cps_password)
    get_fsp_user_and_password(user_dict)
    return user_dict


def get_password_expired_days(cmd_outputs):
    expired_days = 999
    try:
        for item in cmd_outputs:
            key = item.split(":")[0].strip()
            value = item.split(":")[1].strip()
            if key.lower() == 'password expires':
                expired_time = value.lower()
                if expired_time == 'never':
                    expired_days = 999
                else:
                    expirt_date = datetime.datetime.strptime(expired_time,
                                                             '%b %d, %Y')
                    now_date = datetime.datetime.now()
                    expired_days = (expirt_date - now_date).days
                logger.info("The password will expire in " + str(expired_time))
                break
        expired_days += 1
        logger.info("The password will expire in {} days".format(expired_days))
        return expired_days
    except Exception as e:
        logger.error("Get the expire time failed.The reason is:" + str(e))
        return expired_days


def check_ssh_login(server_ip, user, user_password,
                    root_password, goto_expire_days=7):
    # 0:success
    # 1:普通用户密码错误
    # 2:普通用户密码过期或者OS语言不是英语
    # 3:普通用户密码即将过期
    # 4:root用户密码错误
    ssh_client = None
    logger.info("Begin to check fsp and root password successfully.")
    try:
        ssh_client = Ssh.ssh_create_client(server_ip, user, user_password)
        logger.info("Login {0} successfully by user {1}".
                    format(server_ip, user))
    except Exception as e:
        logger.error("The password of the {0} user is incorrect on host {1}."
                     "The reason is:{2}".format(server_ip, user, str(e)))
        Ssh.close_ssh_clinet(ssh_client)
        return 1, user

    try:
        Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 15)
        logger.info("Excute su - root on host {0} successfully by user {1}"
                    .format(server_ip, user))
    except Exception as e:
        logger.error("Excute su - root on host {0} failed."
                     "The reason is:{1}".format(server_ip, str(e)))
        Ssh.close_ssh_clinet(ssh_client)
        return 2, ""

    try:
        Ssh.ssh_send_command(ssh_client, root_password, '#', 15)
        logger.info("su - root successfuly on host " + server_ip)
    except Exception as e:
        logger.error("The password of the root user is incorrect on host {0}."
                     "The reason is:{1}".format(server_ip, str(e)))
        Ssh.close_ssh_clinet(ssh_client)
        return 4, 'root'

    logger.info("Begin to check the age of the {} user.".format(user))
    cmd = "chage -l {}".format(user)
    try:
        cmd_output = Ssh.ssh_exec_command_return_list(ssh_client, cmd)
        expired_days = get_password_expired_days(cmd_output)
        if expired_days < 0:
            expired_days = 999
        if expired_days <= goto_expire_days:
            logger.info("The password of the {} user  will expire in {} days."
                        .format(user, expired_days))
            return 3, str(expired_days)
        logger.info("Check fsp and root password successfully.")
        return 0, ""
    except Exception as e:
        logger.error("Check the the age of the {0} user failed."
                     " The reason is:{1}".format(user, str(e)))
        return 0, ""
    finally:
        Ssh.close_ssh_clinet(ssh_client)


def get_local_omip(cps_username, cps_password):
    global G_CPS_BASE_URL
    logger.info("Begin to get fsp local omip.")
    url = "https://%s/cps/v1/hosts" % G_CPS_BASE_URL
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    try:
        (status, host_name) = cmd.getstatusoutput('hostname')
        host_omip = ""
        rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
        if rsp.status_code == 200:
            host_dict = rsp.json()
            for host in host_dict["hosts"]:
                if host_name == host["alias"]:
                    host_omip = host['omip']
                    break
        else:
            logger.error("Response status code is not 200.The url is" + url)
            raise Exception("Response status code is not 200.")

        if 0 == len(host_omip):
            logger.debug("Query local omip failed. Now ask for it.")
            while True:
                host_omip = six.moves.input('Enter local omip=')
                if host_omip.strip() == '':
                    logger.info("Local omip can not be empty,"
                                "please input again.")
                else:
                    break
        logger.info("Query local omip successfully, ip is " + host_omip)
        return host_omip
    except Exception as e:
        logger.error("Query local omip failed,the reason is " + str(e))
        return ""


class DeployInfoInspect():
    def __init__(self, ip, user, fsp_pwd, root_pwd):
        self.ip = ip
        self.user = user
        self.fsp_pwd = fsp_pwd
        self.root_pwd = root_pwd
        self.ssh_client = None
        self.result = {
            'ip': ip,
            'version': '',
            'patch': 'None',
            'keylist': '',
            'err_msg': ''
        }
        self.cinder_path1 = '/usr/lib/python2.7/site-packages/cinder/backup'
        self.cinder_path2 = '/opt/cloud/services/cascading-cinder/venv/lib/' \
                            'python2.7/site-packages/cinder/backup'
        self.cinder_path3 = '/usr/lib64/python2.7/site-packages/cinder/backup'

    def get_ssh_client(self):
        if not ping_ip(self.ip):
            logger.error(self.ip + " can not be pingd.")
            self.result['err_msg'] = "Network is abnormal"
            return False

        try:
            self.ssh_client = Ssh.ssh_create_client(self.ip,
                                                    self.user, self.fsp_pwd)
            logger.info("Login {0} successfully by user {1}".
                        format(self.ip, self.user))
        except Exception as e:
            logger.error(
                "The password of the {0} user is incorrect on host {1}."
                "The reason is:{2}".format(self.ip, self.user, str(e)))
            Ssh.close_ssh_clinet(self.ssh_client)
            self.result['err_msg'] = "The password of fsp user is wrong."
            return False

        try:
            Ssh.ssh_send_command(self.ssh_client, 'su - root', 'Password:', 15)
            logger.info("Excute su - root on host {0} "
                        "successfully by user {1}".format(self.ip, self.user))
        except Exception as e:
            logger.error("Excute su - root on host {0} failed."
                         "The reason is:{1}".format(self.ip, str(e)))
            Ssh.close_ssh_clinet(self.ssh_client)
            self.result['err_msg'] = "The password of fsp user is expired."
            return False

        try:
            Ssh.ssh_send_command(self.ssh_client, self.root_pwd, '#', 15)
            logger.info("su - root successfuly on host " + self.ip)
        except Exception as e:
            logger.error(
                "The password of the root user is incorrect on host {0}."
                "The reason is:{1}".format(self.ip, str(e)))
            Ssh.close_ssh_clinet(self.ssh_client)
            self.result['err_msg'] = "The password of root user is wrong."
            return False
        logger.info("Create ssh client on {} successfully.".format(self.ip))
        return True

    def inspcet_version(self):
        logger.info("Begin to check version.")
        if not self.ssh_client:
            logger.error("Ssh client is invalid.")
            self.result['err_msg'] = 'Internal error:Ssh client is invalid.'
            return False
        try:
            version_file = '/drivers/ebackupversion.conf'
            file1 = self.cinder_path1 + version_file
            file2 = self.cinder_path2 + version_file
            file3 = self.cinder_path3 + version_file

            version_check_cmds = 'version_info=$(' + \
                                 'cat {} 2>/dev/null || '.format(file1) + \
                                 'cat {} 2>/dev/null || '.format(file2) + \
                                 'cat {} 2>/dev/null);'.format(file3) + \
                                 'echo ${version_info}'
            ret = Ssh.ssh_exec_command_return_list(self.ssh_client,
                                                   version_check_cmds)
            logger.info("eBackup Version: " + str(ret))
            if -1 != str(ret).find("eBackupVersion"):
                self.result['version'] = ret[0].split(' ')[0].split('=')[1]
                logger.info("eBackup Driver version is {0} at {1}".
                            format(self.result['version'], self.ip))
            else:
                logger.error("eBackup Driver is not installed at " + self.ip)
                self.result['version'] = "Not installed"
            logger.info("Check patch version completed.")
            return True
        except Exception as e:
            logger.error("Exception occurs: " + str(e))
            self.result['err_msg'] = 'Internal error: exception occurs.'
            return False

    def inspcet_patch_version(self):
        logger.info("Begin to check patch version.")
        if self.result['version'] == "Not installed":
            logger.info("There is no eBackup driver, "
                        "no need to check patch version")
            self.result['patch'] = 'None'
            return True

        if not self.ssh_client:
            logger.error("Ssh client is invalid.")
            self.result['err_msg'] = 'Internal error:Ssh client is invalid.'
            return False

        path_post = '/ebackup_driver_backup/%s/' % self.result['version']
        path1 = self.cinder_path1 + path_post
        path2 = self.cinder_path2 + path_post
        path3 = self.cinder_path3 + path_post
        patch_check_cmds = \
            'version_info=$(' + \
            'grep "Version:" %s/patch_*.info 2>/dev/null || ' % path1 + \
            'grep "Version:" %s/patch_*.info 2>/dev/null || ' % path2 + \
            'grep "Version:" %s/patch_*.info 2>/dev/null); ''' % path3 + \
            '''echo ${version_info} | awk -F "[ |:]" '{print $NF}' '''
        try:
            ret = Ssh.ssh_exec_command_return_list(self.ssh_client,
                                                   patch_check_cmds)
            logger.info("eBackup Patch Version is: " + str(ret))

            if 0 != len(ret):
                self.result['patch'] = ret[0].strip().replace('\n', '')
                logger.info("eBackup Patch Version is: "
                            + str(self.result['patch']))
                logger.info("eBackup Driver patch version is {0} at {1}".
                            format(self.result['patch'], self.ip))
            else:
                logger.info("There is no patch at " + self.ip)
                self.result['patch'] = "None"
            logger.info("Check patch version completed")
            return True
        except Exception as e:
            logger.error("Exception occurs: " + str(e))
            self.result['err_msg'] = 'Internal error: exception occurs.'
            return False

    def inspect_key_list(self):
        logger.info("Begin to check key list.")
        if not self.ssh_client:
            logger.error("Ssh client is invalid.")
            self.result['err_msg'] = 'Internal error:Ssh client is invalid.'
            return False
        try:
            keys = ['ebackup_login_password_v2', 'ebackup_storage_username_v2',
                    'ebackup_storage_password_v2']
            key_list = '/etc/FSSecurity/keylist.ini'
            key_list_check = \
                'grep -w %s %s >/dev/null 2>&1 && ' % (keys[0], key_list) + \
                'grep -w %s %s >/dev/null 2>&1 && ' % (keys[1], key_list) + \
                'grep -w %s %s >/dev/null 2>&1 && ' % (keys[2], key_list) + \
                'echo "Successfully"'

            logger.info("Check the keylist: " + key_list_check)
            ret = Ssh.ssh_exec_command_return_list(self.ssh_client,
                                                   key_list_check)
            if -1 != str(ret).find("Successfully"):
                self.result['keylist'] = 'OK'
                logger.info("keylist is OK at " + self.ip)
            else:
                self.result['keylist'] = 'Not configured'
                logger.error("Keylist is not configured at " + self.ip)
            logger.info("Check key list completed.")
            return True
        except Exception as e:
            logger.error("Exception occurs: " + str(e))
            self.result['err_msg'] = 'Internal error: exception occurs.'
            return False

    def close_ssh_client(self):
        Ssh.close_ssh_clinet(self.ssh_client)

    def inspect_driver_deploy_info(self):
        ret = self.get_ssh_client()
        ret = ret and self.inspcet_version()
        ret = ret and self.inspcet_patch_version()
        self.format_result()
        if not ret:
            logger.error("Inpsect driver deploy info failed.")
        self.close_ssh_client()
        return self.result

    def inspect_cps_server_key_list(self):
        ret = self.get_ssh_client() and self.inspect_key_list()
        if not ret:
            logger.error("Inspcet_cps server key list failed.")
        self.format_result()
        self.close_ssh_client()
        return self.result

    def format_result(self):
        if 0 != len(self.result['err_msg']):
            self.result['err_msg'] = add_color(self.result['err_msg'])
        for key in ['version', 'patch', 'keylist']:
            if -1 != self.result[key].find('Not'):
                self.result[key] = add_color(self.result[key])


def query_deploy_info(ip, user_info, is_cps_node=False):
    account = user_info['fsp_username']
    account_pwd = user_info['fsp_password']
    root_password = user_info['root_password']
    inspector = DeployInfoInspect(ip, account, account_pwd, root_password)
    if not is_cps_node:
        result = inspector.inspect_driver_deploy_info()
    else:
        result = inspector.inspect_cps_server_key_list()
    return result


def add_color(string, color='31'):
    return '\033[1;{0}m{1}\033[0m'.format(color, string)


def inspect_cps_server_key_list(user_dict):
    global G_CPS_NODES
    logger.info("Begin to inspect cps server keylist.")
    cps_user = user_dict['cps_username']
    csp_pwd = user_dict['cps_password']
    try:
        is_true, G_CPS_NODES = get_cps_server_node(cps_user, csp_pwd)
        if not is_true:
            logger.error("Get cps nodes failed")
            return False
        cps_talbe = PrettyTable(["cps-server omip", "keylist"])
        error_dict = {}
        print_error_table = False
        print_cps_talbe = False
        is_true = True
        for ip in G_CPS_NODES:
            print("Inspect cps-server node " + ip)
            result = query_deploy_info(ip, user_dict, True)
            if 0 != len(result['err_msg']):
                error_dict.update({result['ip']: result['err_msg']})
                print_error_table = True
                is_true = False
            else:
                cps_talbe.add_row([result['ip'], result['keylist']])
                print_cps_talbe = True
                if -1 != result['keylist'].find('Not'):
                    is_true = False
        if print_cps_talbe:
            print(cps_talbe)
        if print_error_table:
            print_dict(error_dict, 'cps-server omip', add_color('error info'))
        msg = add_color('Cps server keylist inspecting result: ', '32')
        if is_true and not print_error_table:
            print(msg + add_color('pass', '32'))
        else:
            print(msg + add_color('unpass'))
        logger.info("Cps server keylist inspecting completed.")
        return True
    except Exception as e:
        logger.error("Exception occurs: " + str(e))
        return False


def diagnose_error_msg(deploy_dict):
    uninstalled_nodes = []
    unpatched_nodes = []
    versions = {}
    patch_versions = {}
    err_msg = []
    logger.info("Begin to diagnose error msg:" + str(deploy_dict))
    for ip, info in deploy_dict.items():
        if -1 != info['version'].find('Not installed'):
            uninstalled_nodes.append(ip)
        else:
            if info['version'] not in versions:
                versions.update({info['version']: [ip]})
            else:
                versions[info['version']].append(ip)

        if info['patch'] == 'None':
            unpatched_nodes.append(ip)
        else:
            if info['patch'] not in patch_versions:
                patch_versions.update({info['patch']: [ip]})
            else:
                patch_versions[info['patch']].append(ip)

    if 0 != len(uninstalled_nodes):
        err_msg.append("eBackup driver is not installed at nodes: "
                       + ','.join(uninstalled_nodes))
    if 1 != len(versions):
        error_msg = "There are different versions:"
        error_msg += str(versions)
        err_msg.append(error_msg)
    if len(unpatched_nodes) != len(deploy_dict):
        if len(unpatched_nodes) != 0:
            err_msg.append("eBackup driver patch is not installed"
                           " at nodes:" + ','.join(unpatched_nodes))
        if 1 != len(patch_versions):
            error_msg = "There are different patch versions: "
            error_msg += str(patch_versions)
            err_msg.append(error_msg)
    logger.info("Diagnose error msg completed.")
    return err_msg


def query_common_env_info(user_dict):
    global G_SUPPORTED_TEMPLATES
    global G_DRIVER_NODES
    logger.info("Begin to query common env info.")
    cps_user = user_dict['cps_username']
    csp_pwd = user_dict['cps_password']
    try:
        is_true, templates = get_cinder_backup_templates(cps_user, csp_pwd)
        if not is_true:
            logger.error("Get cinder backup failed.")
            return False
        is_true, supported_templates = \
            get_supported_backup_templates(templates, cps_user, csp_pwd)
        if not is_true:
            logger.error("Get supported templates failed")
            return False
        G_SUPPORTED_TEMPLATES = supported_templates
        is_true, driver_nodes = \
            get_ebackup_driver_node(supported_templates, cps_user, csp_pwd)
        if not is_true:
            logger.error("Get driver nodes failed")
            return False
        G_DRIVER_NODES = driver_nodes
        logger.info("Query common env info completed.")
        return True
    except Exception as e:
        logger.error("Exception occurs: " + str(e))
        return False


def line_length_limit_ext(data, line_len=LINE_WORD_NUM):
    ret_data = ''
    count = 0
    if type(data) != 'unicode':
        data = str(data)
    while True:
        tmp = len(data) - count
        if (tmp <= line_len):
            ret_data = ret_data + data[count:len(data)] + '\n'
            break
        else:
            ret_data += data[count:(count + line_len)] + '\n'
            count += line_len

    return ret_data[:-1]


def line_length_limit(data, line_len=LINE_WORD_NUM):
    ret_data = ''
    if isinstance(data, list):
        for str_t in data:
            ret_data += line_length_limit_ext(str_t, line_len) + ',\n'
        ret_data = ret_data[:-2]
    elif isinstance(data, dict):
        for str_key, str_value in data.items():
            ret_data += line_length_limit_ext(str(str_key), line_len) \
                        + ':' + line_length_limit_ext(str(str_value), line_len) \
                        + ',\n'
        ret_data = ret_data[:-2]
    else:
        ret_data = line_length_limit_ext(data, line_len)
    return ret_data


def print_dict(d, key_alias, value_alias):
    pt = PrettyTable([key_alias, value_alias], caching=False)
    pt.align = 'l'
    for r in d.items():
        row = list(r)
        row = [line_length_limit(key) for key in row]
        pt.add_row(row)
    print(pt)


def print_all_template_ext_parameters(user_dict):
    global G_CPS_BASE_URL
    global G_SUPPORTED_TEMPLATES
    logger.info("Begin to print all template ext parameters.")

    header = {
        "Content-Type": "application/json",
        "X-Auth-User": user_dict['cps_username'],
        "X-Auth-Password": user_dict['cps_password']
    }
    try:
        for template in G_SUPPORTED_TEMPLATES:
            uri = 'cps/v1/template/params/cinder/%s?commit_state=commited' \
                  % template
            url = "https://%s/%s" % (G_CPS_BASE_URL, uri)
            rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)
            if rsp.status_code == 200:
                rsp_dict = rsp.json()
                params = rsp_dict['params']
                pt = PrettyTable(['Property', 'Value'], caching=False)
                pt.add_row(['template_name', template])
                pt.align = 'l'
                row = []
                for r in params.items():
                    row = list(r)
                    row = [line_length_limit(key) for key in row]
                    pt.add_row(row)
                print(pt)
            else:
                logger.error("Response status code is not 200."
                             "The url is" + url)
                raise Exception("Response status code is not 200.")

        logger.info("Print all template ext parameters successfully.")
        return True
    except Exception as e:
        logger.error("Print all template ext parameters failed:" + str(e))
        return False


def inspect_basic_deploy_info(user_dict):
    global G_DRIVER_NODES
    logger.info("Begin to inspcet basic deploy info.")
    deploy_talbe = PrettyTable(["ebackup driver omip",
                                "version", 'patch version',
                                "cinder backup templates"])
    error_dict = {}
    print_deploy_talbe = False
    print_error_table = False
    deploy_dict = {}
    driver_nodes = G_DRIVER_NODES
    try:
        for ip in driver_nodes:
            print("Inspect eBackup Driver node " + ip)
            item = query_deploy_info(ip, user_dict)
            if 0 != len(item['err_msg']):
                error_dict.update({item['ip']: item['err_msg']})
                print_error_table = True
            else:
                deploy_dict.update({ip: item})
                print_deploy_talbe = True
                deploy_talbe.add_row([item['ip'],
                                      item['version'],
                                      item['patch'],
                                      ','.join(driver_nodes[ip])])
        if print_deploy_talbe:
            print(deploy_talbe)
        if print_error_table:
            print_dict(error_dict, "ebackup driver omip",
                       add_color("error info"))
        error_msg = diagnose_error_msg(deploy_dict)
        if 0 != len(error_msg):
            error_msg_dict = {}
            for i, msg in enumerate(error_msg):
                msg = add_color(msg)
                error_msg_dict.update({str(i + 1): msg})
            print_dict(error_msg_dict, "index", add_color("error info"))
        msg = add_color("Basic deploy info inspecting result: ", '32')
        if 0 == len(error_msg) and not print_error_table:
            print(msg + add_color('pass', '32'))
        else:
            print(msg + add_color("unpass"))
        logger.info("Inspcet basic deploy info completed.")
        return True
    except Exception as e:
        logger.error("Exception occurs: " + str(e))
        return False


def get_backup_extend_parameters(cps_username, cps_password, template_name):
    global G_CPS_BASE_URL
    url = "https://%s/cps/v1/template/params/cinder/%s?" \
          "commit_state=commited" % (G_CPS_BASE_URL, template_name)
    header = {
        "Content-Type": "application/json",
        "X-Auth-User": cps_username,
        "X-Auth-Password": cps_password
    }
    try:
        rsp = requests.get(url=url, headers=header, verify=CPS_CA_CERT)

        if rsp.status_code == 200:
            rsp_dict = rsp.json()
            return rsp_dict
        else:
            logger.error("rsp.status_code is not 200.")
            return None

    except Exception as e:
        logger.error("Query backup extend parameters failed,"
                     "the reason is " + str(e))
        return None


def exec_fsp_cmd(ip, account, account_password, root_pwd,
                 cps_username, cps_pwd, cmds):
    try:
        ssh_client = Ssh.ssh_create_client(ip, account,
                                           account_password)
        logger.info("Login {0} successfully by user {1}".
                    format(ip, account))
    except Exception as e:
        logger.error("The password of the {0} user is incorrect on host {1}."
                     "The reason is:{2}".format(ip, account, str(e)))
        Ssh.close_ssh_clinet(ssh_client)

    try:
        Ssh.ssh_send_command(ssh_client, 'su - root', 'Password:', 15)
        logger.info("Excute su - root on host {0} successfully by user {1}".
                    format(ip, account))
    except Exception as e:
        logger.error("Excute su - root on host {0} failed."
                     "The reason is:{1}".format(ip, str(e)))
        Ssh.close_ssh_clinet(ssh_client)

    try:
        Ssh.ssh_send_command(ssh_client, root_pwd, '#', 15)
        logger.info("su - root successfuly on host " + ip)
    except Exception as e:
        logger.error("The password of the root user is incorrect on host {0}."
                     "The reason is:{1}".format(ip, str(e)))
        Ssh.close_ssh_clinet(ssh_client)

    try:
        Ssh.ssh_send_command(ssh_client, 'export OPERATE_USER=ebackup;'
                                         'source set_env', 'please choose:', 15)
        Ssh.ssh_send_command(ssh_client, '1', 'Enter OS_PASSWORD=', 15)
        Ssh.ssh_send_command(ssh_client, cps_pwd, '#', 15)
        logger.info("source set_env succ on host " + ip)

        outs = Ssh.ssh_exec_command_return_list(ssh_client, cmds, timeout=30)
        Ssh.ssh_close(ssh_client)
        return outs
    except Exception as e:
        logger.error("exec cmd failed on host {0}."
                     "The reason is:{1}".format(ip, str(e)))
        Ssh.close_ssh_clinet(ssh_client)


def check_ip(ip_value):
    pattern = r"^(?:(?:2[0-4][0-9]\.)|(?:25[0-5]\.)|(?:1[0-9][0-9]\.)|" \
              r"(?:[1-9][0-9]\.)|(?:[0-9]\.)){3}(?:(?:2[0-4][0-9])|" \
              r"(?:25[0-5])|(?:1[0-9][0-9])|(?:[1-9][0-9])|(?:[0-9]))$"
    match_obj = re.match(pattern, ip_value, re.I)
    if match_obj:
        return True
    else:
        return False


def check_domain_name(domain_name):
    pattern = r"^[a-zA-Z0-9][-a-zA-Z0-9]{0,62}" \
              r"(\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+$"
    match_obj = re.match(pattern, domain_name, re.I)
    if match_obj:
        return True
    else:
        return False


def check_storage_pwd(param_dict):
    storage_pwd = param_dict.get('default.ebackup_storage_password_v2')
    if storage_pwd is None or storage_pwd == "":
        return "", "NO PASS", "ebackup_storage_password_v2 is empty."
    else:
        try:
            origin_pwd = fscrypt.decrypt(storage_pwd)
            if origin_pwd is None or origin_pwd == "":
                return storage_pwd, "NO PASS", \
                       "maybe ebackup_storage_password_v2 is not encrypted."
            else:
                return storage_pwd, "OK", ""
        except Exception as e:
            return storage_pwd, "NO PASS", \
                   "maybe ebackup_storage_password_v2 is not encrypted."


def check_storage_username(param_dict):
    storage_username = param_dict.get('default.ebackup_storage_username_v2')
    if storage_username is None or storage_username == "":
        return "", "NO PASS", "ebackup_storage_username_v2 is empty."
    else:
        try:
            origin_username = fscrypt.decrypt(storage_username)
            if origin_username is None or origin_username == "":
                return storage_username, "NO PASS", \
                       "maybe ebackup_storage_username_v2 is not encrypted."
            else:
                return storage_username, "OK", ""
        except Exception as e:
            return storage_username, "NO PASS", \
                   "maybe ebackup_storage_username_v2 is not encrypted."


def check_az(param_dict, check_key, user_dict):
    global G_AZ_LIST
    az = param_dict.get(check_key)
    if az == "default":
        return "default", "OK", ""
    cmd = (
        "nova availability-zone-list|awk -F'|' '{print $2}'| sed '/^[ ]*$/d'|grep -E -v 'Name|internal'|sed 's/ //g'")
    if len(G_AZ_LIST) == 0:
        G_AZ_LIST = \
            exec_fsp_cmd(user_dict['local_omip'], user_dict['fsp_username'],
                         user_dict['fsp_password'], user_dict['root_password'],
                         user_dict['cps_username'], user_dict['cps_password'], cmd)
    az_list = az.split(',')
    for sub_az in az_list:
        if sub_az not in G_AZ_LIST:
            return az, "NO PASS", "az is not in availability-zone-list"
    return az, "OK", ""


def check_login_user(param_dict, check_key):
    global G_CHECK3
    G_CHECK3 = False
    login_user = param_dict.get(check_key)
    if login_user is None or login_user == "":
        return "", "NO PASS", "ebackup_login_username is empty."
    else:
        G_CHECK3 = True
        return login_user, "OK", ""


def check_login_ip(param_dict, check_key):
    global G_CHECK1
    G_CHECK1 = False
    ebk_server_ip = param_dict.get(check_key)
    if ebk_server_ip is None or ebk_server_ip == "":
        return "", "NO PASS", "ebackup_server_ip is empty."
    else:
        if not ping_ip(ebk_server_ip):
            return ebk_server_ip, "NO PASS", ebk_server_ip + " is unreachable."
        else:
            G_CHECK1 = True
            return ebk_server_ip, "OK", ""


def check_ebk_port(param_dict, check_key):
    global G_CHECK2
    G_CHECK2 = False
    ebk_server_port = param_dict.get(check_key)
    if ebk_server_port is None or ebk_server_port == "":
        return "", "NO PASS", "ebackup_server_port is empty."
    elif ebk_server_port != "8088":
        return ebk_server_port, "NO PASS", "ebackup_server_port is not 8088."
    else:
        G_CHECK2 = True
        return ebk_server_port, "OK", ""


def check_fs_config(param_dict, check_key):
    fsm_ip = param_dict.get(check_key)
    if fsm_ip is None or fsm_ip == "":
        return "", "NO PASS", "fusion_storage_ip is empty."
    else:
        if not check_ip(fsm_ip):
            logger.error("IP {} is invalid.".format(fsm_ip))
            return fsm_ip, "NO PASS", "fsm ip(%s) is invalid." % fsm_ip
        if not ping_ip(fsm_ip):
            return fsm_ip, "NO PASS", "fsm ip(%s) is unreachable." % fsm_ip
        return fsm_ip, "OK", ""


def check_fs_agent_config(param_dict, check_key):
    fusion_storage_agent_ip = param_dict.get(check_key)
    if fusion_storage_agent_ip is None or fusion_storage_agent_ip == "":
        return "", "NO PASS", "fusion_storage_agent_ip is empty."
    else:
        ip_list = fusion_storage_agent_ip.split(',')
        if len(ip_list) < 3:
            return fusion_storage_agent_ip, "NO PASS", "fusion storage agent ip must >= 3."
        for agent_ip in ip_list:
            if not check_ip(agent_ip):
                logger.error("IP {} is invalid.".format(agent_ip))
                return fusion_storage_agent_ip, "NO PASS", "fsm ip(%s) is invalid." % agent_ip
            if not ping_ip(agent_ip):
                return fusion_storage_agent_ip, "NO PASS", "fsm agent ip(%s) is unreachable." % fusion_storage_agent_ip
        return fusion_storage_agent_ip, "OK", ""


def check_obs_ip(param_dict):
    storage_unit_path = param_dict.get('default.ebackup_storage_unit_path')
    if storage_unit_path is None or storage_unit_path == "":
        return "", "NO PASS", "ebackup_storage_unit_path is empty."
    else:
        if not check_domain_name(storage_unit_path):
            if not check_ip(storage_unit_path):
                return storage_unit_path, "NO PASS", \
                       "ebackup_storage_unit_path is not ip or domain name."
        return storage_unit_path, "OK", ""


def check_login_pwd(param_dict, check_ebk_ip_key, check_ebk_port_key, check_ebk_user_key, check_ebk_pwd_key,
                    check_ebk_ca_cert):
    global G_CHECK1
    global G_CHECK2
    global G_CHECK3
    login_pwd = param_dict.get(check_ebk_pwd_key)
    if login_pwd is None or login_pwd == "":
        return "", "NO PASS", "ebackup_login_password_v2 is empty."

    try:
        origin_pwd = fscrypt.decrypt(login_pwd)
        if not origin_pwd:
            return login_pwd, "No password, maybe ebackup_login_password_v2 is not encrypted."
        else:
            if G_CHECK1 and G_CHECK2 and G_CHECK3:
                ebk_ip = param_dict.get(check_ebk_ip_key)
                ebk_port = param_dict.get(check_ebk_port_key)
                ebk_user = param_dict.get(check_ebk_user_key)
                ebk_ca_cert = param_dict.get(check_ebk_ca_cert)
                if check_ebk_login(ebk_ip, ebk_port, ebk_user, origin_pwd, ebk_ca_cert):
                    return login_pwd, "OK", ""
                else:
                    return login_pwd, "NO PASS", "login(%s) failed by user(%s)" % (ebk_ip, ebk_user)
            return login_pwd, "OK", ""
    except Exception as e:
        return login_pwd, "NO PASS", \
               "maybe ebackup_login_password_v2 is not encrypted."


def check_ebk_login(ebk_ip, ebk_port, ebk_user, ebk_pwd, ebk_ca_cert):
    base_uri = 'https://' + ebk_ip + ':' + ebk_port + '/v3/auth/tokens'
    data = '{"auth":{"identity":{"methods":["password"],"password":{"user":{"name":"' + ebk_user + \
           '","password":"' + ebk_pwd + '","iam_type":4}}}}}'
    headers = {'Accept': 'application/json'}
    ebk_ca_cert_path = write_ca_cert_content_to_file(ebk_ip, ebk_ca_cert)
    rsp = requests.post(base_uri, headers=headers, data=data, verify=ebk_ca_cert_path)
    if rsp.status_code != 200:
        return False

    login_token = rsp.json()["Data"]["token"]
    if login_token is None:
        return False

    try:
        headers = {'Accept': 'application/json', 'X-Auth-Token': login_token}
        requests.delete(base_uri, headers=headers)
        return True
    except Exception as e:
        return True


def check_ebk_server(param_dict, ebk_server_str):
    is_find = False
    ebk_server_list = ebk_server_str.split(',')
    for sub_ebk_server in ebk_server_list:
        for key in param_dict:
            if sub_ebk_server in key:
                is_find = True
                break

        if not is_find:
            return ebk_server_str, "NO PASS", \
                   "%s is not invalid." % sub_ebk_server

    return ebk_server_str, "OK", ""


def change_color(result, suggestion):
    if result != "OK":
        return add_color(result), add_color(suggestion)
    else:
        return result, suggestion


def inspect_configuration_info(user_dict):
    global G_SUPPORTED_TEMPLATES
    global G_CLOUD_SCENE
    logger.info("Begin to inspect eBackupDriver configuration info.")
    cps_user = user_dict['cps_username']
    cps_pwd = user_dict['cps_password']
    config_talbe = PrettyTable(["Template name", "Property",
                                "Value", 'Check result', 'Suggestion'])
    ebk_lazyloading = ""
    try:
        for cinder_backup_template in G_SUPPORTED_TEMPLATES:
            result_dic = get_backup_extend_parameters(cps_user, cps_pwd,
                                                      cinder_backup_template)
            if result_dic is not None:
                param_dic = result_dic.get('params')
                G_CLOUD_SCENE = param_dic.get('default.ebackup_env_type')

                if G_CLOUD_SCENE == '1':
                    value, result, suggestion = check_obs_ip(param_dic)
                    result, suggestion = change_color(result, suggestion)
                    config_talbe.add_row([cinder_backup_template,
                                          "default.ebackup_storage_unit_path",
                                          value, result, suggestion])

                    value, result, suggestion = check_storage_username(param_dic)
                    result, suggestion = change_color(result, suggestion)
                    config_talbe.add_row([cinder_backup_template,
                                          "default.ebackup_storage_username_v2",
                                          value, result, suggestion])

                    value, result, suggestion = check_storage_pwd(param_dic)
                    result, suggestion = change_color(result, suggestion)
                    config_talbe.add_row([cinder_backup_template,
                                          "default.ebackup_storage_password_v2",
                                          value, result, suggestion])

                    ebk_lazyloading = param_dic.get('default.ebackup_lazyloading')
                    if ebk_lazyloading is None or ebk_lazyloading == "":
                        config_talbe.add_row([cinder_backup_template,
                                              "default.ebackup_lazyloading",
                                              "",
                                              add_color("NO PASS"),
                                              add_color("ebackup_lazyloading is empty.")])
                    elif ebk_lazyloading != '0' and ebk_lazyloading != '1' and ebk_lazyloading != '2' \
                            and ebk_lazyloading != '3' and ebk_lazyloading != '4':
                        result, suggestion = change_color("NO PASS", "default.ebackup_lazyloading is not 0、1、2、3、4")
                        config_talbe.add_row([cinder_backup_template,
                                              "default.ebackup_lazyloading",
                                              ebk_lazyloading, result, suggestion])
                    else:
                        config_talbe.add_row([cinder_backup_template,
                                              "default.ebackup_lazyloading",
                                              ebk_lazyloading, "OK", ""])

                ebk_server_str = param_dic.get('default.ebackup_servers')
                if ebk_server_str is None or ebk_server_str == "":
                    config_talbe.add_row([cinder_backup_template,
                                          "default.ebackup_servers",
                                          "",
                                          add_color("NO PASS"),
                                          add_color("default.ebackup_servers is empty.")])
                else:
                    value, result, suggestion = check_ebk_server(param_dic, ebk_server_str)
                    result, suggestion = change_color(result, suggestion)
                    config_talbe.add_row([cinder_backup_template,
                                          "default.ebackup_servers",
                                          value, result, suggestion])

                    ebk_server_list = ebk_server_str.split(',')
                    is_find = False
                    for sub_ebk_server in ebk_server_list:
                        for key in param_dic:
                            if sub_ebk_server in key:
                                is_find = True
                                break
                        if not is_find:
                            continue
                        check_key = sub_ebk_server + ".azs"
                        value, result, suggestion = check_az(param_dic, check_key, user_dict)
                        result, suggestion = change_color(result, suggestion)
                        config_talbe.add_row([cinder_backup_template, check_key,
                                              value, result, suggestion])

                        check_ebk_ip_key = sub_ebk_server + ".ebackup_server_ip"
                        value, result, suggestion = check_login_ip(param_dic, check_ebk_ip_key)
                        result, suggestion = change_color(result, suggestion)
                        config_talbe.add_row([cinder_backup_template, check_ebk_ip_key,
                                              value, result, suggestion])

                        check_ebk_port_key = sub_ebk_server + ".ebackup_server_port"
                        value, result, suggestion = check_ebk_port(param_dic, check_ebk_port_key)
                        result, suggestion = change_color(result, suggestion)
                        config_talbe.add_row([cinder_backup_template, check_ebk_port_key,
                                              value, result, suggestion])

                        check_ebk_user_key = sub_ebk_server + ".ebackup_login_username"
                        check_ebk_ca_cert = sub_ebk_server + ".ebackup_login_crt"

                        value, result, suggestion = check_login_user(param_dic, check_ebk_user_key)
                        result, suggestion = change_color(result, suggestion)
                        config_talbe.add_row([cinder_backup_template, check_ebk_user_key,
                                              value, result, suggestion])

                        check_ebk_pwd_key = sub_ebk_server + ".ebackup_login_password_v2"
                        value, result, suggestion = check_login_pwd(param_dic, check_ebk_ip_key,
                                                                    check_ebk_port_key, check_ebk_user_key,
                                                                    check_ebk_pwd_key, check_ebk_ca_cert)
                        result, suggestion = change_color(result, suggestion)
                        config_talbe.add_row([cinder_backup_template, check_ebk_pwd_key,
                                              value, result, suggestion])

                        if ebk_lazyloading == "1" or ebk_lazyloading == "2":
                            check_key = sub_ebk_server + ".fusion_storage_ip"
                            value, result, suggestion = check_fs_config(param_dic, check_key)
                            result, suggestion = change_color(result, suggestion)
                            config_talbe.add_row([cinder_backup_template, check_key,
                                                  value, result, suggestion])

                            check_key = sub_ebk_server + ".fusion_storage_agent_ip"
                            value, result, suggestion = check_fs_agent_config(param_dic, check_key)
                            result, suggestion = change_color(result, suggestion)
                            config_talbe.add_row([cinder_backup_template,
                                                  check_key, value, result,
                                                  suggestion])

        print(config_talbe)
        logger.info("Inspect eBackupDriver configuration info completed")
        return True
    except Exception as e:
        logger.error("Exception occurs: " + str(e))
        return False


if __name__ == '__main__':
    G_CPS_BASE_URL = get_cps_server_base_url()
    user_dict = get_user_and_password()

    if not query_common_env_info(user_dict):
        print("Internal error,please see log[{}]".format(log_file_path))
        exit(1)
    if not print_all_template_ext_parameters(user_dict):
        print("Internal error,please see log[{}]".format(log_file_path))

    print(add_color("1.Inspect basic deploy info", '33'))
    if not inspect_basic_deploy_info(user_dict):
        print("Internal error,please see log[{}]".format(log_file_path))

    print(add_color("2.Inspect keylist on cps server", '33'))
    if not inspect_cps_server_key_list(user_dict):
        print("Internal error,please see log[{}]".format(log_file_path))

    print(add_color("3.Inspect eBackup driver configuration info", '33'))
    if not inspect_configuration_info(user_dict):
        print("Internal error,please see log[{}]".format(log_file_path))
