import logging
from cinder.backup.drivers.ebackupconst import Constants as cst, ErrorCode
from cinder.backup.lib.ebackuprestful import EbackupConnectionClient
from cinder.backup.drivers.ebackupagent import call_db_fun
import json
import uuid
import copy
import time

LOG = logging.getLogger(__name__)
PHASE_FAILED = -1
PHASE_CREATE_UNIT = 0
PHASE_CREATE_POOL = 1
PHASE_CREATE_REPOSITORY = 2
PHASE_CREATE_BACKUP_PLAN = 3
PHASE_CREATE_TASK = 4
PHASE_MONITOR_TASK = 5
PHASE_COMPLETED = 100


class Uranus(object):
    def __init__(self, context, db, backup, ebackupconnect, driverconf, source_volume_file, evs_body, verify=False):
        self.context = context
        self.backup = backup
        self.db = db
        self.volume_file = source_volume_file
        self.ebackupconnect = EbackupConnectionClient(ebackupconnect)
        self.protocal = driverconf.ebackup_storage_s3_protocol
        self.ak = driverconf.ebackup_storage_username
        self.sk = driverconf.ebackup_storage_password
        self.unit_path = driverconf.ebackup_storage_unit_path + ":/" + driverconf.ebackup_bucket_name + self.backup[
            'project_id']
        self.step = 0
        self.metadata = None
        # tenant resource name
        self.stg_unit_name = 'openstack_unit_' + self.backup['project_id']
        self.stg_pool_name = 'openstack_pool_' + self.backup['project_id']
        self.stg_repository_name = 'openstack_repository_' + self.backup['project_id']
        self.stg_backup_plan_name = ''
        self.storage_unit = None
        self.storage_pool = None
        self.storage_repository = None
        self.backup_plan = None
        self.verify = verify
        self.evs_body = evs_body
        LOG.info("Object of Uranus init Completed!")

    def recover_status(self):
        if self.backup['service_metadata']:
            self.metadata = json.loads(self.backup['service_metadata'])
        else:
            self.metadata = dict()
        if self.metadata.get('ST') == 1:
            self.stg_backup_plan_name = 'openstack_cbr_machine_plan_' + self.volume_file[
                'source_volume_id'] + '_zip_noDedup'
        else:
            self.stg_backup_plan_name = 'openstack_cbr_volume_plan_' + self.volume_file[
                'source_volume_id'] + '_zip_noDedup'
        return

    def update_backup_service_metadata(self, **kwargs):
        for i in kwargs:
            self.metadata[i] = kwargs[i]
        meta_str = json.dumps(self.metadata)
        call_db_fun(self.db.backup_update, self.context, self.backup.get("id"), {'service_metadata': meta_str})
        return

    def create_storage_unit(self):
        data, err, desc = self.ebackupconnect.create_storage_unit(self.stg_unit_name, self.unit_path, self.ak,
                                                                  self.sk, self.protocal)
        if err:
            LOG.error('Create Storage Unit %s Failed %s %s!' % (self.stg_unit_name, err, desc))
            raise Exception(str(err) + ';' + desc)
        else:
            self.storage_unit = data[0]
            LOG.info('Create Storage Unit %s completed!', self.stg_unit_name)
            self.step = PHASE_CREATE_POOL
        return

    def create_storage_pool(self):
        data, err, desc = self.ebackupconnect.create_storage_pool(self.stg_pool_name, self.storage_unit.get('ID'))
        if err:
            LOG.error('Create Storage Pool %s Failed %s %s!' % (self.stg_pool_name, err, desc))
            raise Exception(str(err) + ';' + desc)
        else:
            self.storage_pool = data[0]
            LOG.info('Create Storage Pool %s completed!', self.stg_pool_name)
            self.step = PHASE_CREATE_REPOSITORY
        return

    def create_storage_repository(self):
        data, err, desc = self.ebackupconnect.create_storage_repository(self.stg_repository_name,
                                                                        self.storage_pool.get('ID'),
                                                                        self.storage_pool.get('NAME'),
                                                                        self.storage_unit.get('FREECAPACITY'))
        if err:
            LOG.error('Create Storage Repository %s Failed %s %s!' % (self.stg_repository_name, err, desc))
            raise Exception(str(err) + ';' + desc)
        else:
            self.storage_repository = data[0]
            LOG.info('Create Storage Repository %s completed!', self.stg_repository_name)
            self.step = PHASE_CREATE_BACKUP_PLAN
        return

    def uranus_create_backup_plan(self):
        plan_obj = dict()
        plan_obj["set_name"] = 'openstack_default_set'
        plan_obj["plan_name"] = self.stg_backup_plan_name
        plan_obj["repository_name"] = self.stg_repository_name
        plan_obj["policy_name"] = 'openstack_default_policy_zip_noDedupe'
        data, err, desc = self.ebackupconnect.create_backup_plan(plan_obj)
        if err:
            LOG.error('Create backup plan %s Failed %s %s!' % (self.stg_backup_plan_name, err, desc))
            raise Exception(str(err) + ';' + desc)
        else:
            self.backup_plan = data[0]
            LOG.info('Create backup plan %s completed!', self.stg_backup_plan_name)
            self.step = PHASE_CREATE_TASK
        return

    def get_volume_offset_size(self, volume):
        volume_offset_str = ''
        provider_location_str = volume.get("provider_location")
        if provider_location_str is None:
            LOG.error('The get_volume_offset_size provider location does not exist.')
        else:
            try:
                provider_location = json.loads(provider_location_str)
            except Exception as e:
                LOG.info("get_volume_offset_size provider_location is not json")
                return volume_offset_str
            volume_offset = provider_location.get('offset')
            if volume_offset not in (None, ''):
                volume_offset_str = str(volume_offset)
        return volume_offset_str

    def run_backup_plan(self):
        if self.metadata.get('ebk_T_I'):
            task_id = self.metadata.get('ebk_T_I')
            if task_id.startswith('R_'):
                task_id = task_id[2:]
            else:
                LOG.info('Task %s is running. continue!' % task_id)
        else:
            task_id = "%d" % hash("%s" % uuid.uuid1())
            if task_id.startswith('-'):
                task_id = task_id[1:]
            self.update_backup_service_metadata(ebk_T_I='R_' + task_id)
        if self.volume_file['backup_type'] == cst.FULL_BACKUP:
            backup_mode = 0
        else:
            backup_mode = 1
        offset_size = 0
        if str(self.volume_file['storage_type']) in (cst.STORAGE_TYPE_VRM_FUSION_STORAGE,
                                                     cst.STORAGE_TYPE_VRM_FUSION_RDM,
                                                     cst.STORAGE_TYPE_KVM_FUSION_STORAGE):
            source_vol = call_db_fun(self.db.volume_get, self.context, self.backup['volume_id'])
            vol_offset_size_str = self.get_volume_offset_size(source_vol)
            if vol_offset_size_str not in (None, ''):
                offset_size = int(vol_offset_size_str)
        volume_size = self.backup['size'] * cst.Gi
        run_plan_body = {
            "DISPLAY_NAME": self.backup['display_name'],
            "VOLUME_ID": self.backup['volume_id'],
            "CHANNEL_TYPE": cst.CHANNEL_TYPE,
            "STORAGE_TYPE": self.volume_file['storage_type'],
            "VOLUME_SIZE": volume_size,
            "SNAPSHOT_URL": self.volume_file['snapshot_url'],
            "PARENT_SNAPSHOT_URL": self.volume_file.get('parent_snapshot_url', ""),
            "VOLUME_OFFSET_SIZE": offset_size,
            "ID": str(self.backup_plan['ID']),
            "TYPE": cst.PLAN_TYPE_ID,
            "BACKUPMODE": backup_mode,
            "REQUEST_ID": task_id,
            "REGION": "",
        }
        run_plan_body = dict(run_plan_body, **self.evs_body)
        run_plan_body_info = copy.deepcopy(run_plan_body)
        run_plan_body_info['DATA_ENC_INFO']['VK'] = "******"
        run_plan_body_info['DATA_ENC_INFO']['DEK'] = "******"
        LOG.info("volume %s run plan body:%s" % (self.backup['volume_id'], str(run_plan_body_info)))
        data, err, desc = self.ebackupconnect.active_backup_plan(run_plan_body)
        if err:
            LOG.error('Active backup plan %s Failed %s %s!' % (self.stg_backup_plan_name, err, desc))
            raise Exception(str(err) + ';' + desc)
        else:
            task_id_ebk = self.ebackupconnect.get_task_id_by_request(cst.TASKTYPE_BACKUP, task_id)
            self.update_backup_service_metadata(ebk_T_I=str(task_id_ebk))
            LOG.info('Active backup plan %s completed,task id is %s!' % (self.stg_backup_plan_name, task_id_ebk))
            self.step = PHASE_MONITOR_TASK
        return

    def monit_task(self):
        while True:
            taskid = self.metadata.get('ebk_T_I')
            data, err, desc = self.ebackupconnect.get_task_info(taskid)
            if err:
                LOG.error('Monit task %s Failed %s %s!' % (taskid, err, desc))
                raise Exception(str(err) + ';' + desc)
            else:
                if data.get('STATUS') in (cst.TASK_STATUS_STOP, cst.TASK_STATUS_FAILED):
                    LOG.error('backup %s task %s execute failed!' % (self.backup['id'], taskid))
                    raise Exception('backup %s task %s execute failed!' % (self.backup['id'], taskid))
                elif data.get('STATUS') == cst.TASK_STATUS_COMPLETED:
                    backup_data, err, desc = self.ebackupconnect.get_snap_by_taskid(taskid)
                    if err:
                        LOG.error("get snap id with task id %s failed!" % taskid)
                        raise Exception(str(err) + ';' + desc)
                    self.update_backup_service_metadata(DL=2, CBR=1, LLD=1, ebk_T_I="", progress="0", BT="7", BP=self.unit_path,
                                                        SP=data.get("IMAGEPATH"),
                                                        AT=0 if not data.get("SPEED") else abs(round(float(data.get("SPEED")) / 1024, 1)),
                                                        SS=0 if not data.get("SPACESAVINGRATE") else int(data.get("SPACESAVINGRATE")),
                                                        SNAPSHOTSIZE=data.get("SNAPSHOTSIZE"),
                                                        backupurl=backup_data[0].get("ID"),
                                                        CMKID=self.evs_body['DATA_ENC_INFO']['CMKID'],
                                                        VK=self.evs_body['DATA_ENC_INFO']['VK'])
                    LOG.info('backup %s task %s execute completed!' % (self.backup['id'], self.metadata.get('ebk_T_I')))
                    break
                self.update_backup_service_metadata(progress=data.get("PROGRESS"),
                                                    AT=0 if not data.get("SPEED") else abs(round(float(data.get("SPEED")) / 1024, 1)),
                                                    SS=0 if not data.get("SPACESAVINGRATE") else int(data.get("SPACESAVINGRATE")))
                LOG.info("backup %s task %s executing progress %s" % (
                    self.backup['id'], self.metadata.get('ebk_T_I'), data.get("PROGRESS")))
            time.sleep(cst.TIME_SLEEP_60)
        self.step = PHASE_COMPLETED
        return

    def backup_engine(self):
        LOG.info("--------begin CBR public cloud backup---------")
        # step 1. init all config,recover status
        self.recover_status()
        while self.step not in (PHASE_FAILED, PHASE_COMPLETED):
            try:
                if self.step == PHASE_CREATE_UNIT:
                    self.create_storage_unit()
                elif self.step == PHASE_CREATE_POOL:
                    self.create_storage_pool()
                elif self.step == PHASE_CREATE_REPOSITORY:
                    self.create_storage_repository()
                elif self.step == PHASE_CREATE_BACKUP_PLAN:
                    self.uranus_create_backup_plan()
                elif self.step == PHASE_CREATE_TASK:
                    self.run_backup_plan()
                elif self.step == PHASE_MONITOR_TASK:
                    self.monit_task()
            except Exception as e:
                self.step = PHASE_FAILED
                if self.verify:
                    call_db_fun(self.db.backup_update, self.context, self.backup.get("id"),
                                {'availability_zone': self.volume_file["availability_zone"],
                                 'status': cst.BACKUP_STATUS_ERROR,
                                 'service': 'cinder.backup.drivers.hwsebackup'})
                    LOG.error('update backup %s status to error durning verify!' % self.backup.get("id"))
                LOG.error('backup engine error %s', e)
                raise Exception(e)
        if self.verify:
            call_db_fun(self.db.backup_update, self.context, self.backup.get("id"),
                        {'availability_zone': self.volume_file["availability_zone"],
                         'status': cst.BACKUP_STATUS_AVALIBLE,
                         'service': 'cinder.backup.drivers.hwsebackup'})
            LOG.info('update backup %s status to available durning verify!' % self.backup.get("id"))
        return
