import logging
from cinder.backup.drivers.ebackupconst import Constants as cst, ErrorCode
import time

LOG = logging.getLogger(__name__)


class EbackupConnectionClient(object):
    """ ebackup restful client"""

    def __init__(self, httpconnection, retry=True):
        self.connect = httpconnection
        self.ebackup_unit_uri = '/vbackup_storageunit'
        self.ebackup_pool_uri = '/vbackup_storagepool'
        self.ebackup_manager_pool_uri = '/v1/srv_jobmanager/get_single_storage_pool'
        self.ebackup_manager_pool_list_uri = '/v1/srv_jobmanager/get_list_of_storage_pools'
        self.ebackup_repository_uri = '/vbackup_repository'
        self.ebackup_manager_repository_uri = '/v1/srv_jobmanager/get_list_of_namespaces'
        self.ebackup_set_uri = '/vbackup_protectedset'
        self.ebackup_policy_uri = '/vbackup_policy'
        self.ebackup_plan_uri = '/vbackup_plan'
        self.ebackup_manager_plan_list_uri = '/v1/srv_jobmanager/get_list_of_backup_plans'
        self.ebackup_progress_uri = '/vbackup_task'
        self.ebackup_image_uri = '/vbackup_image'
        self.ebackup_image_copy_uri = '/vbackup_image/copysnaps'
        self.ebackup_restore_disk_uri = '/vbackup_restore/disk'
        self.ebackup_get_backup_size = '/vbackup_backupobj'
        self.ebackup_copy_policy_uri = '/vcopy_policy'
        self.ebackup_copy_plan_uri = '/vcopy_plan'
        self.ebackup_restore_direct_uri = '/vbackup_restore_direct/disk'
        self.ebackup_encrypt_url = '/encrypt'
        self.retry = retry

    @staticmethod
    def _get_response_error(response):
        """ get error num from response """
        dict_error = response['error']
        return dict_error['code']

    @staticmethod
    def _get_response_description(response):
        """ get error description from response """
        dict_error = response['error']
        return dict_error['description']

    @staticmethod
    def _get_response_data(response):
        """ get data from response """
        return response.get("data", {})

    def create_storage_unit(self, name, unit_path, uds_login_ak, uds_login_sk, uds_protocol):
        unit_body = {
            "DESCRIPTION": "Storage unit for eBackup under OpenStack",
            "FILESYSTEMTYPE": "7",
            "NAME": name,
            "PATH": unit_path,
            "USERNAME": uds_login_ak,
            "PASSWORD": uds_login_sk,
            "PROTOCOL": uds_protocol
        }
        result = self.connect.post(self.ebackup_unit_uri, unit_body)
        ret_code = self._get_response_error(result)
        if ret_code == ErrorCode.EXISTS_NAME:
            LOG.info('Storage Unit %s already exist!', name)
            return self.get_storage_unit_by_name(name)

        # return storage unit list
        return [self._get_response_data(result)], self._get_response_error(result), self._get_response_description(
            result)

    def get_storage_unit_by_image_id(self, imageid, bp):
        storage_unit_filter = self.ebackup_unit_uri + "/getimagerepository"
        unit_body = {
            "IMAGEID": imageid,
            "TYPE": 57358,
            "PATH": bp
        }
        LOG.info("get unit with image id %s,%s" % (storage_unit_filter, imageid))
        storage_units = self.connect.put(storage_unit_filter, unit_body)
        return self._get_response_data(storage_units), self._get_response_error(
            storage_units), self._get_response_description(storage_units)

    def get_storage_unit_by_name(self, storage_unit_name):
        storage_unit_filter = self.ebackup_unit_uri + \
                              '?filter=name::' + storage_unit_name
        result = self.connect.get(storage_unit_filter)
        return self._get_response_data(result), self._get_response_error(
            result), self._get_response_description(result)

    def get_storage_pool_by_name(self, storage_pool_name):
        storage_pool_filter = self.ebackup_manager_pool_list_uri + \
                              '?filter=name::' + storage_pool_name
        resp = self.connect.get(storage_pool_filter)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_storage_repository_by_name(self, storage_repository_name):
        storage_repository_filter = self.ebackup_manager_repository_uri + \
                                    '?filter=name::' + storage_repository_name
        resp = self.connect.get(storage_repository_filter)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_storage_pool_by_id(self, storage_pool_id):
        resp = self.connect.get(self.ebackup_manager_pool_uri +
                                '?id=' + storage_pool_id)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def create_storage_pool(self, storage_name, storage_unit_id):
        pool_body = {
            "NAME": storage_name,
            "DESCRIPTION": "Storage pool for eBackup under OpenStack",
            "ALARMTHRESHOLD": cst.ALARM_THRESHOLD,
            "BRICKLIST": [
                {
                    "ID": str(storage_unit_id)
                }
            ]}
        _body_str = 'create pool body is:' + str(pool_body)
        LOG.info(_body_str)
        result = self.connect.post(self.ebackup_pool_uri, pool_body)
        if self._get_response_error(result) in [ErrorCode.EXISTS_NAME, ErrorCode.STORAGE_UNIT_BUSY]:
            LOG.info('Storage pool %s already exist!', storage_name)
            return self.get_storage_pool_by_name(storage_name)

        # return storage pool list
        return [self._get_response_data(result)], self._get_response_error(
            result), self._get_response_description(result)

    def get_plan_by_name(self, plan_name):
        plan_filter = self.ebackup_manager_plan_list_uri + \
                      '?filter=name::' + plan_name
        resp = self.connect.get(plan_filter)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_copy_plan_by_name(self, copy_plan_name):
        copy_plan_filter = self.ebackup_copy_plan_uri + '?filter=NAME::' + copy_plan_name
        resp = self.connect.get(copy_plan_filter)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_backup_statistics(self):
        sh_url = '/vbackup_statistic_summary/show_statistics'
        body = {
            "TYPE": 57369,
            "TIMESPAN": 2
        }
        resp = self.connect.put(sh_url, body, self.retry)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_backup_plan_count(self):
        get_url = self.ebackup_plan_uri + '/count'
        resp = self.connect.get(get_url, self.retry)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def get_copy_plan_count(self):
        get_url = self.ebackup_copy_plan_uri + '/count'
        resp = self.connect.get(get_url, self.retry)
        return self._get_response_data(resp), self._get_response_error(
            resp), self._get_response_description(resp)

    def create_storage_repository(self, name, pool_id, pool_name, fre_cap):
        """ create storage repository. """
        rep_body = {
            "NAME": name,
            "DESCRIPTION": "Repository for eBackup under OpenStack",
            "ALARMTHRESHOLD": cst.ALARM_THRESHOLD,
            "TOTALCAPACITY": fre_cap,
            "PARENTTYPE": cst.POOL_TYPE_ID,
            "PARENTID": str(pool_id),
            "PARENTNAME": pool_name,
            "FULLTOTALFLAG": "1"
        }
        rep_body_str = 'create repository body is:' + str(rep_body)
        LOG.info(rep_body_str)
        result = self.connect.post(self.ebackup_repository_uri, rep_body)
        if self._get_response_error(result) in [ErrorCode.EXISTS_NAME]:
            LOG.info('Storage repository %s already exist!', name)
            return self.get_storage_repository_by_name(name)

        # return storage repository list
        return [self._get_response_data(result)], self._get_response_error(
            result), self._get_response_description(result)

    def create_backup_plan(self, object_name):
        """ create backup plan """
        plan_body = {
            "DESCRIPTION": "Backup Plan for eBackup under OpenStack",
            "HOSTGROUPNAME": object_name["set_name"],
            "NAME": object_name["plan_name"],
            "NAMESPACENAME": object_name["repository_name"],
            "POLICYNAME": object_name["policy_name"],
            "STATUS": cst.PLAN_STATUS_DEACTIVATION}
        plan_body_str = 'create plan body is:' + str(plan_body)
        LOG.info(plan_body_str)
        result = self.connect.post(self.ebackup_plan_uri, plan_body)
        if self._get_response_error(result) in [ErrorCode.EXISTS_NAME]:
            LOG.info('backup plan %s already exist!', object_name["plan_name"])
            return self.get_plan_by_name(object_name["plan_name"])

        # return backup plan
        return [self._get_response_data(result)], self._get_response_error(
            result), self._get_response_description(result)

    def active_backup_plan(self, active_body):
        result = self.connect.put(self.ebackup_plan_uri + '/run', active_body)
        return self._get_response_data(result), self._get_response_error(
            result), self._get_response_description(result)

    def get_task_id_by_request(self, task_type, requestid):
        task_filter = "?filter=TASKTYPE::" + str(task_type) + "%20and%20REQUEST_ID::" + str(
            requestid) + "&range=[0-15]" + '&sortby=CREATETIME,d'
        task_url = self.ebackup_progress_uri + task_filter
        LOG.info('get task id url:%s' % task_url)
        count = 0
        while count < cst.TASK_QUERY_COUNT:
            result = self.connect.get(task_url)
            meta_error = self._get_response_error(result)
            if meta_error:
                LOG.error(meta_error + ';' + self._get_response_description(result))
                raise Exception(meta_error + ';' + self._get_response_description(result))
            data = self._get_response_data(result)
            if data:
                task_id = data[0].get("ID")
                task_type_ret = data[0].get("TASKTYPE")
                LOG.info("get Task %s type %s" % (task_id, task_type_ret))
                return task_id
            time.sleep(cst.TIME_SLEEP_30)
            count += 1
        LOG.error(ErrorCode.TSKMG_TASK_ID_NOT_EXIST + ';' + 'task not exist')
        raise Exception(ErrorCode.TSKMG_TASK_ID_NOT_EXIST + ';' + 'task not exist')

    def get_task_info(self, task_id):
        task_url = self.ebackup_progress_uri + "/%s" % task_id
        LOG.debug('get task info url:%s' % task_url)
        result = self.connect.get(task_url)
        return self._get_response_data(result), self._get_response_error(
            result), self._get_response_description(result)

    def get_snap_by_taskid(self, task_id):
        get_image_id_url = self.ebackup_image_uri + "?filter=JOBID::" + task_id + "&range=[0-15]&sortby=CREATETIME,d"
        result = self.connect.get(get_image_id_url)
        return self._get_response_data(result), self._get_response_error(
            result), self._get_response_description(result)
