#!/usr/bin/python3

import glob
import json
import os
import shlex
import shutil
from functools import wraps

from common.common_define import ArchiveType
from common.common_define import CommonDefine
from common.common_define import Permission
from common.configreader import ConfigHandler
from common.configreader import g_cfg_common
from common.exceptions import ClientNotSupport, DirSpaceNoEnough, DBUserPathError
from common.logutils import Logger
from common.logutils import SUBAGENT_INSTALL_LOG
from common.utils import Utils, log_with_exception
from hostinfo.check_db_install import CheckDbInstallHandle
from hostinfo.check_db_install import DbType
from subagent.sub_common_oper import DbInstallInfo
from subagent.sub_common_oper import DbInstanceInfo
from subagent.sub_common_oper import SubCommonOper

if CommonDefine.IS_WINDOWS:
    import winreg

AI_USER = "root"
SPACE_LIMIT = 1
assist_install_path = Utils.get_install_path()
sub_agent_root_path = Utils.get_sub_agent_root_path()
assist_root_path = Utils.get_agent_assist_root_path()
conf_path = Utils.get_conf_path()
sub_agent_path = os.path.realpath(os.path.join(sub_agent_root_path, "CDMClient/HWClientService/"))
db_agent_path = os.path.realpath(os.path.join(sub_agent_root_path,
                                              "CDMClient/HWClientService/package/AggregateApp/runner"))
aggregateapp_path = os.path.realpath(os.path.join(sub_agent_path, "AggregateApp"))
log = Logger().get_logger(SUBAGENT_INSTALL_LOG)
PKG_PATH = Utils.get_pkg_root_path()
SERVER_PORT = "9704"
SHELL_EXT = ".sh"


def _delete_temp_file(tmp_file):
    file_base_name = os.path.basename(tmp_file)
    if not os.path.exists(os.path.realpath(tmp_file)) and not os.path.isfile(tmp_file):
        log.error(f"{file_base_name} is not exits or {file_base_name} is not file.")
        return False
    try:
        os.remove(tmp_file)
        log.info(f"Remove {file_base_name} success.")
        return True
    except Exception as error:
        log.error(f"Remove {file_base_name} fail:{error}.")
        return False


def _delete_temp_catalogue(catalogue):
    catalogue_base_name = os.path.basename(catalogue)
    if not os.path.exists(os.path.realpath(catalogue)) and not os.path.isdir(catalogue):
        log.error(f"{catalogue_base_name} is not exits or {catalogue_base_name} is not catalogue.")
        return False
    try:
        shutil.rmtree(catalogue)
        log.info(f"Remove {catalogue_base_name} success.")
        return True
    except Exception as error:
        log.error(f"Remove {catalogue_base_name} fail:{error}.")
        return False


def _delete_dpa_install_file():
    if CommonDefine.IS_WINDOWS:
        return True
    install_file = ['install.sh', '2_params/KingBase.sh', 'multiuser.sh',
                    'multiusercmn.sh', '2_params/PostgreSQL.sh']
    for file in install_file:
        des_file = os.path.join(aggregateapp_path, file)
        if os.path.isfile(des_file):
            _delete_temp_file(des_file)
    return True


def delete_all_temp_file(check_install=False):
    def clear_temp_file(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            ret = func(*args, **kwargs)
            tmp_file = os.path.join(sub_agent_root_path, 'testcfg.tmp')
            if ret is not True:
                if os.path.exists("/etc/systemd/system/HWClientService.service"):
                    _delete_temp_file("/etc/systemd/system/HWClientService.service")
                db_info_file = os.path.join(assist_root_path, 'conf/db_info.json')
                if check_install and os.path.isfile(db_info_file):
                    _delete_temp_file(db_info_file)
                log.error("Install failed, about to remove SubAgent dir.")
                SubCommonOper.clear_useless_dir(sub_agent_root_path, True)
            if os.path.isfile(tmp_file):
                _delete_temp_file(tmp_file)
            _delete_dpa_install_file()
            return ret

        return wrapper

    return clear_temp_file


def clear_useless_pkg(func):
    @wraps(func)
    def wrapper(*args, **kw):
        ret = func(*args, **kw)
        files = os.listdir(PKG_PATH)
        for file in files:
            if file.startswith("CDM") or file.startswith("plugins"):
                _delete_temp_file(os.path.join(PKG_PATH, file))
        return ret

    return wrapper


class SubInstallHandle(object):

    def __init__(self):
        self._sys_bit = ''
        self._sys_arch = ''
        self._sys_name = ''
        self._sys_ver = ''
        self._local_ip = ""
        self._server_ip = ""
        self._language = ""
        self._db_flag_exist = "false"
        self._db_install_info = DbInstallInfo()
        self._index_dict = {}
        self._type_dict = {}

        self._file_name = ""
        self._plugin_name = ""
        self._oracle_plugin = ""

    @log_with_exception(log)
    def _get_os_info(self):
        arch = Utils.get_host_os_architecture()
        if CommonDefine.IS_WINDOWS:
            self._sys_name = "Windows"
            if arch.find("64") != -1:
                self._sys_bit = "x64"
            else:
                self._sys_bit = "i386"
        else:
            _, self._sys_name = Utils.execute_cmd(shlex.split("uname -s"))
            _, self._sys_arch = Utils.execute_cmd(shlex.split("uname -m"))
            if not all([self._sys_name, self._sys_arch]):
                log.error("Get system type or arch failed.")
                return False
            if self._sys_arch.find("x86_64") != -1:
                self._sys_bit = "x64"
            else:
                self._sys_bit = self._sys_arch
            if self._sys_name.find("Linux") != -1:
                self._sys_ver = Utils.get_linux_kernel_ver()
        self._sys_name = self._sys_name.strip()
        self._sys_bit = self._sys_bit.strip()
        self._sys_ver = self._sys_ver.strip()
        return True

    def _adapt_sub_agent_pkg(self):
        if not self._get_os_info():
            log.error("Get os info failed.")
            return False
        self._local_ip = Utils.get_host_ip()
        if not Utils.get_ip_type(self._local_ip):
            log.error(f"Get local ip failed, ip[{self._local_ip}].")
            return False
        log.info(f"System: {self._sys_name}, arch:{self._sys_bit}, sys version:"
                 f"{self._sys_ver}, local ip:{self._local_ip}, server ip:"
                 f"{self._server_ip}, file name: {self._file_name}.")
        if (self._file_name.find(self._sys_name) == -1) or (self._file_name.find(self._sys_bit) == -1):
            return False
        if (self._sys_name.find('Linux') != -1) and (self._file_name.find(self._sys_ver) == -1):
            return False
        return True

    def mod_server_ip(self, dpa_server_ip):
        self._server_ip = dpa_server_ip

    @staticmethod
    def _write_db_info(database_info):
        db_info_file = os.path.join(assist_root_path, 'conf/db_info.json')
        try:
            with open(db_info_file, "w", encoding='utf8') as f:
                json.dump(database_info, f, ensure_ascii=False)
        except Exception as err:
            log.error(f"Write db info failed, error:{err}.")
            return False
        Utils.mod_chmod(db_info_file, Permission.PERMISSION_600)
        return True

    @staticmethod
    def _get_oracle_file_type():
        files = os.listdir(PKG_PATH)
        for file in files:
            if file.startswith("plugins-oracle") and file.endswith("zip"):
                return True
        return False

    def _set_oracle_plugin(self):
        self._oracle_plugin = f"--add_plugin={self._plugin_name}"

    def _process_oracle_plugin(self):
        if SubInstallHandle._get_oracle_file_type() or not CommonDefine.IS_WINDOWS:
            self._set_oracle_plugin()
            return True
        Utils.unpack(self._plugin_name, PKG_PATH, ArchiveType.TAR)
        _delete_temp_file(self._plugin_name)
        files = os.listdir(PKG_PATH)
        for file in files:
            if file in ["components", "locale"]:
                _delete_temp_catalogue(os.path.join(PKG_PATH, file))
            if file in ["orasbt.dll"]:
                _delete_temp_file(os.path.join(PKG_PATH, file))
        new_files = os.listdir(PKG_PATH)
        for oracle_file in new_files:
            if "plugins-oracle" in oracle_file:
                self._plugin_name = os.path.join(PKG_PATH, oracle_file)
                self._set_oracle_plugin()
                return True
        log.error("Oracle plugin error.")
        return False

    def _check_msg_param(self, body_dict, is_upgade=False):
        dpa_ip = body_dict.get("server_ip")
        try:
            is_valid_ip = Utils.get_ip_type(dpa_ip)
        except Exception as error:
            is_valid_ip = False
            if not is_upgade:
                log.error(f"Dpa server ip param invalid, dpa_ip: {dpa_ip}, {error}.")
                return False
        if is_valid_ip and not self._server_ip:
            self._server_ip = dpa_ip
        self._file_name = body_dict.get("file_name")
        if not Utils.check_file_name(self._file_name):
            log.error(f"Input wrong name, pls check, file_name:{self._file_name}.")
            return False
        file_full_name = os.path.join(PKG_PATH, self._file_name)
        if not os.path.isfile(file_full_name):
            log.error(f"Pkg not exists, pkg_name:{file_full_name}.")
            return False
        plugin_name = body_dict.get("plugin_name")
        if plugin_name and Utils.check_file_name(plugin_name):
            self._plugin_name = os.path.realpath(os.path.join(PKG_PATH, plugin_name))
            if not os.path.isfile(self._plugin_name):
                log.error(f"Plugin not exists, plugin_name:{self._plugin_name}.")
                return False
            if not self._process_oracle_plugin():
                return False

        self._language = g_cfg_common.get_option('input_info', 'language')
        if not all([self._server_ip, SERVER_PORT, self._language]):
            log.error(f"Cannot read server ip, or server port[{SERVER_PORT}], language[{self._language}].")
            return False
        if body_dict.get("install_info") is None:
            log.info("There is no db info, to be support fileset backup.")
            return True
        install_info = body_dict.get("install_info")
        db_info = json.loads(install_info)
        database_info = db_info.get('database')
        if database_info:
            if not self._write_db_info(database_info):
                log.error("Write db info failed.")
                return False
        return True

    def check_param_valid(self, body_dict, is_upgade=False):
        if not self._check_msg_param(body_dict, is_upgade):
            log.error("Input param invalid.")
            return False
        if not self._adapt_sub_agent_pkg():
            log.error(f"The current system is {self._sys_name}, arch is {self._sys_bit}, "
                      f"the package {self._file_name} is not match.")
            raise ClientNotSupport
        return True

    @staticmethod
    def _check_install_dir():
        if os.path.exists(sub_agent_root_path) and not SubCommonOper.clear_useless_dir(sub_agent_root_path):
            log.error("Remove old installation directory failed.")
            return False
        os.makedirs(sub_agent_root_path)
        Utils.set_path_permissions(sub_agent_root_path, Permission.PERMISSION_755)
        return True

    @staticmethod
    def check_space():
        free = Utils.get_free_space_mb(assist_install_path)
        if free < SPACE_LIMIT:
            log.error(f"The installation directory has no enough space, space[{free}].")
            raise DirSpaceNoEnough
        return True

    def _adjust_db_index(self):
        file_list = sorted(glob.glob(f"{db_agent_path}/*_params/*"))
        db_info_list = []
        for file_name in file_list:
            base_name = os.path.basename(file_name)
            db_info, ext = os.path.splitext(base_name)
            if ext == SHELL_EXT:
                db_info_list.append(db_info.lower())
        for index, db_info in enumerate(db_info_list):
            self._index_dict.update({db_info: str(index + 1)})

    def _update_db_info_dict(self):
        self._type_dict.update({DbType.mysql.name: DbType.mysql})
        self._type_dict.update({DbType.oracle.name: DbType.oracle})
        self._type_dict.update({DbType.dameng.name: DbType.dameng})
        self._type_dict.update({DbType.postgresql.name: DbType.postgresql})
        self._type_dict.update({DbType.kingbase.name: DbType.kingbase})

    def _deserialize_db_info(self, db_info_dict):
        db_type_str = str(db_info_dict.get("database_type")).strip()
        db_info_list = db_info_dict.get("database_info")
        db_user_list = db_info_dict.get("runtime_users")
        db_type = self._type_dict.get(db_type_str)
        self._db_install_info.db_index = self._index_dict.get(db_type_str)
        log.info(f"Type_dict:{self._type_dict} index_dict: {self._index_dict}.")
        if isinstance(db_info_list, list) and db_info_list:
            for db_info in db_info_list:
                instance_info = DbInstanceInfo()
                instance_info.db_user = db_info.get("runtime_user")
                instance_info.db_path = db_info.get("path")
                if db_info.get("application_type"):
                    instance_info.db_application_type = db_info.get("application_type")
                    instance_info.db_instance_path = db_info.get("instance_path")
                self._db_install_info.db_instance_list.append(instance_info)
        if isinstance(db_user_list, list) and db_user_list:
            for db_user in db_user_list:
                instance_info = DbInstanceInfo()
                instance_info.db_user = db_user
                instance_info.db_path = db_info_dict.get("path")
                self._db_install_info.db_instance_list.append(instance_info)
        return db_type

    def _check_db_info(self, db_type: DbType):
        if not (db_type.name in DbType.__members__):
            log.error("Unsupported DB type.")
            return False
        self._db_flag_exist = "true"
        db_install_handle = CheckDbInstallHandle()
        db_install_handle.check_db_install(db_type)
        for db_instance in self._db_install_info.db_instance_list:
            if db_type in {DbType.mysql, DbType.postgresql, DbType.kingbase} \
                    and not os.path.exists(db_instance.db_path):
                log.error("Input db path not exists.")
                raise DBUserPathError
            if not Utils.find_user(db_instance.db_user):
                log.error("Input user not exists.")
                raise DBUserPathError
        return True

    def _renew_db_info(self, db_info_file):
        try:
            with open(db_info_file, 'r') as file:
                db_info_dict = json.load(file)
        except Exception as err:
            log.error(f"Failed to read tmp file, error:{err}.")
            return False
        if not isinstance(db_info_dict, dict):
            log.warning("Db file content type error, will install fileset-type SubAgent.")
            return True
        if db_info_dict.get("database_type") == DbType.oracle.name:
            log.info(f"Oracle plugin is: {self._plugin_name}")
        if CommonDefine.IS_WINDOWS:
            return True
        log.info("Begin to adjust db index.")
        self._adjust_db_index()
        self._update_db_info_dict()
        db_type = self._deserialize_db_info(db_info_dict)
        if CommonDefine.IS_LINUX and not db_type:
            log.error("Not support database type.")
            return False
        if not self._check_db_info(db_type):
            log.error("Check db info failed.")
            return False
        return True

    def _write_param_tmp(self):
        tmp_file = os.path.join(sub_agent_root_path, 'testcfg.tmp')
        db_info_file = os.path.join(assist_root_path, 'conf/db_info.json')
        if os.path.isfile(db_info_file):
            log.info("Begin to read db info.")
            if not self._renew_db_info(db_info_file):
                log.error("Read db info failed.")
                return False
        if CommonDefine.IS_WINDOWS:
            return True
        info_list = []
        for db_instance in self._db_install_info.db_instance_list:
            info_list.append(f"DB_INFO={self._db_install_info.db_index} "
                             f"{db_instance.db_user} {db_instance.db_path} "
                             f"{db_instance.db_application_type} "
                             f"{db_instance.db_instance_path}\n")
        try:
            with open(tmp_file, 'a') as file:
                file.write(f"SYS_NAME={self._sys_name}\n")
                file.write(f"SYS_ARCH={self._sys_arch}\n")
                file.write(f"SYS_BIT={self._sys_bit}\n")
                file.write(f"SERVER_IP={self._server_ip}\n")
                file.write(f"SERVER_PORT={SERVER_PORT}\n")
                file.write(f"DB_FLAG_EXIST={self._db_flag_exist}\n")
                file.writelines(info_list)
            Utils.mod_chmod(tmp_file, Permission.PERMISSION_600)
        except Exception as err:
            log.error(f"Failed to write tmp file, error:{err}.")
            return False
        return True

    def _unpack(self):
        pkg_type = ArchiveType.ZIP
        if not CommonDefine.IS_WINDOWS:
            pkg_type = ArchiveType.TAR
        full_name = os.path.realpath(os.path.join(PKG_PATH, self._file_name))
        Utils.unpack(full_name, sub_agent_root_path, pkg_type)
        return True

    def _replace_script(self):
        try:
            if (self._sys_name.find("Linux") != -1) or (self._sys_name.find("NeoKylin") != -1):
                src_dir = os.path.join(assist_root_path, 'bin/Linux/Linux')
                Utils.execute_cmd(shlex.split(f"chmod +x {src_dir}/*"))
                shutil.copy(os.path.join(src_dir, 'AggregateApp/install.sh'),
                            os.path.join(db_agent_path, 'install.sh'))
                shutil.copy(os.path.join(src_dir, 'AggregateApp/KingBase.sh'),
                            os.path.join(db_agent_path, '2_params/KingBase.sh'))
                shutil.copy(os.path.join(src_dir, 'AggregateApp/PostgreSQL.sh'),
                            os.path.join(db_agent_path, '2_params/PostgreSQL.sh'))
                shutil.copy(os.path.join(src_dir, 'AggregateApp/multiuser.sh'),
                            os.path.join(db_agent_path, 'multiuser.sh'))
                shutil.copy(os.path.join(src_dir, 'AggregateApp/multiusercmn.sh'),
                            os.path.join(db_agent_path, 'multiusercmn.sh'))
                shutil.copy(os.path.join(src_dir, 'start.sh'),
                            os.path.join(sub_agent_path, 'start.sh'))
                shutil.copy(os.path.join(src_dir, 'stop.sh'),
                            os.path.join(sub_agent_path, 'stop.sh'))
                return True
            elif CommonDefine.IS_WINDOWS:
                src_dir = os.path.join(assist_root_path, 'bin/win32')
                shutil.copyfile(os.path.join(src_dir, 'start.bat'),
                                os.path.join(sub_agent_path, 'start.bat'))
                shutil.copyfile(os.path.join(src_dir, 'stop.bat'),
                                os.path.join(sub_agent_path, 'stop.bat'))
                return True
            else:
                log.error(f"Unsurppoted system, sys[{self._sys_name}].")
                return False
        except Exception as err:
            log.error(f"Failed to copy file, error:{err}.")
            return False

    def _query_agent_version(self):
        version_file = os.path.realpath(os.path.join(sub_agent_path, 'AggregateApp/VersionDetails'))
        if not os.path.isfile(version_file):
            log.error("Version file not exists.")
            return False
        agent_version = ConfigHandler.read_conf_without_section(version_file, "Package Version")
        if not agent_version:
            log.error("Read agent version failed, please confirm the SubAgent has been install successfully.")
            return False
        self._agent_version = agent_version.split("\n")[0]
        return True

    def _query_machine_code(self):
        machine_code_file = "/var/lib/CDM/machinecode"
        machine_code_path = "C:/ProgramData/Huawei"
        self._machine_code = ""
        if os.path.isfile(machine_code_file):
            try:
                with open(machine_code_file, "r", encoding='utf8') as f:
                    machine_code = f.read().strip('\n')
                    byte_code = str.encode(machine_code)
                    self._machine_code = byte_code.decode('utf-8-sig')
                    return True
            except Exception as err:
                log.error(f"Read machine code file failed, error: {err}.")
        elif os.path.isdir(machine_code_path):
            for file in os.listdir(machine_code_path):
                file_path = os.path.join(machine_code_path, file)
                if os.path.isfile(file_path):
                    self._machine_code = os.path.basename(file_path)
                    return True
            try:
                winreg_path = r'SOFTWARE\Huawei'
                winreg_handle = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, winreg_path, 0, (winreg.KEY_WOW64_64KEY +
                                                                                           winreg.KEY_READ))
                machine_code, _ = winreg.QueryValueEx(winreg_handle, "MachineCode")
                log.info(f"MachineCode is {machine_code}.")
                self._machine_code = machine_code
                return True
            except Exception as err:
                log.error(f"Read regedit fail,error: {err}.")
                return False
        else:
            log.error("Cannot read machine code, please make sure that you have correctly installed the SubAgent.")
        return False

    def _read_agent_info(self, f, agent_info_dict):
        old_info_dict = json.load(f)
        log.info(f"old_info_dict:{old_info_dict}")
        # 获取到的info_list是个列表,里面至多两条信息一个
        # 是agent的一个是dpa的，判断这个列表长度是否大于1
        # 大于1则是DPA升级情况，需要更新信息。
        info_list = old_info_dict.get("agent_list")
        if len(info_list) > 1:
            info_list.pop()
        info_list.append({self._file_name: agent_info_dict})
        new_info_dict = {"agent_list": info_list}
        log.info(f"The new_info_dict:{new_info_dict}.")
        return new_info_dict

    def _renew_agent_info(self):
        if not self._query_agent_version():
            log.error("Query agent version failed.")
            return False
        if not self._query_machine_code():
            log.error("Query machine code failed.")
            return False
        agent_info_file = os.path.join(assist_root_path, 'conf/agent_list.json')
        agent_info_dict = {"agent_name": self._file_name,  # agent安装状态
                           "agent_version": self._agent_version,  # agent版本
                           "agent_id": self._machine_code}  # agent id
        if not os.path.exists(agent_info_file):
            new_info_dict = {
                "agent_list": [{self._file_name: agent_info_dict}]}
        else:
            try:
                with open(agent_info_file, "r", encoding='utf8') as f:
                    new_info_dict = self._read_agent_info(f, agent_info_dict)
            except Exception as err:
                log.error(f"Read agent info file failed, error[{err}].")
                return False
        try:
            with open(agent_info_file, "w", encoding='utf8') as f:
                json.dump(new_info_dict, f, ensure_ascii=False)
        except Exception as err:
            log.error(f"Write agent info file failed, error[{err}].")
            return False
        return True

    def install_client(self):
        if not self._check_install_dir():
            log.error("Install path check error.")
            return False
        self._unpack()
        if not self._replace_script():
            log.error("Replace file error.")
            return False
        if not self._write_param_tmp():
            log.error("Write tmp file error.")
            return False
        os.chdir(sub_agent_path)
        if self._language == "Chinese":
            self._language = "zh"
        else:
            self._language = "en"
        if CommonDefine.IS_WINDOWS:
            ret_code, _ = Utils.execute_cmd(shlex.split(f"install.bat --selfip={self._local_ip} "
                                                        f"--serverip={self._server_ip} --language={self._language}"
                                                        f" --silent=y --volume_cdp=n {self._oracle_plugin}",
                                                        posix=False))
        else:
            ret_code, _ = Utils.execute_cmd([
                "bash", "install.sh", f"--selfip={self._local_ip}", f"--serverip={self._server_ip}",
                f"--language={self._language}", "--silent=y --volume_cdp=n --xenserver=n", f"{self._oracle_plugin}"
            ])
        if ret_code != 0:
            log.error(f"Install subAgent failed, ret[{ret_code}].")
            return False
        if not self._renew_agent_info():
            log.error("Renew agent info failed.")
            return False
        if os.path.isfile(self._plugin_name):
            _delete_temp_file(self._plugin_name)
        os.chdir(assist_install_path)
        return True

    @delete_all_temp_file(check_install=True)
    @clear_useless_pkg
    @log_with_exception(log)
    def install_handle(self, body_str):
        self.check_space()
        try:
            body_dict = json.loads(body_str)
        except Exception as err:
            log.error(f"Input param invalid，body_str is {body_str}, error[{err}].")
            return False
        cur_path = os.path.join(assist_root_path, 'bin/assist/subagent/')
        os.chdir(cur_path)
        if not isinstance(body_dict, dict):
            log.error("Input param invalid.")
            return False
        if not self.check_param_valid(body_dict):
            log.error("Input param invalid.")
            return False
        if not self.install_client():
            log.error("Install client failed.")
            return False
        log.info("Install subAgent success.")
        return True
