#!/usr/bin/python3

import json
import os
import shlex
import shutil
import time
from functools import wraps

from common.common_define import CommonDefine
from common.configreader import ConfigHandler
from common.logutils import Logger, SUBAGENT_INSTALL_LOG
from common.utils import Utils, log_with_exception
from common.utils import VAR_CDM_LIB
from common.utils import VAR_CDM_LOG

assist_path = Utils.get_agent_assist_root_path()
sub_agent_root_path = Utils.get_sub_agent_root_path()
sub_agent_path = os.path.realpath(os.path.join(sub_agent_root_path, "CDMClient/HWClientService/"))
log = Logger().get_logger(SUBAGENT_INSTALL_LOG)


def clear_sub_path(func):
    @wraps(func)
    def wrapper(*args, **kw):
        ret = func(*args, **kw)
        time.sleep(2)
        db_info_file = os.path.join(assist_path, 'conf/db_info.json')
        if not os.path.isfile(db_info_file):
            log.info("DB info file not exists.")
            return ret
        try:
            os.remove(db_info_file)
        except Exception as err:
            log.error(f"Remove db info file failed, error:{err}.")
        log.info("Uninstall the subAgent success.")
        return ret

    return wrapper


class SubUninstallHandle(object):
    def __init__(self):
        self._file_name = ""

    def _remove_agent_info(self):
        if not self._file_name:
            log.info("Remove all subAgent.")
            return True
        agent_info_file = os.path.join(assist_path, 'conf/agent_list.json')
        if not os.path.exists(agent_info_file):
            log.warning("Agent info file not exists.")
            return True
        try:
            with open(agent_info_file, "r", encoding='utf8') as f:
                old_info_dict = json.load(f)
        except Exception as err:
            log.error(f"Read agent info file failed, error[{err}].")
            return False
        log.info(f"The Old_info_dict: {old_info_dict}, agent_name: {self._file_name}.")
        if not isinstance(old_info_dict, dict):
            log.error("Agent info file type error.")
            return False
        info_list = old_info_dict.get("agent_list")
        if not isinstance(info_list, list):
            log.error("Agent info list type error.")
            return False
        remove_info = None
        for info in info_list:
            if not isinstance(info, dict):
                log.error("Agent info type error.")
                return False
            agent_info_dict = list(info.values())[0]
            if not isinstance(agent_info_dict, dict):
                log.error("Agent info type error.")
                return False
            agent_name = agent_info_dict.get("agent_name")
            if agent_name.find(self._file_name) != -1:
                remove_info = info
                break
        if not remove_info:
            log.info("No match record.")
            return True
        info_list.remove(remove_info)
        try:
            with open(agent_info_file, "w", encoding='utf8') as f:
                json.dump({"agent_list": info_list}, f, ensure_ascii=False)
        except Exception as err:
            log.error(f"Write agent info file failed, error[{err}].")
            return False
        return True

    @staticmethod
    def get_agent_version():
        version_file = os.path.realpath(os.path.join(sub_agent_path, 'AggregateApp/VersionDetails'))
        if not os.path.isfile(version_file):
            version_file = os.path.realpath(os.path.join(sub_agent_path, 'VersionDetails'))
            if not os.path.isfile(version_file):
                return None
        agent_version = ConfigHandler.read_conf_without_section(version_file, "Package Version")
        if not agent_version:
            log.error("Read agent version failed, please confirm the SubAgent has been install successfully.")
            return None
        return agent_version.split("\n")[0]

    @staticmethod
    def delete_sub_agent_dir():
        try:
            if os.path.exists(sub_agent_root_path):
                shutil.rmtree(sub_agent_root_path)
                log.info(f"Remove {sub_agent_root_path} success.")
        except Exception as error:
            log.error(f"Delete sub agent dir failed, error:{error}.")

    @staticmethod
    def delete_var_catalogue():
        try:
            if not CommonDefine.IS_WINDOWS:
                if os.path.exists(VAR_CDM_LIB):
                    shutil.rmtree(VAR_CDM_LIB)
                    log.info("Remove /var/lib/CDM success.")
                if os.path.exists(VAR_CDM_LOG):
                    shutil.rmtree(VAR_CDM_LOG)
                    log.info("Remove /var/log/CDM success.")
        except Exception as error:
            log.error(f"Delete var catalogue failed, error:{error}.")

    @staticmethod
    def uninstall_dpa():
        agent_version = SubUninstallHandle.get_agent_version()
        if Utils.compare_version(agent_version, "8.0.1.7"):
            if CommonDefine.IS_WINDOWS:
                ret_code, _ = Utils.execute_cmd("uninstall.bat --force --silent=y")
            else:
                ret_code, _ = Utils.execute_cmd(shlex.split("bash uninstall.sh --force --silent=y"))
        else:
            if CommonDefine.IS_WINDOWS:
                ret_code, _ = Utils.execute_cmd("uninstall.bat")
            else:
                ret_code, _ = Utils.execute_cmd(shlex.split("bash uninstall.sh"))
        if ret_code != 0:
            log.error(f"Uninstall subAgent failed, ret[{ret_code}].")
            return False

        time.sleep(2)
        return True

    @clear_sub_path
    @log_with_exception(log)
    def uninstall_handle(self, file_name="", is_uninstall=True):
        self._file_name = file_name
        client_path = os.path.join(sub_agent_root_path, "CDMClient/HWClientService")
        pkg_path = os.path.join(Utils.get_pkg_root_path(), self._file_name)
        if os.path.isfile(pkg_path) and is_uninstall:
            os.remove(pkg_path)
            log.info("Remove agent package success.")
        if not os.path.exists(client_path):
            log.info("Not found the SubAgent.")
            return True
        if CommonDefine.IS_WINDOWS:
            script_name = os.path.join(client_path, "uninstall.bat")
        else:
            script_name = os.path.join(client_path, "uninstall.sh")
        if not os.path.isfile(script_name):
            log.error(f"Not found the {script_name}.")
            return False
        os.chdir(client_path)
        if not SubUninstallHandle.uninstall_dpa():
            return False
        os.chdir(assist_path)
        SubUninstallHandle.delete_sub_agent_dir()
        SubUninstallHandle.delete_var_catalogue()
        if not self._remove_agent_info():
            log.error("Remove agent info failed.")
            return False
        log.info("Uninstall subAgent success, the install path will be deleted.")
        return True
