import json
import os
import threading
from enum import Enum

import time

from agentassisthandle.replace_cert import AgentAssistCertHandle
from agentassisthandle.upgradehandle import AgentAssistUpgradeHandle
from common.common_define import CommonDefine
from common.logutils import Logger
from common.utils import Utils, log_with_exception

assist_conf_path = Utils.get_conf_path()
install_path = Utils.get_install_path()
PKG_PATH = Utils.get_pkg_root_path()
cert_download_path = os.path.join(assist_conf_path, 'cert_new')
log = Logger().get_logger()


class FLAG(Enum):
    pending = 1
    failed = 2
    success = 3


g_uninstall_flag = FLAG.pending
g_mutex = threading.Lock()


class AgentAssistMessage(object):
    def __init__(self):
        self._nul_ret = {"job_status": "failed",  # 执行状态
                         "type": "0",  # agent类型：0-agentframework,1-subagent
                         "version": "",  # agent版本
                         "client_id": ""}  # agent id
        self._file_name = ""
        self._machine_code = ""
        self._agent_version = ""

    def _check_msg_param(self, body_dict):
        self._file_name = body_dict.get("file_name")
        if not Utils.check_file_name(self._file_name):
            log.error(f"check file_name failed, pls check:{self._file_name}")
            return False
        full_name = os.path.realpath(os.path.join(PKG_PATH, self._file_name))
        if not os.path.isfile(full_name):
            return False
        return True

    def write_unistall_flag(self, value):
        global g_uninstall_flag
        g_mutex.acquire()
        g_uninstall_flag = value
        g_mutex.release()

    def _get_agent_id(self):
        agent_info_file = os.path.join(assist_conf_path, 'agent_list.json')
        if not os.path.exists(agent_info_file):
            return False
        with open(agent_info_file, 'r', encoding='utf8')as fp:
            json_data = json.load(fp)
        return json_data["agent_list"][0]["file_name"]["agent_id"]

    def _get_agent_version(self, action):
        if action == "upgrade":
            images_path = os.path.join(install_path, 'AgentAssist_Image')
            agent_info_file = os.path.join(images_path, 'conf/agent_list.json')
        else:
            agent_info_file = os.path.join(assist_conf_path, 'agent_list.json')
        try:
            with open(agent_info_file, 'r', encoding='utf8')as fp:
                json_data = json.load(fp)
            return json_data["agent_list"][0]["file_name"]["agent_version"]
        except Exception as err:
            log.error(f"Failed to get version. {err}")
            return ""

    def uninstall_agentassist(self, body_dict):
        self._machine_code = self._get_agent_id()
        self._agent_version = self._get_agent_version("uninstall")
        if self._agent_version == '':
            return self._nul_ret
        t = threading.Thread(target=_uninstall_process)
        t.start()
        ret_val = {"job_status": "success",  # agent升级状态
                   "type": "0",  # agent类型：0-agentframework,1-subagent
                   "version": self._agent_version,  # agent版本
                   "client_id": self._machine_code}  # agent id
        return ret_val

    def upgrade_agentassist(self, body_dict):
        if not self._check_msg_param(body_dict):
            return self._nul_ret

        self._machine_code = self._get_agent_id()
        upgrade = AgentAssistUpgradeHandle()
        if not upgrade.upgrade_handle(self._file_name):
            return self._nul_ret
        self._agent_version = self._get_agent_version("upgrade")
        if self._agent_version == "":
            return self._nul_ret

        ret_val = {"job_status": "success",  # agent升级状态
                   "type": "0",  # agent类型：0-agentframework,1-subagent
                   "version": self._agent_version,  # agent版本
                   "client_id": self._machine_code}  # agent id
        return ret_val

    def replace_cert(self, body_dict):
        self._machine_code = self._get_agent_id()
        if not AgentAssistCertHandle.replace_cert():
            return self._nul_ret

        ret_val = {"job_status": "success",  # agent升级状态
                   "type": "0",  # agent类型：0-agentframework,1-subagent
                   "version": self._agent_version,  # agent版本
                   "client_id": self._machine_code}  # agent id
        return ret_val


@log_with_exception(log)
def _uninstall_process():
    agentassist_handle = AgentAssistMessage()
    agentassist_handle.write_unistall_flag(FLAG.success)
    time.sleep(1)
    file = 'AgentAssist/uninstall.bat' if CommonDefine.IS_WINDOWS else 'AgentAssist/uninstall.sh'
    Utils.execute_cmd(os.path.join(install_path, file))
