#!/usr/bin/python3
import os
import shlex
import shutil
import multiprocessing
import time
from functools import wraps

from common.common_define import CommonDefine, Permission
from common.logutils import Logger
from common.utils import Utils
from subagent.sub_common_oper import SubCommonOper

log = Logger().get_logger()
assist_path = Utils.get_agent_assist_root_path()


def clear_download_path(func):
    @wraps(func)
    def wrapper(*args, **kw):
        ret = func(*args, **kw)
        if not SubCommonOper.clear_useless_dir(
                AgentAssistCertHandle.cert_download_path):
            log.error(f"Clear cert download dir failed.")
        return ret

    return wrapper


def _restart_assist():
    agent_handle_path = os.path.join(assist_path,
                                     'bin/assist/agentassisthandle')
    if not os.path.isdir(agent_handle_path):
        log.error(f"The {agent_handle_path} not exists.")
        return False
    os.chdir(agent_handle_path)
    if CommonDefine.IS_WINDOWS:
        script_name = os.path.join(agent_handle_path, "restart.bat")
    else:
        script_name = os.path.join(agent_handle_path, "restart.sh")
    if not os.path.isfile(script_name):
        log.error(f"Not found the {script_name}.")
        return False
    time.sleep(5)
    try:
        cmd = shlex.split(f"{script_name} {assist_path}", posix=False) if CommonDefine.IS_WINDOWS else shlex.split(
            f"{script_name} {assist_path}")
        Utils.execute_cmd(cmd)
    except Exception as error:
        log.error(f"Restart agent assist failed, error:{error}.")
    log.info("Success to restart the assist.")
    return True


class AgentAssistCertHandle(object):
    cert_download_path = os.path.join(assist_path, 'conf/cert_new')
    cert_bak_path = os.path.join(assist_path, 'conf/cert_bak')
    cert_path = os.path.join(assist_path, 'conf/cert')

    @staticmethod
    def _backup_old_cert():
        try:
            if os.path.exists(AgentAssistCertHandle.cert_bak_path):
                shutil.rmtree(AgentAssistCertHandle.cert_bak_path)
            shutil.copytree(AgentAssistCertHandle.cert_path,
                            AgentAssistCertHandle.cert_bak_path)
            Utils.mod_chmod(AgentAssistCertHandle.cert_bak_path,
                            Permission.PERMISSION_700)
        except Exception as err:
            log.error(f"Backup old cert file failed, error:{err}.")
            return False
        return True

    @staticmethod
    @clear_download_path
    def replace_cert():
        log.info("Start to replace the Assist certificates.")
        src_dir = AgentAssistCertHandle.cert_download_path
        des_dir = AgentAssistCertHandle.cert_path
        cert_files = ('ca-cert.pem', 'ca.pem', 'ca-key.pem')
        conf_src_file_list = [os.path.join(src_dir, file_name) for file_name in
                              cert_files]
        conf_des_file_list = [os.path.join(des_dir, file_name) for file_name in
                              cert_files]
        if not AgentAssistCertHandle._backup_old_cert():
            log.error(f"Backup old certs failed.")
            return False
        try:
            for src_file, des_file in zip(conf_src_file_list,
                                          conf_des_file_list):
                if not os.path.isfile(src_file):
                    log.error(f"The {src_file} is not exists.")
                    return False
                shutil.copyfile(src_file, des_file)
                Utils.mod_chmod(des_file, Permission.PERMISSION_600)
        except Exception as err:
            log.error(f"Copy cert file failed, error:{err}.")
            return False
        log.info("Replace cert success, begin to restart assist.")
        t = multiprocessing.Process(target=_restart_assist)
        t.start()
        return True
