# -*- coding:utf-8 -*-
import json
import os
import requests

import utils.common.log as logger
from utils.business.manageone_console_util import ManageOneConsoleUtil
from utils.business.manageone_cert_manager import MOCertManager
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util import ManageOneUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.service_monitor.service_monitor import AuthProvider
from utils.business.service_monitor.service_monitor import Host
from utils.business.service_monitor.service_monitor import deploy_agent
from utils.business.unified_pd_platform.account_query import AccountQuery
from utils.common.exception import HCCIException

from plugins.eReplication.common.constant import CSDRSilvanInfos, HTTP_STATUS_OK
from plugins.eReplication.common.constant import ADAPTATION_PKG_UPLOAD_SUCCESS_STATUS
from plugins.eReplication.common.constant import MO_EXCEPT_CODE
from plugins.eReplication.common.constant import CSHASilvanInfos
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import VHASilvanInfos
from plugins.eReplication.common.constant import VMScale
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.constant import Pkg


class API(object):
    CMDB = ManageOneCmdbUtil

    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.condition = Condition(self.project_id)
        self.cmdb_ins = ManageOneCmdbUtil(project_id)
        self.mo_util = ManageOneUtil()
        self.mo_util2 = ManageOneUtil2()

    def get_primary_region_vms_scale(self):
        regions = self.cmdb_ins.get_region_info()
        primary_region_id = \
            [region["regionCode"] for region in regions if
             region["regionType"] == "master"][0]
        service_info = self.cmdb_ins.get_cloud_service_info(
            primary_region_id, Component.REPLICATION)
        if not service_info:
            logger.error(
                "Get cloud service info from CMDB for eReplication failed.")
            raise Exception(
                "Get cloud service info from CMDB for eReplication failed.")
        extend_info = service_info[0].get("extendInfos", [])
        primary_scale = [_info["value"] for _info in extend_info if
                         _info["key"] == "global_service_scale"][0]
        if int(primary_scale.strip("VM")) > 10000:
            return VMScale.CPU_8, VMScale.RAM_16
        return VMScale.CPU_4, VMScale.RAM_8

    def update_cloud_service_extend_info(self, region_id, extend_infos):
        logger.info(f"Update extend info: {extend_infos}.")
        self.cmdb_ins.update_cloud_service_extend_infos(
            region_code=region_id, index_name=Component.REPLICATION,
            version=self.get_version(region_id), extend_infos=extend_infos)

    def get_version(self, region_id):
        cloud_service_info = self.get_cloud_service_info(region_id)
        service_version = cloud_service_info[0].get("version")
        if not service_version:
            raise Exception("Failed to obtain the service version.")
        return service_version

    def get_cloud_service_info(self, region_id):
        cloud_service_info = \
            self.cmdb_ins.get_cloud_service_info(
                region_id, Component.REPLICATION)
        if not cloud_service_info:
            raise Exception(
                f"Failed to get {Component.REPLICATION} service "
                "information from CMDB.")
        logger.info(
            f"Get {Component.REPLICATION} service information "
            f"return {cloud_service_info}.")
        return cloud_service_info

    def upload_console_pkg(self, unzip_path):
        logger.info("Start upload console release package.")
        console_release_path = PKG_API.get_file_abs_path_by_re(
            Pkg.CONSOLE_RELEASE_RE, pkg_path=unzip_path)
        pkg_info = {"pkg_path": console_release_path}
        mo_console_util = ManageOneConsoleUtil()
        result = mo_console_util.upload_static_pkg_to_static_node(
            self.pod_id, is_region=False, pkg_info=pkg_info, check_pkg=False)
        if not result:
            error_msg = "Upload console package to static node failed."
            logger.error(error_msg)
            raise Exception(error_msg)
        logger.info("Upload console pkg success.")

    def deploy_console(self, source_path):
        service_type = f'{Component.CSDR_L},{Component.CSHA_L},' \
                       f'{Component.VHA_L}'
        file_name = os.path.basename(source_path)
        return self.mo_util.deploy_console(
            self.pod_id, "", "", "", "", service_type, is_region=False,
            file_name=file_name)

    def deploy_ic_agent(self, hosts, ssh_user, auth_provider):
        logger.info("Begin to install MOICAgent.")
        ret = deploy_agent(
            self.pod_id, auth_provider, self._host_list(hosts),
            ssh_user, "su")
        if not ret:
            logger.error(f"Failed to install MOICAgent in {hosts}.")
            raise HCCIException("663609", "Failed to install MOICAgent.")
        logger.info(f"Install MOICAgent to {hosts} success.")

    @classmethod
    def auth_provider(cls, dmk_ip, dmk_port, dmk_user_name, dmk_pwd):
        """
        DMK认证
        :return: DMK的认证信息实例,为AuthProvider实例
        """
        return AuthProvider(dmk_ip, dmk_port, dmk_user_name, dmk_pwd)

    @staticmethod
    def _host_list(host_list):
        """
        获取节点配置文件
        :param host_list:
        :return:
        """
        # 需要安装OpsMonitorAgent的主机的IP。
        agent_hosts = []
        for host_ip in host_list:
            ansible_ssh_ip = host_ip
            ansible_ssh_port = 22
            agent_listen_address = host_ip
            agent_client_address = host_ip
            host = Host(
                ansible_ssh_ip, ansible_ssh_port, agent_listen_address,
                agent_client_address)
            agent_hosts.append(host)
        logger.info(f"Init MOICAgent hosts return: {agent_hosts}.")
        return agent_hosts

    def get_mo_certs(self):
        need_sync_mo_cert = False
        if not os.path.exists(Path.TEMP_CERT_PATH):
            os.mkdir(Path.TEMP_CERT_PATH)
            need_sync_mo_cert = True
        else:
            oc_cer = f"{Path.TEMP_CERT_PATH}/trust_apimlb_oc.cer"
            sc_cer = f"{Path.TEMP_CERT_PATH}/trust_apimlb_sc.cer"
            iam_cer = f"{Path.TEMP_CERT_PATH}/trust_apimlb_iam.cer"
            log_cer = f"{Path.TEMP_CERT_PATH}/cert.pem"
            mo_cert_exit = os.path.exists(oc_cer) and os.path.exists(sc_cer) and os.path.exists(
                iam_cer) and os.path.exists(log_cer)
            if not mo_cert_exit:
                logger.info(f"MO certs not exists in {Path.TEMP_CERT_PATH}.")
                need_sync_mo_cert = True
            else:
                logger.info(
                    f"MO certs exists in {Path.TEMP_CERT_PATH}, "
                    f"no need to get mo certs.")
        if need_sync_mo_cert:
            mo_cert_tool = MOCertManager(self.pod_id)
            result = mo_cert_tool.get_mo_apimlb_certs(Path.TEMP_CERT_PATH)
            if not result:
                logger.error(f"Get MO certs process return {result}.")
                raise Exception("Get MO certs failed.")
            log_result = mo_cert_tool.get_log_cert(Path.TEMP_CERT_PATH)
            if not log_result:
                logger.error(f"Get MO LOG certs process return {result}.")
                raise Exception("Get MO LOG certs failed.")
            logger.info("Get MO certs to temp path success.")

    def register_sso(self, sso_info):
        logger.info(f"Process register sso start: {sso_info}.")
        om_info = json.dumps(sso_info)
        result = self.mo_util2.registerSystemAndOperationURL(self.pod_id, om_info)
        if not result:
            logger.error("Register eReplication sso failed.")
            raise HCCIException(
                "663616",
                "Process registerSystemAndOperationURL return false.")
        logger.info("Process registerSystemAndOperationURL success.")

    def add_sso_white_list(self, white_list):
        result = self.mo_util2.addUnissoTrustedIP(self.pod_id, white_list)
        if not result:
            raise HCCIException(
                "663616", "Process addUnissoTrustedIP return false.")
        logger.info(f"Process addUnissoTrustedIP success: {white_list}.")

    def global_domain(self, is_central_cloud=True):
        if self.condition.is_cloud_unify and is_central_cloud:
            return self.get_central_cloud_global_domain()
        return self.mo_util2.get_global_domain_name(self.pod_id)

    def get_vdc_admin_token(self):
        return self.mo_util.get_vdc_admin_token(self.pod_id)

    def get_central_cloud_global_domain(self):
        return self._get_cloud_domain("global_domain_name")

    def get_central_cloud_iam_domain(self):
        return f"iam-cache-proxy.{self.get_central_cloud_global_domain()}"

    def get_central_cloud_console_domain(self):
        global_domain_name = self._get_cloud_domain("external_global_domain_name")
        console_domain_name_prefix = self._get_cloud_domain("console_domain_name_prefix")
        return f"{console_domain_name_prefix}.{global_domain_name}"

    def _get_cloud_domain(self, key):
        region_info = self.cmdb_ins.get_region_info()
        for region in region_info:
            if region.get("regionType") not in ["master", "dr_standby"]:
                continue
            region_id = region.get("regionCode")
            mo_service_infos = self.cmdb_ins.get_cloud_service_info(region_id, "ManageOne")
            if not mo_service_infos:
                mo_service_infos = self.cmdb_ins.get_cloud_service_info(region_id, "ManageOne_B2B")
            if not mo_service_infos:
                logger.error("Get MO Service info failed.")
                raise Exception("Get MO Service info failed.")
            mo_service_info = mo_service_infos[0]
            mo_extend_info = mo_service_info.get("extendInfos")
            global_domain_name = list(
                [ele.get("value") for ele in mo_extend_info if ele.get("key") == key])[0]
            return global_domain_name

    def get_oc_external_domain(self):
        om_info = self.mo_util2.getMoOmInfo(self.pod_id)
        if len(om_info) >= 5:
            oc_info = f"{om_info[4]}:{om_info[3]}"
        else:
            oc_info = f"{om_info[0]}:{om_info[3]}"
        return oc_info

    def get_oc_domain_info(self, region_id):
        if self.condition.is_primary:
            oc_info, _, _ = \
                self.mo_util2.get_mo_domain_name_list(self.project_id, self.pod_id)
            return oc_info, 26335
        oc_info = f"oc.{region_id}.{self.global_domain()}"
        return oc_info, 26335

    def get_tenant_info(self):
        if self.condition.is_cloud_unify:
            return f"sc.{self.get_central_cloud_global_domain()}", 26335
        tenant_info = self.mo_util2.getMoTanentInfo(self.pod_id)
        tenant_domain = tenant_info[0]
        if len(tenant_info) >= 5:
            tenant_domain = tenant_info[4]
        return tenant_domain, 26335

    def get_sc_domain_info(self):
        if self.condition.is_cloud_unify:
            return f"sc.{self.get_central_cloud_global_domain()}", 26335
        _, sc_info, _ = \
            self.mo_util2.get_mo_domain_name_list(self.project_id, self.pod_id)
        return sc_info, 26335

    def get_third_party_info(self):
        return self.mo_util2.getThirdpartyInfor(self.pod_id)

    def get_region_info(self, region_id=None):
        return self.cmdb_ins.get_region_info(region_code=region_id)

    def get_all_server_nodes_from_cmdb(self, region_id=None):
        """
        对接CMDB后，获取主备Region的所有Server IP地址
        :return:
        """
        logger.info("Get all server nodes from cmdb start.")
        all_server_node_lst = []
        if self.condition.is_primary and not \
                self.condition.is_current_dr_installed:
            return all_server_node_lst
        all_region_id_lst = \
            [region_id
             ] if region_id else self.get_all_region_id_from_cmdb()
        for region_code in all_region_id_lst:
            deploy_server_nodes = self.cmdb_ins.get_deploy_node_info(
                region_code, Component.REPLICATION)
            for server_node in deploy_server_nodes:
                server_ip_info = server_node.get("ipAddresses", None)
                server_ip = server_ip_info[0].get("ip", None)
                if server_ip not in all_server_node_lst:
                    all_server_node_lst.append(server_ip)
        logger.info(
            f"Get all server nodes from cmdb return {all_server_node_lst}.")
        return all_server_node_lst

    def get_all_region_id_from_cmdb(self):
        all_region_id_lst = list()
        all_region_info = self.cmdb_ins.get_region_info()
        for region_info in all_region_info:
            region_id = region_info.get("regionCode", None)
            if region_id and region_id not in all_region_id_lst:
                all_region_id_lst.append(region_id)
        logger.info(f"Get all region id return {all_region_id_lst}.")
        return all_region_id_lst

    def get_iam_lvs_ip_and_port(self):
        return self.mo_util2.get_iam_lvs_ipaddress_and_port(self.pod_id)

    def get_mo_er_ip_and_port(self):
        tenant_address_info = self.mo_util2.getMoTanentInfo(self.pod_id)
        return tenant_address_info[0], tenant_address_info[3]

    def get_lvs_ip(self):
        if self.condition.is_cloud_unify:
            return f"mo-lvs.{self.get_central_cloud_global_domain()}"
        return self.mo_util2.getlvsIp(self.pod_id)

    def get_tls_port(self):
        return self.mo_util2.getTLSPort(self.pod_id)

    def get_endpoints(self, region_id):
        """
        获取指定语言的Endpoint列表
        """
        endpoints_list_cn = self.mo_util.get_endpoints_by_region(
            self.pod_id, region_id)
        return endpoints_list_cn

    def refresh_dr_params(self):
        """
        扩容完成之后，刷新容灾数据
        :return:
        """
        # 扩容安装容灾服务场景，需要调用MO接口刷新数据，防止因MO查询不到数据导致region切换失败
        if self.condition.is_expansion_service:
            logger.info("Begin to refresh DR params.")
            self.mo_util2.refreshDRParams(self.pod_id)
            logger.info("Refresh DR params success.")

    def set_cloud_service_info(self, region_id, cloud_service_info):
        self.cmdb_ins.set_cloud_service_info(region_id, cloud_service_info)

    def get_unify_password_info(self, account_list):
        account_ins = AccountQuery(self.project_id)
        account_info = account_ins.get_account_info(
            "", account_list, "", Component.REPLICATION, Component.REPLICATION)
        logger.info("Get unify password success.")
        return account_info

    def mo_register_endpoint(self, region_id, endpoint):
        return self.mo_util.register_endpoint_information(
            self.pod_id, region_id, endpoint)

    def mo_register_region(self, endpoint_id, region_id, address, port):
        return self.mo_util.register_region_to_endpoint_information(
            self.pod_id, endpoint_id, region_id, address, port)

    def mo_register_product(self, region_id, product_infos):
        return self.mo_util.register_product_information(
            self.pod_id, region_id, product_infos)

    def mo_register_action(self, action):
        return self.mo_util.register_product_action(self.pod_id, action)

    def mo_register_notifications(self, product_notifications):
        return self.mo_util.register_notifications(
            self.pod_id, product_notifications)

    def mo_register_quota(self, service_type, quota_indicators):
        return self.mo_util.register_quota_indicators(
            self.pod_id, service_type, quota_indicators)

    def mo_register_meter(self, meter_metrics):
        return self.mo_util.register_meter_metrics(self.pod_id, meter_metrics)

    @classmethod
    def _get_constant_class(cls, service_type):
        if service_type == Component.CSDR_L:
            s_class = CSDRSilvanInfos
        elif service_type == Component.CSHA_L:
            s_class = CSHASilvanInfos
        elif service_type == Component.VHA_L:
            s_class = VHASilvanInfos
        else:
            raise HCCIException("663715", "Service type invalid.")
        return s_class

    @classmethod
    def register_endpoint(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Register endpoint for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        zh_endpoint = s_class(region_id, console_home).zh_endpoint_infos
        en_endpoint = s_class(region_id, console_home).en_endpoint_infos
        logger.info(
            f"Start register endpoint, endPoint info: {zh_endpoint}.")
        cls(project_id, pod_id).mo_register_endpoint(region_id, zh_endpoint)
        logger.info(
            f"Start register endpoint, endPoint info: {en_endpoint}.")
        cls(project_id, pod_id).mo_register_endpoint(region_id, en_endpoint)
        logger.info(f"Register endpoint for {service_type} success.")

    @classmethod
    def _do_update_service_endpoints(cls, uri, iam_token, new_endpoint, sc_domain):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        headers.update({'host': sc_domain})
        resp = requests.request(method="put", url=uri, headers=headers, json=new_endpoint, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="put", url=uri, headers=headers, json=new_endpoint, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do update endpoint failed.")
                raise Exception("Do update endpoint failed.")
        if "success" not in str(resp.content):
            logger.error(f"Update endpoints request return: {resp.content}.")
            raise Exception(f"Update endpoints request return: {resp.content}.")
        logger.info("Do update endpoint success.")

    @classmethod
    def _get_endpoint_info(cls, mo_lvs_ip, mo_port, service):
        endpoint_lst = list()
        uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints?start=0&limit=1000&id={service}"
        logger.info(f"Get {service}'s endpoints.")
        endpoint_zh = cls._do_get_service_endpoints(uri)
        endpoint_en = cls._do_get_service_endpoints(uri, language="en-us")
        endpoint_lst.extend(endpoint_zh)
        endpoint_lst.extend(endpoint_en)
        logger.info(f"Get endpoints return {endpoint_lst}.")
        return endpoint_lst

    @staticmethod
    def _do_get_service_endpoints(uri, language='zh-cn'):
        service = uri.split("&id=")[1]
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'X-Language': language})
        resp = requests.request(method="get", url=uri, headers=headers, verify=False)
        if not resp.content:
            resp = requests.request(method="get", url=uri, headers=headers, verify=False)
            if not resp.content:
                raise Exception(f"Get {service}'s endpoints failed: {language}.")
        endpoints_info = json.loads(resp.content)
        total = endpoints_info['total']
        logger.info(f"Get {service}'s endpoints return {endpoints_info}")
        if total <= 0:
            logger.error(f"Get {service}'s endpoints return total {total} records.")
            raise Exception(f"Get {service}'s endpoints return total {total} records.")
        return endpoints_info["endpoints"]

    @classmethod
    def register_links(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Register links for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        links_infos = s_class(region_id, console_home).links_infos
        for links_info in links_infos:
            key, text, icon, link, link_type = links_info
            cls(project_id, pod_id).mo_util.register_links_information(
                        pod_id, key, text, icon, link, link_type,
                        domainType="SC", region_id="")
        logger.info(f"Register links for {service_type} success.")

    @classmethod
    def register_region(
            cls, project_id, pod_id, region_id, server_port, service_type):
        logger.info(
            "Register region to endpoint information for "
            f"{service_type} start.")
        global_domain_name = cls(project_id, pod_id).global_domain()
        logger.info("Register region to endpoint information start.")
        # Global独立部署场景下，只需要注册cdrs相关的silvan region信息
        condition = Condition(project_id)
        if not condition.is_hcs_global:
            cls(project_id, pod_id).mo_register_region(
                service_type, region_id, f"{service_type}.{global_domain_name}", server_port)
        if condition.is_cloud_unify:
            cls(project_id, pod_id).mo_register_region(
                service_type, "*", f"{service_type}.{global_domain_name}", server_port)
            cls(project_id, pod_id).mo_register_region(
                "cdrs", "*", f"cdrs.{global_domain_name}", server_port)
        cls(project_id, pod_id).mo_register_region(
            "cdrs", region_id, f"cdrs.{global_domain_name}", server_port)
        logger.info(
            "Register region to endpoint information successfully.")

    @classmethod
    def register_product(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Start register product for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        product_infos = s_class(region_id, console_home).product_infos
        cls(project_id, pod_id).mo_register_product(region_id, product_infos)
        logger.info(f"Register product for {service_type} success.")

    @classmethod
    def register_action(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Register product action for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        product_actions = s_class(region_id, console_home).product_action_infos
        for action in product_actions:
            result = cls(project_id, pod_id).mo_register_action(action)
            if not result:
                logger.error(
                    f"Register product action failed, action info:{action}.")
                raise Exception("Register product action failed.")
        logger.info("Register product action success.")

    @classmethod
    def register_quota(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Register quota indicators for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        quota_indicators = \
            s_class(region_id, console_home).quota_indicators_infos
        result = cls(project_id, pod_id).mo_register_quota(
            service_type, quota_indicators)
        if not result:
            logger.error(f"Register {service_type} quota information failed.")
            raise Exception("Register quota information failed.")
        logger.info(f"Register {service_type} quota information success.")

    @classmethod
    def register_meter(
            cls, project_id, pod_id, region_id, console_home, service_type):
        logger.info(f"Register meter metrics for {service_type} start.")
        s_class = cls._get_constant_class(service_type)
        meter_metrics = s_class(region_id, console_home).meter_metrics_infos
        result = cls(project_id, pod_id).mo_register_meter(meter_metrics)
        if not result:
            logger.error(f"Register {service_type} meter metrics failed.")
            raise Exception("Register meter metrics failed.")
        logger.info(f"Register {service_type} meter metrics success.")

    def register_union_adaptation_pkg(self, region_id, file_path):
        package_infos = {
            "regionId": region_id,
            "file_path": file_path
        }
        logger.info(f"Start to register adaptation package, package_infos: {package_infos}.")
        result = self.mo_util2.upload_adaptation_package(self.project_id, self.pod_id, package_infos)
        if result.get("status") == "failed":
            error_msg = f"Failed use manage service api to register adaptation package, file path : {file_path}," \
                        f" err_msg:{result.get('msg')}."
            logger.error(error_msg)
            raise HCCIException(663620, error_msg)
        logger.info("Succeeded to register adaptation package.")

    def query_old_adaptation_pkg_version(self, region_id):
        logger.info('Start to query old adaptation pkg.')
        result = self.mo_util2.query_adaptation_package_record(self.project_id, Component.REPLICATION, region_id)
        origin_version = None
        for pkg in result:
            if pkg.get('status') == ADAPTATION_PKG_UPLOAD_SUCCESS_STATUS:
                logger.info(f"Query adaptation pkg success: {pkg}.")
                origin_version = pkg.get("version")
                if origin_version:
                    break

        if not origin_version:
            error_msg = 'Query old adaptation pkg failed.'
            logger.error(error_msg)
            raise HCCIException(663620, error_msg)
        logger.info('Query old adaptation pkg success.')
        return origin_version

    def unregister_adaptation_package(self, region_id, old_version):
        logger.info(f"Start to unregister {Component.REPLICATION} adaptation package on {region_id}.")
        result = self.mo_util2.rollback_adaptation_package(self.project_id, region_id,
                                                           Component.REPLICATION, old_version)
        if result.get("status") == "failed" and MO_EXCEPT_CODE not in str(result["msg"]):
            error_msg = f"Unregister {Component.REPLICATION} adaptation package failed : {result.get('msg')}."
            logger.error(error_msg)
            raise HCCIException(663621, error_msg)
        logger.info(
            f"Ended to unregister {Component.REPLICATION} adaptation package to OC successfully on {region_id}.")
