# -*- coding: utf-8 -*-
import json
from collections import defaultdict

import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.project_util import ProjectApi
from utils.constant.business_constant import CMDBConstant
from plugins.eReplication.common.lib.conditions import Condition

from plugins.eReplication.common.constant import Component

SUPPORT_BLOCK = "CSDR_supported_block"
SUPPORT_NAS = "CSDR_supported_nas"
GLOBAL_SUPPORT_BLOCK = "global_CSDR_supported_block"
GLOBAL_SUPPORT_NAS = "global_CSDR_supported_nas"
CLOUD_UNIFY = "is_cloud_unify"


class CustomizeCondition:
    def __init__(self, project_id, cmdb_ins):
        self.project_id = project_id
        self.cmdb_ins = cmdb_ins
        self.project_api = ProjectApi()

    def execute(self):
        # 添加场景值
        condition = Condition(self.project_id)
        project_info = BaseOps().get_project_info(self.project_id)
        is_project_deploy = project_info.get("scene") == "ProjectDeploy"

        if is_project_deploy:
            replication_scene_info = {SUPPORT_BLOCK: 0, SUPPORT_NAS: 0}
        else:
            cur_region_id = self.project_api.get_regionlist_by_project_Id(self.project_id)[0]
            replication_scene_info = self._deal_with_global_replication_customize_condition(region_id=cur_region_id)

        if condition.is_primary and is_project_deploy:
            deploy_mode_scene_info = {CLOUD_UNIFY: 0}
            global_replication_scene_info = {GLOBAL_SUPPORT_BLOCK: 0, GLOBAL_SUPPORT_NAS: 0}
        else:
            deploy_mode_scene_info = self._deal_with_deploy_mode_customize_condition()
            global_replication_scene_info = self._deal_with_global_replication_customize_condition(is_global=True)
        replication_scene_info.update(deploy_mode_scene_info)
        replication_scene_info.update(global_replication_scene_info)
        return json.dumps(replication_scene_info)

    @staticmethod
    def _get_replication_cmdb_info(cmdb_ins, cur_region_id):
        """获取cmdb中eReplication服务注册信息

        :param cmdb_ins: 接入cmdb的实例
        :param cur_region_id: 当前升级region_id
        :return:
        """
        if cmdb_ins:
            cmdb_data = cmdb_ins.get_region_infos(cur_region_id)
            service_info = cmdb_data.get(CMDBConstant.SERVICE_LAYER, [])
            if not service_info:
                return defaultdict()
            for unit in service_info:
                if unit.get(CMDBConstant.INDEXNAME) != Component.REPLICATION:
                    continue
                return unit
            return defaultdict()
        raise Exception("The incoming param is not an instance of accessing the CMDB.")

    def _deal_with_global_replication_customize_condition(self, region_id=None, is_global=False):
        project_info = BaseOps().get_project_info(self.project_id)
        if project_info.get("scene") == "ProjectDeploy":
            return {SUPPORT_BLOCK: 0, SUPPORT_NAS: 0}
        replication_scene_info = {GLOBAL_SUPPORT_BLOCK: 0, GLOBAL_SUPPORT_NAS: 0} if is_global else {
            SUPPORT_BLOCK: 0, SUPPORT_NAS: 0}
        block_key = GLOBAL_SUPPORT_BLOCK if is_global else SUPPORT_BLOCK
        nas_key = GLOBAL_SUPPORT_NAS if is_global else SUPPORT_NAS
        flag = "global" if is_global else "current"
        all_region_ids = [region_id] if region_id else self.cmdb_ins.get_region_list()
        for region_id in all_region_ids:
            replication_cmdb_info = self._get_replication_cmdb_info(self.cmdb_ins, region_id)
            logger.info(f"The {region_id}'s eReplication extend infos is: {replication_cmdb_info}.")
            if not replication_cmdb_info:
                continue
            for extends_info in replication_cmdb_info.get(CMDBConstant.EXTENDS_INFO, []):
                extends_info_key = extends_info.get(CMDBConstant.EXTENDS_INFO_KEY)
                extends_info_value = extends_info.get(CMDBConstant.EXTENDS_INFO_VALUE)
                if extends_info_key == SUPPORT_BLOCK and extends_info_value == "True":
                    replication_scene_info[block_key] = 1
                if extends_info_key == SUPPORT_NAS and extends_info_value == "True":
                    replication_scene_info[nas_key] = 1
        logger.info(f"The {flag} eReplication service scenario is {replication_scene_info}.")
        return replication_scene_info

    @staticmethod
    def _get_region_cmdb_info(cmdb_ins, cur_region_id):
        """获取cmdb中regions信息

        :param cmdb_ins: 接入cmdb的实例
        :return:
        """
        if not cmdb_ins:
            raise Exception("The incoming param is not an instance of accessing the CMDB.")
        cmdb_data = cmdb_ins.get_region_infos(cur_region_id)
        region_info = cmdb_data.get(CMDBConstant.REGION_LAYER, [])
        if not region_info:
            return defaultdict()
        return region_info[0]

    def _deal_with_deploy_mode_customize_condition(self):
        global_domain_name_set = set()
        deploy_mode_scene_info = {CLOUD_UNIFY: 0}
        all_region_ids = self.cmdb_ins.get_region_list()
        for region_id in all_region_ids:
            region_info = self._get_region_cmdb_info(self.cmdb_ins, region_id)
            global_domain_name = region_info.get("globalDomainNamePostfix")
            logger.info(f"The {region_id}'s global domain name is: {global_domain_name}.")
            if not global_domain_name:
                continue
            global_domain_name_set.add(global_domain_name)
        if len(global_domain_name_set) >= 2:
            deploy_mode_scene_info[CLOUD_UNIFY] = 1
        logger.info(f"The deploy mode scenario is {deploy_mode_scene_info}.")
        return deploy_mode_scene_info
