# -*- coding:utf-8 -*-
import os
import threading

import utils.common.log as logger
from utils.common.exception import HCCIException

from plugins.eReplication.common.lib.model import DmkDeploymentInfo
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import BACK_CERT_TYPE
from plugins.eReplication.common.constant import BACK_CONF_TYPE
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.base import BaseSubJob
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.decorator import handle_task_result
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import get_install_mode
from plugins.eReplication.common.lib.utils import get_local_role
from plugins.eReplication.common.os_api import API as OS_API

logger.init("InstallDRService")


class InstallDRService(BaseSubJob):
    """
    完成eReplication服务软件的安装部署
    由于容灾软件可能是单独部署的, 此时无法依赖到其它组件参数,因此,其它组件的参数也要配置在自身服务中
    获取参数时如果从其它组件获取失败,则容灾服务自身获取,此时需要用户配置这些参数
    """

    def __init__(self, project_id, pod_id, regionid_list=None):
        super(InstallDRService, self).__init__(
            project_id, pod_id, regionid_list)
        self.type = get_install_mode(self.project_id, self.pod_id)
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.params = Params(self.project_id, self.pod_id)
        pkg_prefix = Pkg.SERVER_PKG_PREFIX
        pkg_suffix = Pkg.SERVER_PKG_SUFFIX
        self.condition = Condition(self.project_id)
        self.pkg_path = PKG_API.get_file_abs_path(pkg_prefix, pkg_suffix)
        self.pkg_version = PKG_API.get_version(self.pkg_path)
        self.mo_api = MO_API(self.project_id, self.pod_id)
        self.dmk_api = DMK_API()

    def upload_package(self):
        """
        上传安装包到DMK环境
        :return:Message对象
        """
        # 从服务器上查找软件包
        logger.info(f"Get eReplication package return {self.pkg_path}.")
        PKG_API.check_compressed_file(
            self.pkg_path, Capacity.GB_S, Pkg.SERVER_PKG_SIZE_LIMIT,
            Pkg.SERVER_PKG_FILE_LIMIT)
        # 登录dmk
        self.dmk_api.upload_oversize_pkg_to_dmk(self.pkg_path, self.params.dmk_float_ip,
                                                self.params.dmk_os_business_user,
                                                self.params.dmk_os_business_user_pwd)
        logger.info("Upload eReplication package success.")

    def install_dr_service(self, account_id):
        """
        向DMK下发安装BCManager命令
        :return: Message对象
        """
        logger.info("Start install eReplication server.")
        host_info = FILE_API.get_config_content(os.path.join(
            os.path.dirname(os.path.dirname(
                os.path.dirname(os.path.dirname(__file__)))), "conf",
            "server_hosts"))
        if self.type == "1":
            host_info = host_info.format(self.nodes.hosts[0])
        else:
            host_info = host_info.format(",".join(self.nodes.hosts))
        external_oc_address = self.mo_api.get_oc_external_domain()
        region_id = self.params.project_region_id
        if self.condition.is_sub:
            region_id = DR_API(Auth(self.nodes.primary_ip, self.nodes.ssh_user,
                                    self.nodes.ssh_pwd, self.nodes.sudo_user,
                                    self.nodes.sudo_pwd)).get_current_region_id()
        oc_domain, oc_port = self.mo_api.get_oc_domain_info(region_id)
        om_info = f"{oc_domain}:{oc_port}"
        er_om_info = f"{oc_domain}:{oc_port}"
        cert_info = f"{oc_domain}:{oc_port}"
        # 构造安装配置信息
        config_info = FILE_API.get_config_content(os.path.join(
            os.path.dirname(os.path.dirname(
                os.path.dirname(os.path.dirname(__file__)))), "conf",
            "server_deploy.yml"))
        config_info = config_info.format(
            self.pkg_version, self.nodes.ssh_user, self.type,
            external_oc_address, self.nodes.local_ip, self.nodes.peer_ip,
            get_local_role(self.pod_id, self.project_id),
            self.nodes.float_ip, self.nodes.gw_ip,
            cert_info, self.params.project_region_id, om_info,
            er_om_info)
        logger.info(f"Host info: {host_info}.")
        logger.info(f"Config info: {config_info}.")
        # 向DMK下发部署命令
        result = self.dmk_api.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, self.pkg_version, "Install",
                              host_info, config_info, account_id))
        if not result:
            raise HCCIException(
                "663614", "Task excute_dmk_deployment return false.")

    def config_one_a_three_s(self, account_id):
        logger.info("Start config eReplication server.")
        # 构造节点信息
        host_info = FILE_API.get_config_content(os.path.join(
            os.path.dirname(os.path.dirname(
                os.path.dirname(os.path.dirname(__file__)))), "conf",
            "server_hosts"))
        host_info = host_info.format(",".join(
            [self.nodes.primary_ip, self.nodes.second_ip,
             self.nodes.local_ip, self.nodes.peer_ip]))
        primary_region_id = DR_API(Auth(self.nodes.primary_ip, self.nodes.ssh_user,
                                        self.nodes.ssh_pwd, self.nodes.sudo_user,
                                        self.nodes.sudo_pwd)).get_current_region_id()
        region_ids = \
            f"{primary_region_id},{self.params.project_region_id}"
        # 构造安装配置信息
        config_info = \
            f"---\nversion: {self.pkg_version}\n\n" \
            f"remote_ssh_user: {self.nodes.ssh_user}"
        config_info += "\n".join((
            '\n# 配置一主三备,在安装备region的时候配置以下参数',
            '# 主端Server IP地址',
            f'primary_server_ip1: \'{self.nodes.primary_ip}\'\n',
            f'primary_server_ip2: \'{self.nodes.second_ip}\'\n',
            '# 备端Server IP地址',
            f'standby_server_ip1: \'{self.nodes.local_ip}\'\n',
            f'standby_server_ip2: \'{self.nodes.peer_ip}\'\n',
            '# 主备Region对应的Region ID，'
            '格式: \'主Region Region ID,备Region Region ID\'',
            f'region_ids: \'{region_ids}\'\n'))
        logger.info(f"Host info: {host_info}.")
        logger.info(f"Config info: {config_info}.")
        result = self.dmk_api.execute_dmk_deployment(DmkDeploymentInfo(Component.REPLICATION, True, self.pkg_version,
                                                                       "configOneAThreeS", host_info, config_info,
                                                                       account_id))
        if not result:
            raise HCCIException(
                "663614", "Task excute_dmk_deployment return false.")

    def config_active_standby(self):
        """ 仅用于CSDR场景配置主备模式

        export localRole=1 IS_ACTIVE_STANDBY_AUTO=true local_IP= peer_IP=
        :return:
        """
        # CSDR场景且本端脚本为备时,才需要配置
        if self.type != "1":
            logger.info(
                f"No need config active-standby mode, mode: {self.type}.")
            return True
        logger.info("Start config active_standby mode.")
        # 配置主端
        DR_API(Auth(self.nodes.primary_ip, self.nodes.ssh_user,
                    self.nodes.ssh_pwd, self.nodes.sudo_user,
                    self.nodes.sudo_pwd)).config_active_standby(
            "1", self.nodes.primary_ip, self.nodes.local_ip)
        # 配置备端
        DR_API(Auth(self.nodes.local_ip, self.nodes.ssh_user,
                    self.nodes.ssh_pwd, self.nodes.sudo_user,
                    self.nodes.sudo_pwd)).config_active_standby(
            "2", self.nodes.local_ip, self.nodes.primary_ip)
        return True

    def config_watchdog(self):
        """
        内部接口,供配置软件狗调用
        :return:
        """
        logger.info("Start config watchdog now.")
        config_funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.hosts:
            dr_api = DR_API(Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                                 self.nodes.sudo_user, self.nodes.sudo_pwd))
            config_func = (dr_api.config_watch_dog, thread_name, (), {})
            config_funcs.append(config_func)
        ExcThread.exec_func_in_thread(config_funcs)

        # 检查虚拟机是否重启OK
        check_funcs = list()
        for host in self.nodes.hosts:
            check_func = (
                OS_API.check_os_connection, thread_name,
                (host, self.nodes.ssh_user, self.nodes.ssh_pwd), {}
            )
            check_funcs.append(check_func)
        ExcThread.exec_func_in_thread(check_funcs)
        logger.info("Config watchdog successfully.")

    @handle_task_result
    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装前预检查&安装&配置
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        # 第一步，配置软件狗并启动自动喂狗程序
        self.config_watchdog()
        # 第二步，拷贝主端的证书和配置到备端
        if self._is_installed_csdr():
            #  拷贝主端证书到备端
            DR_API.sync_bcm_files_to_standby(self.project_id, self.pod_id, BACK_CERT_TYPE)
            #  拷贝配置文件到备端
            DR_API.sync_bcm_files_to_standby(self.project_id, self.pod_id, BACK_CONF_TYPE)
            #  备端创建拷贝成功标志文件(对应eReplication安装脚本内的判断标志)
            DR_API.create_sync_flag_file_to_standby(self.project_id, self.pod_id)

        # 第三步, 获取帐户ID
        self.dmk_api.login_dmk(
            self.params.dmk_float_ip, self.params.dmk_deploy_user,
            self.params.dmk_user_new_pwd)
        account_id = self.dmk_api.get_dmk_account_id(
            self.params.dmk_deploy_user, self.nodes.ssh_user)
        # 第四步,上传安装包
        self.upload_package()
        # 第五步,下发部署操作
        # 检查环境上是否已经装过eReplication, 如果装过， 则直接返回，不再安装
        for host in self.nodes.hosts:
            if OS_API.check_result_in_os(
                    host, self.nodes.sudo_user, self.nodes.sudo_pwd,
                    "ls /home/ICUser/RDInstalled.xml"):
                logger.info(
                    f"RDInstalled.xml find in {host}, no need install.")
                continue
            else:
                self.install_dr_service(account_id)
        # 第六步，下发配置一主三备操作 -- 安装备region CSDR时进行配置
        # 安装备region CSDR的时候向DMK下发一主三备配置命令
        if self.type == "2" and self._is_installed_csdr():
            self.config_one_a_three_s(account_id)
        # 第七步,主备模式下,单独在此配置eReplication主备
        self.config_active_standby()
        # 第八步，同步主节点数据至备节点
        if self._is_installed_csdr():
            DR_API.sync_dr_system_data_to_standby(self.project_id, self.pod_id)
        # 第九步，配置对接kms相关系统配置信息
        DR_API.set_hcs_domain_sys_config(self.project_id, self.pod_id)
        # 第十步，配置多云合一场景相关配置
        if self.condition.is_cloud_unify:
            DR_API.set_multi_cloud_adapter_switch_status(self.project_id, self.pod_id)

    def _is_installed_csdr(self):
        return self.condition.is_sub and self.condition.is_global_con_dr and (
                    self.condition.is_csdr or self.condition.is_hcs_global)
