# -*- coding:utf-8 -*-
import configparser
import os
import stat

import utils.common.log as logger


class API(object):

    def __init__(self, conf_file):
        self.conf_file = conf_file
        if not (conf_file and os.path.isfile(conf_file)):
            raise Exception(f"The file does not exist, file path:{conf_file}.")
        self.config = configparser.ConfigParser()
        self.config.read(conf_file)

    @classmethod
    def open_file_manager(cls, file_name, mode='r', encoding=None, **kwargs):
        if 'r' in mode and encoding:
            return open(file_name, mode=mode, encoding=encoding, **kwargs)
        elif 'r' in mode and not encoding:
            with open(file_name, 'rb') as file:
                context = file.read()
            for encoding_item in ['UTF-8', 'GBK', 'ISO-8859-1']:
                try:
                    context.decode(encoding=encoding_item)
                except UnicodeDecodeError:
                    pass
                encoding = encoding_item
                break
            return open(file_name, mode=mode, encoding=encoding, **kwargs)
        else:
            # 多用户系统下写文件，只对脚本运行用户有可读可写权限
            # os.O_WRONLY --- 以只写的方式打开
            # os.O_CREAT --- 创建并打开一个新文件
            # os.O_TRUNC --- 如果文件存在，覆盖文件内容
            # os.O_APPEND --- 如果文件存在，追加内容
            # stat.S_IWUSR --- 拥有者具有写权限0o200
            # stat.S_IRUSR --- 拥有者具有读权限0o400
            flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
            if "+" in mode or "a" in mode:
                flags = os.O_WRONLY | os.O_CREAT | os.O_APPEND
            modes = stat.S_IWUSR | stat.S_IRUSR
            return os.fdopen(os.open(file_name, flags, modes), mode)

    @classmethod
    def get_config_content(cls, file_path):
        file_object = cls.open_file_manager(file_path)
        content = ""
        try:
            for line in file_object:
                content = "{0}{1}".format(content, line)
        except Exception as err:
            logger.error(f"Get file content failed: {err}")
        finally:
            file_object.close()
        return content

    def get_params_dict_by_key_name(self, key_name, sub_keys=None):
        """Obtains the value of key in the .ini file.

        :param key_name: Key of the .ini file, parameter type: str
        :param sub_keys:sub_key of key_name, parameter type: list or tuple
        :return:dict()
        """
        if sub_keys and not isinstance(sub_keys, (list, tuple)):
            raise Exception(
                f"The parameter type of sub_keys must be list or tuple."
                f"sub_keys is {str(sub_keys)}, type is {type(sub_keys)}")
        param_dict = dict()
        if sub_keys:
            for key, value in self.config.items(key_name):
                if key in sub_keys:
                    param_dict[key] = value
        else:
            for key, value in self.config.items(key_name):
                param_dict[key] = value
        return param_dict

    def get_value_by_key_and_sub_key(self, key, sub_key):
        try:
            return self.config.get(key, sub_key)
        except Exception as err:
            logger.warn(err)
            return ""

    def set_value_by_key_and_sub_key(self, key, sub_key, value):
        if not self.config.has_section(key):
            self.config.add_section(key)
        self.config.set(key, sub_key, str(value))
        if not self.config.has_option(key, sub_key):
            return False
        if str(value) != self.config.get(key, sub_key):
            return False
        with API.open_file_manager(self.conf_file, 'w') as file:
            self.config.write(file)
        return True
