# -*- coding:utf-8 -*-
import base64
import json
import os
import re
import threading
from collections import defaultdict
from copy import deepcopy
from itertools import product
from time import sleep
from tenacity import retry, stop_after_attempt, wait_fixed

import requests
import utils.common.log as logger
import yaml
from utils.business.dns_utils2 import DNSApi2
from utils.business.manageone_data_migrate import ManageOneDataMigrateUtil
from utils.business.manageone_data_migrate.manageone_iam_util import ManageOneIamUtil
from utils.business.manageone_data_migrate.manageone_oc_util import ManageOneOcUtil
from utils.business.manageone_data_migrate.service_monitor.service_monitor import ServiceMonitorCfg
from utils.business.manageone_data_migrate.service_monitor.service_monitor import ServiceNodeCfg
from utils.business.manageone_data_migrate.service_monitor.service_monitor import del_service_m_cfg
from utils.business.manageone_data_migrate.service_monitor.service_monitor import set_service_m_cfg
from utils.business.param_util import CheckResult
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.business.manageone_data_migrate import MOMigratedNodeInfos

from plugins.eReplication.common.api.file_api import API as FILE_API
from plugins.eReplication.common.api.vm_api import API as VM_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Action, Capacity
from plugins.eReplication.common.constant import Component
from plugins.eReplication.common.constant import HTTP_STATUS_OK
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.conditions import Condition
from plugins.eReplication.common.lib.model import Auth, DmkDeploymentInfo, DmkAccountInfo
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import check_host_connection
from plugins.eReplication.common.lib.utils import check_value_null
from plugins.eReplication.common.request_api import RequestApi
from plugins.eReplication.common.request_api import SERVICE_OK
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.constant import Pkg


class API(object):

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.params = Params(self.project_id, self.pod_id)
        self.condition = Condition(self.project_id)
        self.oc_util = ManageOneOcUtil()
        self.iam_util = ManageOneIamUtil()
        self.results = list()
        self.config_path = os.path.join(
            os.path.dirname(os.path.dirname(__file__)), 'conf', 'env.ini')
        DR_API(Auth(self.nodes.target_ip, self.nodes.ssh_user,
                    self.nodes.target_ssh_pwd, self.nodes.sudo_user,
                    self.nodes.target_sudo_pwd)).do_confirm_tool_env_config()
        self.config_tool = FILE_API(self.config_path)

    def check_dr_service_env(self):
        """Check dr service running environment"""

        thread_name = threading.current_thread().name
        funcs = [
            (self._check_designated_site_all_nodes_env, thread_name, (),
             {"is_target": False}),
            (self._check_designated_site_all_nodes_env, thread_name, (),
             {"is_target": True})]
        exc_result = ExcThread.exec_func_in_thread_and_return(funcs)
        for result in exc_result:
            self.results.extend(result)
        return self.results

    def _check_designated_site_all_nodes_env(self, is_target=False):
        """Checking the Service Running Environment in a Specified DR system

        :param is_target: Indicates whether to check source or target
        :return:
        """

        results = list()
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        for host in all_hosts:
            if not check_host_connection(host):
                # 网络通的情况下再进行后续检查
                msg = {"IP": host}
                logger.error(f"Check host connected failed, {msg}.")
                ex = HCCIException('665001', msg)
                results.append(CheckResult(
                    itemname_ch="检查网络连接", itemname_en="Check network connection",
                    status="failure", error_msg_cn=ex))
        if results:
            return results
        funcs = list()
        thread_name = threading.current_thread().name
        for host in all_hosts:
            sudo_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, ssh_pwd, self.nodes.sudo_user,
                                                           sudo_pwd))
            func = (self._do_env_check_task, thread_name,
                    (sudo_client, host), {})
            funcs.append(func)
        exc_result = ExcThread.exec_func_in_thread_and_return(funcs)
        for result in exc_result:
            results.extend(result)
        return results

    @staticmethod
    def _do_env_check_task(sudo_client, host):
        results = list()
        disk = "/home/DRManager"
        disk_size = 4
        runtime_path = f"{Path.MIGRATE_DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        results = DR_API.check_whether_backup_file_exists(
            sudo_client, host, runtime_path, results)
        results = DR_API.check_whether_disk_free_space_is_enough(
            sudo_client, host, disk, disk_size, results)
        # 检查磁盘空间是否可写: /opt
        results = DR_API.check_whether_disk_can_be_writen(
            sudo_client, host, disk, results)
        return results

    def check_communication_matrix(self):
        target_results = self._check_designated_site_all_nodes_connection(
            is_target=True)
        self.results.extend(target_results)
        if self.condition.is_primary:
            source_results = self._check_designated_site_all_nodes_connection()
            self.results.extend(source_results)
            check_result = self._check_communication_matrix(
                self.nodes.source_all_hosts, self.nodes.target_all_hosts)
            self.results.extend(check_result)
        return self.results

    def _check_designated_site_all_nodes_connection(self, is_target=False):
        results = list()
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        funcs = list()
        thread_name = threading.current_thread().name
        for host in all_hosts:
            func = (self._do_connect_check_task, thread_name,
                    (Auth(host, self.nodes.ssh_user, ssh_pwd, self.nodes.sudo_user,
                          sudo_pwd),), {})
            funcs.append(func)
        exc_result = ExcThread.exec_func_in_thread_and_return(funcs)
        for result in exc_result:
            results.extend(result)
        return results

    @staticmethod
    def _do_connect_check_task(auth_provider):
        results = list()
        ssh_client = None
        logger.info(f"Check whether node[{auth_provider.host}] can be logged in.")
        try:
            ssh_client = SSH_API.get_sudo_ssh_client(auth_provider)
        except HCCIException as err:
            check = CheckResult(
                itemname_ch=f"检查可登录性[{auth_provider.host}]",
                itemname_en=f"Check Login[{auth_provider.host}]", status="failure",
                error_msg_cn=err)
            results.append(check)
            return results
        except Exception as err:
            ex = HCCIException('665002', str(err))
            check = CheckResult(
                itemname_ch=f"检查可登录性[{auth_provider.host}]",
                itemname_en=f"Check Login[{auth_provider.host}]", status="failure",
                error_msg_cn=ex)
            results.append(check)
            return results
        finally:
            SSH_API.close_ssh(ssh_client)
        results.append(
            CheckResult(
                itemname_ch=f"检查可登录性[{auth_provider.host}]",
                itemname_en=f"Check Login[{auth_provider.host}]", status="success"))
        return results

    def _check_communication_matrix(self, source_site_ips, target_site_ips):
        """Checking the Data Migration Communication Matrix

        :param source_site_ips: Physical IP address of the source site
        :param target_site_ips: Physical IP address of the target site
        :return:
        """

        funcs = list()
        results = list()
        thread_name = threading.current_thread().name
        combination_list = product(source_site_ips, target_site_ips)
        for source_ip, target_ip in combination_list:
            auth_provider = Auth(
                source_ip, self.nodes.ssh_user, self.nodes.source_ssh_pwd,
                self.nodes.sudo_user, self.nodes.source_sudo_pwd)
            func = (
                self._do_communication_matrix_check_task, thread_name,
                (auth_provider, target_ip), {}
            )
            funcs.append(func)
        exc_result = ExcThread.exec_func_in_thread_and_return(funcs)
        for result in exc_result:
            results.extend(result)
        if results:
            return results
        service_check_result = self._do_service_check_task()
        results.extend(service_check_result)
        return results

    @staticmethod
    def _do_communication_matrix_check_task(auth_provider, target_ip):
        results = list()
        source_ip = auth_provider.host
        try:
            ssh_client = SSH_API.get_sudo_ssh_client(auth_provider)
            result = SSH_API.exec_command_return_list(
                ssh_client, f"wget --spider -T 10 -t 3 {target_ip}:22")
        except HCCIException as err:
            logger.error(f"Communication matrix check failed: {err}.")
            results.append(CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check", status="failure",
                error_msg_cn=err))
            return results
        if "connected" in ",".join(result):
            logger.info(
                f"Port 22 between {source_ip} and {target_ip} is enabled.")
        elif "failed" in ",".join(result):
            logger.error(f"Port 22 between {source_ip} and "
                         f"{target_ip} is disabled.")
            check = CheckResult(
                itemname_ch="通信矩阵校验", itemname_en="Communication matrix check",
                status="failure", error_msg_cn=HCCIException(675028, source_ip, target_ip, "22"))
            results.append(check)
        return results

    def _do_service_check_task(self):
        results = list()
        # 校验目标站点服务正常
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user,
            self.nodes.target_ssh_pwd, self.nodes.sudo_user,
            self.nodes.target_sudo_pwd)
        request_api = RequestApi(
            listen_ip, self.nodes.service_name, self.nodes.service_pwd,
            9443, raise_ex=False)
        status_code, message = request_api.check_dr_service()
        if status_code != SERVICE_OK:
            logger.error("The DR service is unavailable.")
            check = CheckResult(
                itemname_ch="通信矩阵校验",
                itemname_en="Communication matrix check",
                status="failure", error_msg_cn=HCCIException(
                    675101, request_api.node_ip, request_api.port,
                    message))
            results.append(check)
        return results

    def check_version(self):
        source_version = self._get_designated_site_service_version(
            self.nodes.source_all_hosts, self.nodes.ssh_user,
            self.nodes.source_ssh_pwd, self.nodes.sudo_user,
            self.nodes.source_sudo_pwd)
        target_version = self._get_designated_site_service_version(
            self.nodes.target_all_hosts, self.nodes.ssh_user,
            self.nodes.target_ssh_pwd, self.nodes.sudo_user,
            self.nodes.target_sudo_pwd)
        if source_version != target_version and (
                len(source_version) != 1 or len(target_version) != 1):
            logger.error(
                f"The version of the source site DR software[{source_version}] does not match the "
                f"target site DR software[{target_version}].")
            check = CheckResult(
                itemname_ch="容灾服务版本号检查", itemname_en="Checking the DR software Version",
                status="failure", error_msg_cn=HCCIException(675100, source_version, target_version))
            self.results.append(check)
        if not self.results:
            self.results.append(CheckResult(
                itemname_ch="容灾服务版本号检查",
                itemname_en="Checking the DR software Version",
                status="success"))
        return self.results

    @staticmethod
    def _get_designated_site_service_version(all_hosts, ssh_user, ssh_pwd, sudo_user, sudo_pwd):
        result = set()
        for host in all_hosts:
            version = DR_API(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd)).get_current_version()
            result.add(version)
        return result

    def stop_service(self, rollback=False):
        # 如果是回退场景下不需要停止源端服务
        if not rollback:
            self._do_stop_service_task(is_target=False)
        self._do_stop_service_task(is_target=True)

    def _do_stop_service_task(self, is_target=False):
        dmk_float_ip = self.params.target_dmk_float_ip if is_target else self.params.source_dmk_float_ip
        server_ip = self.nodes.target_ip if is_target else self.nodes.source_ip
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        dmk_deploy_user = self.params.dmk_deploy_user
        dmk_user_new_pwd = self.params.dmk_user_new_pwd
        ssh_user = self.nodes.ssh_user
        version = DR_API(
            Auth(server_ip, ssh_user, ssh_pwd, self.nodes.sudo_user, sudo_pwd)).get_current_version()
        DMK_API.login_dmk(dmk_float_ip, dmk_deploy_user, dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(dmk_deploy_user, ssh_user)
        if check_value_null(account_id):
            raise Exception("Get dmk account id return false.")
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)), "conf", "server_hosts"))
        host_info = host_info.format(",".join(all_hosts))
        config_info = f"---\nversion: {version}\n\nremote_ssh_user: {ssh_user}"
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        result = DMK_API.execute_dmk_deployment(DmkDeploymentInfo(
            Component.REPLICATION, True, version, Action.STOP_SERVER,
            host_info, config_info, account_id))
        if not result:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def backup_dr_service_data(self):
        thread_name = threading.current_thread().name
        funcs = [(self._do_backup_data_task, thread_name, (), {"is_target": False}),
                 (self._do_backup_data_task, thread_name, (), {"is_target": True})]
        ExcThread.exec_func_in_thread(funcs)

    def _do_backup_data_task(self, is_target=False):
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        for ssh_ip in all_hosts:
            logger.info(f"Start backup data on {ssh_ip}.")
            sudo_client = SSH_API.get_sudo_ssh_client(Auth(ssh_ip, self.nodes.ssh_user, ssh_pwd, self.nodes.sudo_user,
                                                           sudo_pwd))
            self._backup_sys_data(sudo_client, self.params.data_encrypt_pwd)

    @staticmethod
    def _backup_sys_data(sudo_client, data_encrypt_pwd):
        SSH_API.send_command(
            sudo_client,
            f"echo y|sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh", "successfully", 180)
        # 导出容灾系统数据
        SSH_API.send_command(
            sudo_client, f"sh {Path.BCM_ROOT_SCRIPTS_PATH}backupData.sh",
            'Please enter the password')
        dest_str = "Export system configuration data successfully"
        cmd_out = SSH_API.send_command(
            sudo_client, data_encrypt_pwd, dest_str, 180)
        data_back_file = ""
        for result in cmd_out:
            if dest_str in result and '/' in result:
                result = result.replace('\n', '')
                data_back_file = result[result.index('/'):]
        if check_value_null(data_back_file):
            logger.error(f"Get database config file failed: {cmd_out}.")
            raise Exception(f"Get database config file failed: {cmd_out}.")
        logger.info(f"Get database config file return: {data_back_file}.")
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {Path.MIGRATE_DATA_BACKUP_PATH} && mkdir -p "
            f"{Path.MIGRATE_DATA_BACKUP_PATH};echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 备份系统配置数据
        dest_file = \
            f"{Path.MIGRATE_DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        SSH_API.send_command(
            sudo_client,
            f"rm -fr {dest_file} && cp -frp {data_back_file} {dest_file};"
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        # 拷贝bcm.keystore证书到用户目录和备份目录
        cert_back_path1 = f"{Path.DR_USER_PATH}/{Path.BCM_CERT_BACK_NAME}"
        cert_back_path2 = \
            f"{Path.MIGRATE_DATA_BACKUP_PATH}/{Path.BCM_CERT_BACK_NAME}"
        copy_cmd = f"echo {cert_back_path1} {cert_back_path2} | " \
                   f"xargs -n 1 cp -v {Path.KEYSTORE_FILE_PATH};" \
                   "echo CMD_RESULT=$?"
        SSH_API.send_command(sudo_client, copy_cmd, "CMD_RESULT=0")
        SSH_API.send_command(
            sudo_client,
            f"chown -h DRManager {cert_back_path1};echo CMD_RESULT=$?", "CMD_RESULT=0")
        SSH_API.send_command(
            sudo_client,
            f"chown -Rh ICUser:LEGO {Path.MIGRATE_DATA_BACKUP_PATH} && chmod 740 {Path.MIGRATE_DATA_BACKUP_PATH} &&"
            "echo CMD_RESULT=$?", "CMD_RESULT=0")

    def migrate_dr_service_data(self, rollback=False):
        if not rollback:
            self._transfer_files_to_target_site_all_nodes()
        self._import_dr_system_data(rollback)
        self._import_dr_system_certs(rollback)
        if self.condition.is_multi_cloud_integration and not rollback:
            self._rewrite_multi_cloud_adapter_switch_status()

    def _transfer_files_to_target_site_all_nodes(self):
        thread_name = threading.current_thread().name
        funcs = list()
        for target_ip in self.nodes.target_all_hosts:
            func = (self._do_migrate_data_task, thread_name,
                    (target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def _do_migrate_data_task(self, target_ip, ssh_user, ssh_pwd):
        source_sudo_client = SSH_API.get_sudo_ssh_client(Auth(self.nodes.source_ip, self.nodes.ssh_user,
                                                              self.nodes.source_ssh_pwd, self.nodes.sudo_user,
                                                              self.nodes.source_sudo_pwd))
        cert_back_path = f"{Path.DR_USER_PATH}/{Path.BCM_CERT_BACK_NAME}"
        data_back_path = f"{Path.DR_USER_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        dest_path = Path.DR_USER_PATH
        logger.info(f"Begin to transfer files to {target_ip}.")
        scp_cmd = \
            f"scp -o 'StrictHostKeyChecking no' {cert_back_path} " \
            f"{data_back_path} {ssh_user}@{target_ip}:" \
            f"{dest_path} && echo CMD_RESULT=$?"
        SSH_API.send_command(source_sudo_client, scp_cmd, "password")
        SSH_API.send_command(source_sudo_client, ssh_pwd, "CMD_RESULT=0")
        logger.info(f"Migrate DR system data to {target_ip} success.")

    def _import_dr_system_data(self, rollback=False):
        thread_name = threading.current_thread().name
        funcs = list()
        for target_ip in self.nodes.target_all_hosts:
            auth_provider = Auth(target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                                 self.nodes.sudo_user, self.nodes.target_sudo_pwd)
            func = (self._do_import_data_task, thread_name,
                    (auth_provider, self.params.data_encrypt_pwd, rollback), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    @staticmethod
    def _do_import_data_task(auth_provider, data_encrypt_pwd, rollback):
        target_ip = auth_provider.host
        sudo_client = SSH_API.get_sudo_ssh_client(auth_provider)
        SSH_API.send_command(
            sudo_client,
            f"echo y|sh {Path.BCM_SCRIPTS_PATH}shutdownSystem.sh && "
            "echo CMD_RESULT=$?", "CMD_RESULT=0")
        data_back_path = f"{Path.DR_USER_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        if rollback:
            data_back_path = f"{Path.MIGRATE_DATA_BACKUP_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        SSH_API.send_command(
            sudo_client, f"cd {Path.BCM_SCRIPTS_PATH} && sh import.sh",
            'do you want to continue')
        SSH_API.send_command(sudo_client, "y", 'Please input the restore file')
        SSH_API.send_command(sudo_client, data_back_path, "encrypt password")
        SSH_API.send_command(sudo_client, data_encrypt_pwd, "Restore successfully", 300)
        logger.info(f"Do import data on {target_ip} success.")

    def _import_dr_system_certs(self, rollback):
        thread_name = threading.current_thread().name
        funcs = list()
        for target_ip in self.nodes.target_all_hosts:
            logger.info(f"Do import GR certs to {target_ip} in thread.")
            sudo_client = SSH_API.get_sudo_ssh_client(Auth(target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                                                           self.nodes.sudo_user,
                                                           self.nodes.target_sudo_pwd))
            func = (self._do_import_certs_task, thread_name,
                    (sudo_client, self.params.bcm_keystore_pwd, rollback), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    def _do_import_certs_task(self, sudo_client, bcm_keystore_pwd, rollback):
        if rollback:
            cert_back_path = f"{Path.MIGRATE_DATA_BACKUP_PATH}/{Path.BCM_CERT_BACK_NAME}"
            sync_cert_cmd = rf"\cp {cert_back_path} {Path.KEYSTORE_FILE_PATH}"
            SSH_API.send_command(sudo_client, sync_cert_cmd, "#", 20)
            logger.info("Overwrite bcm.keystore success.")
            return

        exists_certs = self._do_export_certs(sudo_client, bcm_keystore_pwd)
        self._do_import_certs(sudo_client, exists_certs)

    @staticmethod
    def _do_export_certs(sudo_client, bcm_keystore_pwd):
        SSH_API.send_command(sudo_client, f"rm -rf {Path.EXPORT_CERT_PATH} && mkdir {Path.EXPORT_CERT_PATH}")
        SSH_API.send_command(
            sudo_client, f"export PATH={Path.JRE_PATH}:$PATH")
        cert_back_path = f"{Path.DR_USER_PATH}/{Path.BCM_CERT_BACK_NAME}"
        query_cmd = \
            f"keytool -list -keystore {cert_back_path} | " \
            "grep trustedCertEntry | awk -F', ' '{print $1}'"
        SSH_API.send_command(sudo_client, query_cmd, "password")
        result = SSH_API.send_command(sudo_client, bcm_keystore_pwd)
        exists_certs = \
            [ele.strip("\n") for ele in result if "cert" in ele.lower()
             and "The JKS keystore" not in ele]
        logger.info(f"Get GR exists certs return {exists_certs}.")
        for cert in exists_certs:
            logger.debug(f"Do export cert {cert} file.")
            export_cmd = f"keytool -export -alias {cert} " \
                         f"-file '{Path.EXPORT_CERT_PATH}/{cert}.crt' " \
                         f"-keystore {cert_back_path}"
            SSH_API.send_command(sudo_client, export_cmd, "password")
            SSH_API.send_command(sudo_client, bcm_keystore_pwd)
            logger.debug(f"Export cert {cert} file success.")
        # 修改导出证书目录属主
        SSH_API.send_command(sudo_client, f"chown -Rh ICUser:LEGO {Path.EXPORT_CERT_PATH}")
        return exists_certs

    @staticmethod
    def _do_import_certs(sudo_client, exists_certs):
        for cert in exists_certs:
            logger.debug(f"Do import cert {cert} file.")
            # 导入证书
            run_cmd = f"cd {Path.BCM_SCRIPTS_PATH} && sudo -u ICUser -s sh import_cert.sh; echo CMD_RESULT=$?"
            SSH_API.send_command(sudo_client, run_cmd, "Please input the cert alias: ")
            SSH_API.send_command(sudo_client, f"GR.{cert}", "Please input the cert file path: ")
            result = SSH_API.send_command(sudo_client, f"{Path.EXPORT_CERT_PATH}/{cert}.crt")
            if "CMD_RESULT=0" not in str(result):
                SSH_API.send_command(sudo_client, f"rm -rf {Path.EXPORT_CERT_PATH}")
                err_msg = f"Import cert {cert} failed."
                logger.error(err_msg)
                raise Exception(err_msg)
            logger.info(f"Import cert {cert} success.")
        SSH_API.send_command(sudo_client, f"rm -rf {Path.EXPORT_CERT_PATH}")
        logger.info("Import certs to GR BCM node success.")

    def _rewrite_multi_cloud_adapter_switch_status(self):
        thread_name = threading.current_thread().name
        funcs = list()
        for target_ip in self.nodes.target_all_hosts:
            logger.info(f"Do rewrite multi cloud adapter switch to {target_ip} in thread.")
            sudo_client = SSH_API.get_sudo_ssh_client(
                target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                self.nodes.sudo_user, self.nodes.target_sudo_pwd)
            func = (self._do_rewrite_multi_cloud_adapter_switch, thread_name, (sudo_client,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)

    @staticmethod
    def _do_rewrite_multi_cloud_adapter_switch(sudo_client):
        logger.info("Do execute rewrite multi cloud adapter switch task.")
        clear_cmd = "sed -i -e '/#是否是多云合一场景.*/d' -e '/multi.cloud.adapter.switch.*/d'" \
                    f" {Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?"
        SSH_API.send_command(sudo_client, clear_cmd, "CMD_RESULT=0")
        rewrite_cmd = r"sed -i '/#是否支持console.*/i\#是否是多云合一场景\n" \
                      rf"multi.cloud.adapter.switch=true' {Path.LEGO_FILE_PATH} && echo CMD_RESULT=$?"
        SSH_API.send_command(sudo_client, rewrite_cmd, "CMD_RESULT=0")
        logger.info("Rewrite multi cloud adapter switch success.")

    def start_service(self, rollback=False):
        # 如果是回退场景，需要将源站点和目标站点都还原到初始状态，两端都需要启动服务，否则只启动目标站点服务
        if rollback:
            self._do_start_service_task(is_target=False)
        self._do_start_service_task(is_target=True)

    def _do_start_service_task(self, is_target=False):
        dmk_float_ip = self.params.target_dmk_float_ip if is_target else \
            self.params.source_dmk_float_ip
        server_ip = self.nodes.target_ip if is_target else self.nodes.source_ip
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        dmk_deploy_user = self.params.dmk_deploy_user
        dmk_user_new_pwd = self.params.dmk_user_new_pwd
        ssh_user = self.nodes.ssh_user
        version = DR_API(Auth(server_ip, ssh_user, ssh_pwd, self.nodes.sudo_user, sudo_pwd)).get_current_version()
        DMK_API.login_dmk(dmk_float_ip, dmk_deploy_user, dmk_user_new_pwd)
        account_id = DMK_API.get_dmk_account_id(dmk_deploy_user, ssh_user)
        if check_value_null(account_id):
            raise Exception("Get dmk account id return false.")
        host_info = FILE_API.get_config_content(
            os.path.join(os.path.dirname(os.path.dirname(__file__)), "conf", "server_hosts"))
        host_info = host_info.format(",".join(all_hosts))
        config_info = f"---\nversion: {version}\n\nremote_ssh_user: {ssh_user}"
        logger.info(f"Host_info:[{host_info}].")
        logger.info(f"Config_info:[{config_info}].")
        result = DMK_API.execute_dmk_deployment(
            DmkDeploymentInfo(Component.REPLICATION, True, version, Action.START_SERVER,
                              host_info, config_info, account_id))
        if not result:
            logger.error("Execute Server install failed.")
            raise Exception("Execute dmk deployment return false.")

    def clear_env(self):
        thread_name = threading.current_thread().name
        # 删除源端备份数据
        source_clear_funcs = list()
        for source_ip in self.nodes.source_all_hosts:
            source_clear_func = (
                self._do_clear_env_task, thread_name,
                (Auth(source_ip, self.nodes.ssh_user, self.nodes.source_ssh_pwd,
                      self.nodes.sudo_user, self.nodes.source_sudo_pwd),), {})
            source_clear_funcs.append(source_clear_func)
        ExcThread.exec_func_in_thread(source_clear_funcs)
        # 删除目标端备份数据
        target_clear_funcs = list()
        for target_ip in self.nodes.target_all_hosts:
            target_clear_func = (
                self._do_clear_env_task, thread_name,
                (Auth(target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                      self.nodes.sudo_user, self.nodes.target_sudo_pwd),), {})
            target_clear_funcs.append(target_clear_func)
        ExcThread.exec_func_in_thread(target_clear_funcs)
        # 删除目标端主节点env.ini备份文件
        self._do_clear_env_config_on_remote()

    @staticmethod
    def _do_clear_env_task(auth_provider):
        cert_back_path = f"{Path.DR_USER_PATH}/{Path.BCM_CERT_BACK_NAME}"
        data_back_path = f"{Path.DR_USER_PATH}/{Path.BACKUP_DATA_FILE_NAME}"
        migrate_data_back_path = Path.MIGRATE_DATA_BACKUP_PATH
        clear_cmd = \
            f"rm -rf {migrate_data_back_path};echo {cert_back_path} {data_back_path} | " \
            f"xargs rm -rf;echo CMD_RESULT=$?"
        sudo_client = SSH_API.get_sudo_ssh_client(auth_provider)
        SSH_API.send_command(sudo_client, clear_cmd, "CMD_RESULT=0")
        logger.info(f"Clear env on {auth_provider.host} success.")

    def _do_clear_env_config_on_remote(self):
        ssh_client = SSH_API.get_sudo_ssh_client(Auth(self.nodes.target_ip, self.nodes.ssh_user,
                                                      self.nodes.target_ssh_pwd, self.nodes.sudo_user,
                                                      self.nodes.target_sudo_pwd))
        remote_file = f"{Path.DR_USER_PATH}/env.ini"
        clear_cmd = f"rm -rf {remote_file} && echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, clear_cmd)
        if "CMD_RESULT=0" not in result:
            logger.error(f"Clear remote file {remote_file} failed.")
            raise Exception(f"Clear remote file {remote_file} failed.")
        logger.info(f"Clear remote file {remote_file} success.")

    def delete_manage_vms(self):
        """Delete management VMs"""

        logger.info("Begin to delete all dr service management VMs.")
        all_vm_infos = self._get_manage_vms_info()
        for name, group_name, vm_list, network_plane in all_vm_infos:
            VM_API.delete_manage_vm(
                self.pod_id, name, group_name, vm_list, network_plane)
        logger.info("Delete all dr service management VMs success.")

    def _get_manage_vms_info(self):
        data = list()
        server_vm_list = list()
        console_vm_list = list()
        all_vm_infos = ParamUtil().get_vm_all_infos(self.project_id)
        for vm_info in all_vm_infos:
            if "Service-eReplication" not in vm_info.get("vm_name") and \
                    "Console-eReplication" not in vm_info.get("vm_name"):
                continue
            data, server_vm_list, console_vm_list = \
                self._analysis_manage_vm_info(vm_info, data, server_vm_list, console_vm_list)
        if not (self.condition.is_region_con_ha and self.condition.is_primary):
            if server_vm_list:
                data.append(server_vm_list)
            if console_vm_list:
                data.append(console_vm_list)
        logger.info(f"Get manage vms info return {data}.")
        return data

    def _analysis_manage_vm_info(
            self, vm_info, data, server_vm_list, console_vm_list):
        vm_ip = vm_info.get("ip")
        vm_name = vm_info.get("vm_name")
        vm_network_plane = vm_info.get("network_plane")
        server_name, vm_group_name = vm_name.split("-")
        if self.condition.is_region_con_ha and self.condition.is_primary:
            data.append(
                (server_name, vm_group_name, [vm_ip], vm_network_plane))
        else:
            vm_group_name = re.sub(r"\d", "", vm_group_name)
            if server_name == "Service":
                if server_vm_list:
                    server_vm_list[2].extend([vm_ip])
                else:
                    server_vm_list = [server_name, vm_group_name, [vm_ip], vm_network_plane]
            elif server_name == "Console":
                if console_vm_list:
                    console_vm_list[2].extend([vm_ip])
                else:
                    console_vm_list = [server_name, vm_group_name, [vm_ip], vm_network_plane]
        return data, server_vm_list, console_vm_list

    def update_replication_sso(self):
        # 更新sso白名单
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user,
            self.nodes.target_ssh_pwd, self.nodes.sudo_user,
            self.nodes.target_sudo_pwd)
        self._update_sso_white_list()
        # 更新sso单点登录
        uri = f"https://{listen_ip}:{self.nodes.service_port}"
        om_info = {
            "id": "eReplication", "name": {"zh-cn": "eReplication", "en-us": "eReplication"},
            # global部署
            "region_id": '', "url": uri, "operations": []
        }
        result = self.oc_util.register_system_and_operation_url(self.pod_id, om_info)
        if not result:
            logger.error("Register eReplication url to ManageOne failed.")
            raise HCCIException("675102", "Task registerSystemAndOperationURL return false.")
        logger.info("Register eReplication url to ManageOne success.")

    def _update_sso_white_list(self):
        ip_lst = deepcopy(self.nodes.target_all_hosts)
        ssh_user = self.nodes.ssh_user
        ssh_pwd = self.nodes.target_ssh_pwd
        sudo_user = self.nodes.sudo_user
        sudo_pwd = self.nodes.target_sudo_pwd
        primary_float_ip = DR_API(Auth(self.nodes.target_primary_hosts[0], ssh_user, ssh_pwd, sudo_user,
                                       sudo_pwd)).get_server_float_ip()
        ip_lst.append(primary_float_ip)
        target_standby_hosts = self.nodes.target_standby_hosts
        if target_standby_hosts:
            standby_float_ip = DR_API(Auth(target_standby_hosts[0], ssh_user, ssh_pwd, sudo_user,
                                           sudo_pwd)).get_server_float_ip()
            ip_lst.append(standby_float_ip)
        ip_lst = ",".join(ip_lst)
        result = self.oc_util.add_unisso_trusted_ip(self.pod_id, ip_lst)
        if not result:
            raise HCCIException("675102", "Task addUnissoTrustedIP return false.")
        logger.info(f"Update sso white list {ip_lst} success.")

    def get_migrate_site_all_region_ids(self, is_target=False):
        all_region_ids = set()
        ssh_user = self.nodes.ssh_user
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_user = self.nodes.sudo_user
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        for host in all_hosts:
            region_id = DR_API(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd)).get_current_region_id()
            all_region_ids.add(region_id)
        return all_region_ids

    def _delete_service_monitor_info(self):
        all_region_ids = self.get_migrate_site_all_region_ids()
        for region_id in all_region_ids:
            self._delete_server_monitor_info(region_id)
            if self.condition.is_cloud_platform6:
                self._delete_console_monitor_info(region_id)

    def _delete_server_monitor_info(self, region_id):
        service_monitor_cfg = ServiceMonitorCfg(
            Component.REPLICATION, "eReplication Server", region_id,
            Component.REPLICATION, "SAAS", None)
        result = del_service_m_cfg(self.pod_id, service_monitor_cfg)
        if not result:
            logger.error("Delete server service monitor return False.")
            raise Exception("Delete server service monitor return False.")
        logger.info(f"Delete {region_id} server service monitor info success.")

    def _delete_console_monitor_info(self, region_id):
        service_monitor_cfg = ServiceMonitorCfg(
            Component.CSDR_U, "eReplication console service monitor",
            region_id, Component.CSDR_U, "SAAS", None)
        result = del_service_m_cfg(self.pod_id, service_monitor_cfg)
        if not result:
            logger.error(
                "Delete server service monitor return False.")
            raise Exception(
                "Delete server service monitor return False.")
        logger.info("Delete console service monitor info success.")

    def _get_service_monitor_host_info(self):
        """Obtain service monitoring node information"""

        data = defaultdict()
        for host in self.nodes.target_all_hosts:
            node_region_id = DR_API(Auth(host, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                                         self.nodes.sudo_user, self.nodes.target_sudo_pwd)).get_current_region_id()
            if node_region_id in data:
                data[node_region_id].append(host)
            else:
                data[node_region_id] = [host]
        logger.info(f"Get service monitor host info return {data}.")
        return data

    def update_service_monitor_info(self):
        self._delete_service_monitor_info()
        data = self._get_service_monitor_host_info()
        for region_id, hosts in data.items():
            node_info = []
            for host in hosts:
                for template_id in ['os_template', 'eReplication_service_template_id']:
                    node_cfg = ServiceNodeCfg(host, template_id, "eReplication_log_id")
                    node_info.append(node_cfg)
            monitor_cfg = ServiceMonitorCfg(
                Component.REPLICATION, "eReplication Server", region_id,
                Component.REPLICATION, "SAAS", node_info)
            result = set_service_m_cfg(self.pod_id, monitor_cfg)
            if not result:
                raise HCCIException(
                    "675103", "Set service monitor cfg return false")
        logger.info("Install ic agent successfully in all node.")

    def _get_logic_region_info(self, is_target=False):
        """Obtain the region display name"""

        data = defaultdict()
        ssh_user = self.nodes.ssh_user
        ssh_pwd = self.nodes.target_ssh_pwd if is_target else self.nodes.source_ssh_pwd
        sudo_user = self.nodes.sudo_user
        sudo_pwd = self.nodes.target_sudo_pwd if is_target else self.nodes.source_sudo_pwd
        all_hosts = self.nodes.target_all_hosts if is_target else self.nodes.source_all_hosts
        for host in all_hosts:
            dr_api = DR_API(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
            node_region_id = dr_api.get_current_region_id()
            node_float_ip = dr_api.get_server_float_ip()
            logic_region = self._get_logic_region_by_region_id(node_region_id)
            if logic_region not in data:
                data[logic_region] = node_float_ip
        logger.info(f"Get logic region info return {data}.")
        return data

    def _get_logic_region_by_region_id(self, region_id):
        """Obtain the region display name by region id

        :param region_id: ID of the region to be queried
        :return:
        """

        region_name = None
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user, project_id=self.project_id, pod_id=self.pod_id)
        region_info_list = mo_cmdb_ins.get_region_info()
        for region_info in region_info_list:
            if region_info.get("regionCode") == region_id:
                region_name = region_info.get("name")
                break
        return region_name

    def update_xaas_white_list(self):
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        data = self._get_logic_region_info()
        target_data = self._get_logic_region_info(is_target=True)
        for logic_region in data.keys():
            old_system_float_ip = data[logic_region]
            instance_lst = self.oc_util.get_instance_list(
                self.project_id, "XAAS", om_ip, old_system_float_ip)
            if instance_lst:
                for instance in instance_lst:
                    instance_info = instance.get("common", {})
                    instance_id = instance_info.get("instanceId")
                    self.oc_util.delete_instance(
                        self.project_id, instance_id, om_ip)

        for logic_region in target_data.keys():
            system_float_ip = target_data[logic_region]
            self.oc_util.add_xaas_white_list(
                self.project_id, f"{Component.REPLICATION}_",
                [system_float_ip], None, None, logic_region, "1", point="9443")
        sleep(600)

    def update_external_driver(self):
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        data = self._get_logic_region_info()
        for logic_region in data.keys():
            old_system_float_ip = data[logic_region]
            instance_lst = self.oc_util.get_instance_list(
                self.project_id, "External Driver", om_ip, old_system_float_ip)
            if not instance_lst:
                continue
            logger.debug("Delete source external driver instance: "
                         f"{old_system_float_ip}.")
            for instance in instance_lst:
                instance_info = instance.get("common", {})
                instance_id = instance_info.get("instanceId")
                self.oc_util.delete_instance(
                    self.project_id, instance_id, om_ip)

    @staticmethod
    def _delete_source_cmdb_data(mo_cmdb_ins, region_id, version):
        """Delete discarded CMDB data"""

        node_infos = mo_cmdb_ins.get_deploy_node_info(
            region_id, "eReplication")
        for node_info in node_infos:
            node_name = node_info.get("name")
            if not node_name.startswith("Service-eReplication") and \
                    not node_name.startswith("Console-eReplication"):
                continue
            mo_cmdb_ins.remove_deploy_node_info(region_id, node_name)
        mo_cmdb_ins.remove_cloud_service_info(
            region_id, Component.REPLICATION, version)
        logger.info("Delete discarded CMDB data success.")

    def update_cmdb_data(self):
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user,
            project_id=self.project_id, pod_id=self.pod_id)
        all_region_ids = self.get_migrate_site_all_region_ids()
        for region_id in all_region_ids:
            cloud_service_info = self._get_cloud_service_info(
                mo_cmdb_ins, region_id)
            version = self._get_version(mo_cmdb_ins, region_id)
            extend_infos = cloud_service_info.get("extendInfos")
            new_extend_infos = list()
            for extend_ in extend_infos:
                if extend_.get("key") == "global_service_scale":
                    new_extend_infos.append(
                        {"key": "global_service_scale", "value": ""})
                    continue
                new_extend_infos.append(extend_)
            new_cloud_service_info = {
                "indexName": Component.REPLICATION, "version": version,
                "extendInfos": new_extend_infos}
            self._delete_source_cmdb_data(mo_cmdb_ins, region_id, version)
            mo_cmdb_ins.set_cloud_service_info(
                region_id, new_cloud_service_info)

    def _get_version(self, mo_cmdb_ins, region_id):
        cloud_service_info = self._get_cloud_service_info(
            mo_cmdb_ins, region_id)
        service_version = cloud_service_info.get("version")
        if not service_version:
            raise Exception("Failed to obtain the service version.")
        return service_version

    @staticmethod
    def _get_cloud_service_info(mo_cmdb_ins, region_id):
        cloud_service_info = \
            mo_cmdb_ins.get_cloud_service_info(
                region_id, "eReplication")
        if not cloud_service_info:
            raise Exception(
                "Failed to get eReplication service information from CMDB.")
        logger.info(
            f"The service information obtained from the CMDB "
            f"is as follows:{cloud_service_info}.")
        return cloud_service_info[0]

    def update_iam_data(self):
        services = self._get_all_installed_service_from_new_cmdb()
        for service in services:
            account_msg, delete_account_msg = self._get_iam_account_data(
                service, self.params.iam_account_preset_pwd)
            role_msg = self._get_iam_role_data(service)
            self._update_iam_account(account_msg, delete_account_msg)
            self._update_iam_role(role_msg)
            logger.info(f"Preset iam account and role for {service} success.")

    @staticmethod
    def _get_iam_account_data(service, iam_account_preset_pwd):
        account_msg = {
            "user": {
                "name": f"{service}_service", "mobile": "", "email": "",
                "password": iam_account_preset_pwd,
                "domain_name": "op_service",
                "bind_group": "services,cred,auth,bss"
            }
        }
        delete_account_msg = {
            "user": {
                "name": f"{service}_service"
            }
        }
        return account_msg, delete_account_msg

    @staticmethod
    def _get_iam_role_data(service):
        role_msg = {
            "name": f"{service}_adm",
            "display_name": f"{service.upper()} Administrator",
            "description": f"{service.upper()} Administrator",
            "catalog": f"{service.upper()}",
            "policy": {
                "Version": "1.0",
                "Statement": [
                    {
                        "Effect": "Allow",
                        "Action": [
                            f"{service.upper()}:{service.upper()}:*"
                        ]
                    }
                ],
                "Depends": [
                    {
                        "catalog": "BASE",
                        "display_name": "Server Administrator"
                    },
                    {
                        "catalog": "BASE", "display_name": "Tenant Guest"
                    }
                ]
            }
        }
        return role_msg

    def _update_iam_account(self, account_msg, delete_account_msg):
        self.iam_util.preset_account(
            delete_account_msg, self.project_id, preset_type="deleteUser")
        self.iam_util.preset_account(account_msg, self.project_id)

    def _update_iam_role(self, role_msg):
        self.iam_util.preset_role(role_msg, self.project_id)

    def _get_dr_service_dns_data(self, dmk_float_ip, dns_user, dns_pwd):
        DMK_API.login_dmk(dmk_float_ip, dns_user, dns_pwd)
        old_config, host, env_ver = DMK_API.get_dmk_new_deploy_config(
            component="DNS-Internal", action=Action.REGISTER_DNS)
        dns_yaml_cfg = yaml.safe_load(old_config)
        for key in dns_yaml_cfg:
            if key != "om_config":
                continue
            data = self._analysis_dns_om_config(dns_yaml_cfg, key)
            logger.info(f"Get DNS config return {data}.")
            return data
        return defaultdict()

    @staticmethod
    def _analysis_dns_om_config(dns_yaml_cfg, domain_key):
        data = defaultdict()
        om_config = dns_yaml_cfg.get(domain_key)
        domains = om_config.get("domains")
        for domain in domains:
            service_lst = list()
            rrsets = domain.get("rrsets")
            domain_name = domain.get("zone")
            for rrset in rrsets:
                service_name = rrset.get("name")
                if service_name not in ["csdr", "csha", "vha", "cdrs"]:
                    continue
                service_lst.append(service_name)
            if not service_lst:
                continue
            data[domain_name] = service_lst
        return data

    def update_dns_data(self):
        dmk_info_lst = self._get_dns_dmk_infos()
        for dmk_info in dmk_info_lst:
            dmk_float_ip = dmk_info.get("dmk_float_ip")
            dmk_dns_user = dmk_info.get("dmk_user")
            dmk_dns_user_pwd = dmk_info.get("dmk_user_pwd")
            self._update_specified_region_dmk_dns(dmk_float_ip, dmk_dns_user, dmk_dns_user_pwd)
            if self.condition.is_multi_cloud_integration:
                self._register_specified_region_dmk_dns(dmk_float_ip)

    def _get_dns_dmk_infos(self):
        """Get dns dmk information"""

        dns_dmk_infos = list()
        source_dmk_float_ip = self.params.source_dmk_float_ip
        source_dmk_dns_user = self.params.source_dmk_dns_user
        source_dmk_dns_user_pwd = self.params.source_dmk_dns_user_pwd
        source_dmk_dns_info = {
            "dmk_float_ip": source_dmk_float_ip, "dmk_user": source_dmk_dns_user,
            "dmk_user_pwd": source_dmk_dns_user_pwd}
        dns_dmk_infos.append(source_dmk_dns_info)
        if self.condition.is_primary or (
                not self.condition.is_primary and
                self.condition.is_global_con_dr):
            target_dmk_float_ip = self.params.target_dmk_float_ip
            target_dmk_dns_user = self.params.target_dmk_dns_user
            target_dmk_dns_user_pwd = self.params.target_dmk_dns_user_pwd
            target_dmk_dns_info = {
                "dmk_float_ip": target_dmk_float_ip, "dmk_user": target_dmk_dns_user,
                "dmk_user_pwd": target_dmk_dns_user_pwd}
            dns_dmk_infos.append(target_dmk_dns_info)
        return dns_dmk_infos

    def _update_specified_region_dmk_dns(self, dmk_float_ip, dns_user, dns_pwd):
        """Correcting DNS information using DMK

        :param dmk_float_ip: float ip of dmk
        :param dns_user: dmk portal user of dns
        :param dns_pwd: dmk portal user password of dns
        :return:
        """

        dns_ins = DNSApi2()
        data = self._get_dr_service_dns_data(dmk_float_ip, dns_user, dns_pwd)
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
            self.nodes.sudo_user, self.nodes.target_sudo_pwd)
        for domain_name, services in data.items():
            rrsets = list()
            dns_type = "AAAA" if ":" in listen_ip else "A"
            for service in services:
                rrsets.append({'name': service, 'type': dns_type, 'records': [listen_ip]})
            dns_ins.add_rrsets2(
                domain_name, rrsets, "om", self.pod_id, self.region_id, dmk_float_ip)
            logger.info(f"Register eReplication domain for {services} successfully.")

    def _register_specified_region_dmk_dns(self, dmk_float_ip):
        """In the multi-cloud integration scenario, register the central cloud domain name with each DMK.

        :param dmk_float_ip: float ip of dmk
        :return:
        """

        dns_ins = DNSApi2()
        all_services = self._get_all_installed_service_from_new_cmdb()
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
            self.nodes.sudo_user, self.nodes.target_sudo_pwd)
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user, project_id=self.project_id, pod_id=self.pod_id)
        global_domain = self._get_migrated_global_domain(mo_cmdb_ins)
        rrsets = list()
        dns_type = "AAAA" if ":" in listen_ip else "A"
        for service in all_services + ['cdrs']:
            rrsets.append({'name': service, 'type': dns_type, 'records': [listen_ip]})
        dns_ins.add_rrsets2(
            global_domain, rrsets, "om", self.pod_id, self.region_id, dmk_float_ip)
        logger.info(f"Register central cloud eReplication domain for {all_services} successfully.")

    def backup_system_config(self):
        logger.info("Start backup DR system config data.")
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
            self.nodes.sudo_user, self.nodes.target_sudo_pwd)
        request_api = RequestApi(
            listen_ip, self.nodes.service_name, self.nodes.service_pwd, self.nodes.service_port)
        site_id = self._get_site_id(request_api)
        iam_domain = self._get_iam_data(request_api, site_id)
        alarm_domain, alarm_port = self._get_alarm_interconnect_data(request_api)
        tenant_domain, tenant_port = self._get_tenant_api_data(request_api)
        syslog_domain, syslog_port = self._get_syslog_data(request_api)
        cts_domain, cts_port = self._get_cts_data(request_api)
        self.config_tool.set_value_by_key_and_sub_key("iam_config", "domain", iam_domain)
        self.config_tool.set_value_by_key_and_sub_key("alarm_interconnect", "domain", alarm_domain)
        self.config_tool.set_value_by_key_and_sub_key("alarm_interconnect", "port", alarm_port)
        self.config_tool.set_value_by_key_and_sub_key("tenant_api", "domain", tenant_domain)
        self.config_tool.set_value_by_key_and_sub_key("tenant_api", "port", tenant_port)
        self.config_tool.set_value_by_key_and_sub_key("syslog_config", "domain", syslog_domain)
        self.config_tool.set_value_by_key_and_sub_key("syslog_config", "port", syslog_port)
        self.config_tool.set_value_by_key_and_sub_key("cts_config", "domain", cts_domain)
        self.config_tool.set_value_by_key_and_sub_key("cts_config", "port", cts_port)
        DR_API.sync_env_config_to_remote(
            self.nodes.target_ip, self.nodes.ssh_user, self.nodes.target_ssh_pwd, self.config_path)
        logger.info("Backup DR system config data success.")

    @staticmethod
    def _get_site_id(request_api):
        state, req_body = request_api.send_get_request("ws/drmservers")
        if state != 200 or len(req_body) == 0:
            logger.error(
                f"Get drm server info failed, status={state}, response={str(req_body)}.")
            raise Exception(
                f"Get drm server info failed, status={state}, response={str(req_body)}.")
        server_id = dict(req_body[0]).get('uuid')
        logger.info(f"DRM server uuid is [{server_id}]")
        state, req_body = request_api.send_get_request(f"ws/sites?serverUuid={server_id}")
        if state != 200 or len(req_body) == 0:
            logger.error(
                f"Get dr site info failed, status={state}, response={str(req_body)}.")
            raise Exception(
                f"Get dr site info failed, status={state}, response={str(req_body)}.")
        return dict(req_body[0]).get('siteId')

    @staticmethod
    def _get_iam_data(request_api, site_id):
        uri = f"ws/fusionsphere?siteId={site_id}&cloudType="
        state, resp_body = request_api.send_get_request(uri)
        if state != 200:
            logger.error(f"Get syslog data failed: {state}, {resp_body}.")
            raise Exception(f"Get syslog data failed: {state}, {resp_body}.")
        for resp in resp_body:
            domain = resp.get("ipAddress")
            return domain

    @staticmethod
    def _get_alarm_interconnect_data(request_api):
        uri = "ws/system/thirdservice/oc"
        state, resp_body = request_api.send_get_request(uri)
        if state != 200:
            logger.error(
                f"Get alarm interconnect data failed: {state}, {resp_body}.")
            raise Exception(
                f"Get alarm interconnect data failed: {state}, {resp_body}.")
        domain = resp_body.get("domain")
        port = resp_body.get("port")
        return domain, port

    @staticmethod
    def _get_tenant_api_data(request_api):
        uri = "ws/system/thirdservice/manageone"
        state, resp_body = request_api.send_get_request(uri)
        if state != 200:
            logger.error(
                f"Get tenant api data failed: {state}, {resp_body}.")
            raise Exception(
                f"Get tenant api data failed: {state}, {resp_body}.")
        domain = resp_body.get("domain")
        port = resp_body.get("port")
        return domain, port

    @staticmethod
    def _get_syslog_data(request_api):
        uri = "ws/system/action/querylognotify?logType=administratorlog"
        state, resp_body = request_api.send_get_request(uri)
        if state != 200:
            logger.error(f"Get syslog data failed: {state}, {resp_body}.")
            raise Exception(f"Get syslog data failed: {state}, {resp_body}.")
        resp = resp_body[0]
        protocal_props = json.loads(resp.get("protocalProps"))
        domain = protocal_props.get("ip")
        port = protocal_props.get("port")
        return domain, port

    @staticmethod
    def _get_cts_data(request_api):
        uri = "ws/system/action/querylognotify?logType=tenantlog"
        state, resp_body = request_api.send_get_request(uri)
        if state != 200:
            logger.error(f"Get tls data failed: {state}, {resp_body}.")
            raise Exception(f"Get tls data failed: {state}, {resp_body}.")
        for resp in resp_body:
            protocal_props = json.loads(resp.get("protocalProps"))
            domain = protocal_props.get("ip")
            port = protocal_props.get("port")
            return domain, port

    def update_dr_service_configure_data(self):
        """Modify DR service configuration data"""

        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        physical_ip, listen_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
            self.nodes.sudo_user, self.nodes.target_sudo_pwd)
        request_api = RequestApi(
            listen_ip, self.nodes.service_name, self.nodes.service_pwd, self.nodes.service_port)
        device_sn = self._get_device_sn(request_api)
        iam_host = self.config_tool.get_value_by_key_and_sub_key("iam_config", "domain")
        self._modify_openstack(
            request_api, device_sn, iam_host, "csdr_service", self.params.iam_account_preset_pwd)
        logger.info("Modify OpenStack info success.")
        self._refresh_device(request_api, device_sn)
        oc_domain = self.config_tool.get_value_by_key_and_sub_key("alarm_interconnect", "domain")
        oc_port = self.config_tool.get_value_by_key_and_sub_key("alarm_interconnect", "port")
        self._config_oc(request_api, oc_domain, oc_port, oc_user, oc_user_pwd)
        tenant_domain = self.config_tool.get_value_by_key_and_sub_key("tenant_api", "domain")
        tenant_port = self.config_tool.get_value_by_key_and_sub_key("tenant_api", "port")
        self._config_tenant_api(request_api, tenant_domain, tenant_port)
        syslog_domain = self.config_tool.get_value_by_key_and_sub_key("syslog_config", "domain")
        syslog_port = self.config_tool.get_value_by_key_and_sub_key("syslog_config", "port")
        self._config_syslog(request_api, syslog_domain, syslog_port)
        cts_domain = self.config_tool.get_value_by_key_and_sub_key("cts_config", "domain")
        cts_port = self.config_tool.get_value_by_key_and_sub_key("cts_config", "port")
        self._config_cts(request_api, cts_domain, cts_port)
        logger.info("Update dr service configure data success.")

    @staticmethod
    def _get_device_sn(request_api):
        state, req_body = request_api.send_get_request("ws/drmservers")
        if state != 200:
            logger.error(
                f"Get drm server info failed, status={state}, response={str(req_body)}.")
            raise Exception(
                f"Get drm server info failed, status={state}, response={str(req_body)}.")
        server_id = dict(req_body[0]).get('uuid')
        logger.info(f"DRM server uuid is [{server_id}]")
        state, req_body = request_api.send_get_request(f"ws/sites?serverUuid={server_id}")
        if state != 200:
            logger.error(
                f"Get dr site info failed, status={state}, response={str(req_body)}.")
            raise Exception(
                f"Get dr site info failed, status={state}, response={str(req_body)}.")
        for siteinfo in req_body:
            site_id = dict(siteinfo).get("siteId")
            state, req_body = request_api.send_get_request(
                f"ws/fusionsphere?siteId={site_id}&cloudType=")
            if state != 200:
                logger.error(
                    f"Get dr site info failed, status={state}, response={str(req_body)}.")
                raise Exception(
                    f"Get dr site info failed, status={state}, response={str(req_body)}.")
            for fsp_info in req_body:
                return dict(fsp_info).get("deviceSn")
        raise Exception("Get openstack info failed.")

    @staticmethod
    def _modify_openstack(request_api, ops_uuid, iam_host, iam_user, iam_pwd):
        encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        rediscover_url = \
            f"ws/resources/{encode_uuid}/action/modifyMgrProtocol"
        data = {"networkIpAddress": iam_host, "networkUserName": iam_user,
                "isModifyPwd": True, "networkPassword": iam_pwd,
                "extendParams": {"port": 26335, "domain": "op_service"}}
        state, req_body = request_api.send_put_request(
            data, rediscover_url)
        if state != 200:
            logger.info("Modify OpenStack authentication information failed.")
            raise Exception(
                'Failed to modify OpenStack authentication information.')
        logger.info("Modify OpenStack authentication information success.")

    @staticmethod
    def _refresh_device(request_api, ops_uuid):
        encode_uuid = base64.b64encode(
            ops_uuid.encode(encoding='utf-8')).decode(encoding='utf-8')
        refresh_url = f"ws/fusionsphere/{encode_uuid}/action/refresh"
        state, body = request_api.send_put_request(
            None, refresh_url)
        if state != 200:
            if 'errorCode' in dict(body) and (str(dict(body).get(
                    'errorCode', None)) == "1073947428" or str(dict(body).get('errorCode', None)) == "606"):
                # 正在刷新中
                logger.info("The device is being refresh, no need refress again.")
                sleep(20)
                return True
            raise Exception(f'Failed to refresh device[{str(body)}].')
        logger.info("Refresh_device cmd send successfully.")
        sleep(30)
        return True

    @staticmethod
    def _config_oc(request_api, domain, port, user, pwd):
        logger.info("Begin to update oc domain info.")
        body_data = {
            "domain": domain, "port": port, "username": user, "password": pwd}
        logger.info(f"Start config oc server in [{request_api.node_ip}].")
        try:
            API._do_config_oc(body_data, request_api)
        except Exception as err:
            logger.error(f"Config oc failed, reason is {str(err)}.")
            raise Exception(f'Config oc failed, reason is {str(err)}.') from err

    @staticmethod
    def _do_config_oc(body_data, request_api):
        state, body = request_api.send_put_request(
            body_data, "ws/system/thirdservice/oc")
        if state == 200:
            logger.info(f"Config oc successfully in {request_api.node_ip}.")
        else:
            logger.error(f"Config oc failed, msg: [{body}].")
            raise Exception(f"Config oc failed, msg: [{body}].")

    @staticmethod
    def _config_tenant_api(request_api, domain, port):
        logger.info(f"Start config tenant client: {domain}, {port}.")
        # ws/system/thirdservice/manageone
        req_body = {"domain": domain, "port": port}
        state, body = request_api.send_put_request(
            req_body, "ws/system/thirdservice/manageone")
        if state != 200:
            logger.error(
                f"Failed to config tenant northbound client[{body}].")
            raise Exception(body)
        logger.info("Config telnet client successfully.")

    @staticmethod
    def _config_syslog(request_api, domain, port):
        body_data = {
            "logType": "administratorlog", "isEnabled": True,
            "protocalType": "syslog",
            "protocalProps":
                f"{{\"ip\":\"{domain}\", \"port\":\""
                f"{port}\", \"encryptionMode\":\"TLS_TCP\"}}"}
        body = dict()
        for _ in range(0, 5):
            state, body = request_api.send_put_request(
                body_data, "ws/system/action/setlognotify")
            if state == 200:
                logger.info(f"Config syslog successfully in {request_api.node_ip}.")
                return
            else:
                logger.info("Retry set syslog after 60 seconds")
                sleep(60)
        logger.error(f"Config syslog failed: {body}.")
        err_code = str(dict(body).get("errorCode", None))
        if err_code == "1073947407":
            raise HCCIException(
                '663704', f"syslog: {domain}",
                f"eReplication: {request_api.node_ip}", "syslog")
        raise HCCIException('663701', domain, port, body, request_api.node_ip)

    @staticmethod
    def _config_cts(request_api, domain, port):
        body_data = {
            "logType": "tenantlog", "isEnabled": True, "protocalType": "CTS",
            "protocalProps": f"{{\"ip\":\"{domain}\", \"port\":\"{port}\"}}"
        }
        logger.info(f"Start config cts in {request_api.node_ip}: {body_data}.")
        q_state, q_body = request_api.send_get_request(
            "ws/system/action/querylognotify?logType=talentlog")
        if not DR_API.is_need_config(q_body):
            logger.info("The cts already existed, no need execute.")
            return
        state, body = request_api.send_put_request(
            body_data, "ws/system/action/setlognotify")
        if state == 200:
            logger.info(f"Config cts successfully in {request_api.node_ip}.")
            return
        else:
            logger.error(f"Config cts failed: {body}")
            err_code = str(dict(body).get("errorCode", None))
            if err_code == "1073947407":
                raise HCCIException(
                    '663704', f"CTS: {domain}",
                    f"eReplication: {request_api.node_ip}", "CTS")
            raise HCCIException(
                '663702', domain, port, body, request_api.node_ip)

    def update_unify_password(self):
        """Delete unify password & Modify unify password config"""

        account_data = self.init_delete_unify_password_request_data()
        self.oc_util.uniform_password_util(self.project_id, account_data)

        self._modify_sys_config()

    def init_delete_unify_password_request_data(self):
        """Initial unify password information to be delete"""

        account_list = list()
        ssh_user = self.nodes.ssh_user
        ssh_pwd = self.nodes.source_ssh_pwd
        sudo_user = self.nodes.sudo_user
        sudo_pwd = self.nodes.source_sudo_pwd
        for host in self.nodes.source_all_hosts:
            dr_api = DR_API(Auth(host, ssh_user, ssh_pwd, sudo_user, sudo_pwd))
            server_region_id = dr_api.get_current_region_id()
            server_float_ip = dr_api.get_server_float_ip()
            drm_account_info = {
                'accountName': 'DRManager', 'region': server_region_id,
                'ip': host, 'accountType': 1, 'operationType': 2}
            root_account_info = {
                'accountName': 'root', 'region': server_region_id,
                'ip': host, 'accountType': 1, 'operationType': 2}
            sync_account_info = {
                'accountName': 'SyncAdmin', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            csdr_iam_account_info = {
                'accountName': 'csdr_service', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            vha_iam_account_info = {
                'accountName': 'vha_service', 'region': server_region_id,
                'ip': server_float_ip, 'accountType': 5, 'operationType': 2}
            if drm_account_info not in account_list:
                account_list.append(drm_account_info)
            if root_account_info not in account_list:
                account_list.append(root_account_info)
            if sync_account_info not in account_list:
                account_list.append(sync_account_info)
            if csdr_iam_account_info not in account_list:
                account_list.append(csdr_iam_account_info)
            if vha_iam_account_info not in account_list:
                account_list.append(vha_iam_account_info)
        data = \
            {'componentName': 'eReplication', 'subComponents': [
                {'subComponentName': 'eReplication',
                 'createdAccountList': account_list}]}
        return data

    def _modify_sys_config(self):
        """Modify sys config: lego.properties"""

        installed_service = self._get_all_installed_service_from_new_cmdb()
        installed_service_iam = [service.lower() + "_service" for service in
                                 installed_service]
        global_domain = self._get_global_domain_from_new_cmdb()
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        cmdb = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user,
            project_id=self.project_id, pod_id=self.pod_id)
        sdr_infos = DR_API.get_sdr_info(cmdb)
        for host in self.nodes.target_all_hosts:
            target_client = SSH_API.get_sudo_ssh_client(Auth(host, self.nodes.ssh_user, self.nodes.target_ssh_pwd,
                                                             self.nodes.sudo_user,
                                                             self.nodes.target_sudo_pwd))
            DR_API.do_set_iam_service_name_info(
                target_client, installed_service_iam)
            DR_API.do_set_sdr_sys_config(
                target_client, sdr_infos, global_domain)
            DR_API.do_set_hcs_domain_sys_config(target_client, global_domain)
        logger.info("Modify sys config success.")

    def _get_global_domain_from_new_cmdb(self):
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user,
            project_id=self.project_id, pod_id=self.pod_id)
        region_info = mo_cmdb_ins.get_region_info()
        for region in region_info:
            region_type = region.get("regionType")
            if region_type not in ["master", "dr_standby"]:
                continue
            global_domain = region.get("globalDomainNamePostfix")
            logger.info(
                f"Get global domain from new cmdb return {global_domain}.")
            return global_domain

    def _get_all_installed_service_from_new_cmdb(self):
        """Get all installed dr service from new cmdb"""

        installed_service = list()
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user,
            project_id=self.project_id, pod_id=self.pod_id)
        region_info = mo_cmdb_ins.get_region_info()
        for region in region_info:
            region_id = region.get("regionCode")
            cloud_service_info = mo_cmdb_ins.get_cloud_service_info(
                region_id, Component.REPLICATION)
            if not cloud_service_info:
                logger.info(
                    f"Current region [{region_id}] has no DR service.")
                continue
            extend_info = cloud_service_info[0].get("extendInfos", [])
            for extend_ in extend_info:
                extend_key = extend_.get("key")
                extend_value = extend_.get("value")
                if extend_key in [Component.CSDR_U, Component.CSHA_U,
                                  Component.VHA_U] and \
                        extend_value == "True" and \
                        extend_key.lower() not in installed_service:
                    installed_service.append(extend_key.lower())
        logger.info(f"Get all installed services from new cmdb return: "
                    f"{installed_service}.")
        return installed_service

    @staticmethod
    def _get_all_business_region_ids(cmdb):
        regions = cmdb.get_region_info()
        region_ids = set()
        for region in regions:
            region_id = region.get("regionCode")
            region_type = region.get("regionType")
            if region_type in ["master", "dr_standby"]:
                continue
            region_ids.add(region_id)
        return region_ids

    def _get_all_supported_dr_region_ids(self, cmdb):
        region_ids = self._get_all_business_region_ids(cmdb)
        supported_dr_region_ids = \
            [region_id for region_id in region_ids if
             cmdb.get_cloud_service_info(region_id, Component.REPLICATION)]
        return supported_dr_region_ids

    def clear_dr_service_alarm(self):
        logger.info("Start clear dr alarm info.")
        _, float_ip = DR_API.get_server_node(
            self.nodes.target_all_hosts, self.nodes.ssh_user,
            self.nodes.target_ssh_pwd, self.nodes.sudo_user,
            self.nodes.target_sudo_pwd)
        request_api = RequestApi(
            float_ip, self.nodes.service_name, self.nodes.service_pwd,
            self.nodes.service_port)
        alarm_lst = self._query_alarm_data(request_api)
        if not alarm_lst:
            logger.info("Alarm info is empty, no action is required")
            return
        self._clear_alarm_data(request_api, alarm_lst)
        logger.info("Clear dr alarm info success.")

    @staticmethod
    def _query_alarm_data(request_api):
        logger.info("Start query dr alarm info.")
        query_alarm_uri = 'ws/alarms?startPage=0&pageSize=100'
        state, resp_body = request_api.send_get_request(query_alarm_uri)
        if state != 200 or len(resp_body) == 0:
            logger.error(
                f"Failed to get the DR alarm information. "
                f"status={state}, response={str(resp_body)}.")
            return []
        return dict(resp_body).get('records')

    @staticmethod
    def _clear_alarm_data(request_api, alarm_lst):
        logger.info(f"Start clear alarm info, size={len(alarm_lst)}")
        clear_alarm_uri = 'ws/alarms/action/clearAlarmConfirm'
        data = list(map(lambda item: {"extTargetId": item['entity'], "targetName": item['alarmNameKey']}, alarm_lst))
        state, resp_body = request_api.send_put_request(data, clear_alarm_uri)
        if state != 200:
            logger.error(f"status={state}, response={str(resp_body)}.")
            raise Exception("Failed to clear the DR alarm information.")
        logger.info("The DR alarm is cleared successfully.")

    def update_silvan(self):
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        om_ip = migrate_util.get_mo_oc_info()[0]
        oc_user, oc_user_pwd = migrate_util.get_nbi_user_password()
        mo_cmdb_ins = MO_API.CMDB.get_instance(
            om_ip, oc_user_pwd, oc_username=oc_user, project_id=self.project_id, pod_id=self.pod_id)
        migrate_util = ManageOneDataMigrateUtil(self.project_id)
        mo_lvs_ip = migrate_util.get_migrated_lvs_ip()
        mo_port = 26335
        iam_ip = migrate_util.get_iam_float_ip()
        iam_port = 26335
        iam_token = self._get_token(
            "csdr_service", self.params.iam_account_preset_pwd, iam_ip, iam_port)
        self._update_endpoints(mo_lvs_ip, mo_port, iam_token, mo_cmdb_ins)
        self._update_links(mo_lvs_ip, mo_port, iam_token, mo_cmdb_ins)
        self._update_region_endpoints(mo_lvs_ip, mo_port, iam_token, mo_cmdb_ins)
        self._update_html_templates(mo_lvs_ip, mo_port, iam_token)

    @staticmethod
    def _get_token(user_name: str, password: str, iam_address: str, port: int):
        logger.info(f"Start get {user_name} token.")
        header = {'Content-Type': 'application/json', 'Accept': 'application/json'}
        token_url = "/v3/auth/tokens"
        # 获取机机账号的token
        url_token_get = f"https://{iam_address}:{port}{token_url}"
        token_param = {"auth": {"identity": {"methods": ["password"], "password": {"user": {
            "domain": {"name": "op_service"}, "name": user_name, "password": password}}}, "scope": {
            "domain": {"name": "op_service"}}}}
        resp = requests.post(url_token_get, headers=header, data=json.dumps(token_param), verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            logger.error(f"Failed to get {user_name} token.")
            raise Exception(f"Failed to get {user_name} token.")
        token = resp.headers.get('x-subject-token')
        return token

    @staticmethod
    def _get_migrated_console_domain(cmdb):
        region_info = cmdb.get_region_info()
        for region in region_info:
            if region.get("regionType") not in ["master", "dr_standby"]:
                continue
            region_id = region.get("regionCode")
            no_b2b_mo_service_info = cmdb.get_cloud_service_info(region_id, "ManageOne")
            b2b_mo_service_info = cmdb.get_cloud_service_info(region_id, "ManageOne_B2B")
            mo_service_info = no_b2b_mo_service_info[
                0] if no_b2b_mo_service_info else b2b_mo_service_info[0]
            mo_extend_info = mo_service_info.get("extendInfos", None)
            global_domain_name = API._get_external_global_domain_name(mo_extend_info)
            console_domain_name_prefix = API._get_console_domain_name_prefix(mo_extend_info)
            return f"{console_domain_name_prefix}.{global_domain_name}"

    @staticmethod
    def _get_console_domain_name_prefix(mo_extend_info):
        for ele in mo_extend_info:
            if ele.get("key", None) == "console_domain_name_prefix":
                return ele.get("value", None)
        raise Exception("Get console domain name prefix failed.")

    @staticmethod
    def _get_external_global_domain_name(mo_extend_info):
        for ele in mo_extend_info:
            if ele.get("key", None) == "external_global_domain_name":
                return ele.get("value", None)
        raise Exception("Get external global domain name failed.")

    @staticmethod
    def _get_migrated_global_domain(cmdb):
        region_info = cmdb.get_region_info()
        for region in region_info:
            if region.get("regionType") not in ["master", "dr_standby"]:
                continue
            region_id = region.get("regionCode")
            no_b2b_mo_service_info = cmdb.get_cloud_service_info(region_id, "ManageOne")
            b2b_mo_service_info = cmdb.get_cloud_service_info(region_id, "ManageOne_B2B")
            mo_service_info = no_b2b_mo_service_info[
                0] if no_b2b_mo_service_info else b2b_mo_service_info[0]
            mo_extend_info = mo_service_info.get("extendInfos", None)
            global_domain_name = API._get_global_domain_name(mo_extend_info)
            return global_domain_name

    @staticmethod
    def _get_global_domain_name(mo_extend_info):
        for ele in mo_extend_info:
            if ele.get("key", None) == "global_domain_name":
                return ele.get("value", None)
        raise Exception("Get global domain name failed.")

    def _update_endpoints(self, mo_lvs_ip, mo_port, iam_token, cmdb):
        logger.info("Do update endpoints.")
        endpoint_lst = self._get_endpoint_info(mo_lvs_ip, mo_port, iam_token)
        migrated_console_domain = self._get_migrated_console_domain(cmdb)
        for endpoint in endpoint_lst:
            seq_id = endpoint.get("seqId")
            del endpoint["seqId"]
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints/{seq_id}"
            source_uri = endpoint.get("endpoint")
            console_domain = source_uri.strip("https://").split("/csdr/")[0]
            new_uri = source_uri.replace(console_domain, migrated_console_domain)
            endpoint.update({"endpoint": new_uri})
            endpoint.update({"deployMode": "global"})
            self._do_update_service_endpoints(uri, iam_token, endpoint)
        logger.info("Endpoints update completed.")

    def _get_endpoint_info(self, mo_lvs_ip, mo_port, iam_token):
        endpoint_lst = list()
        all_services = self._get_all_installed_service_from_new_cmdb()
        for service in all_services:
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints?start=0&limit=1000&id={service}"
            logger.info(f"Get {service}'s endpoints.")
            endpoint_zh = self._do_get_service_endpoints(uri, iam_token)
            endpoint_en = self._do_get_service_endpoints(uri, iam_token, language="en-us")
            endpoint_lst.extend(endpoint_zh)
            endpoint_lst.extend(endpoint_en)
        logger.info(f"Get endpoints return {endpoint_lst}.")
        return endpoint_lst

    @staticmethod
    def _do_get_service_endpoints(uri, iam_token, language='zh-cn'):
        service = uri.split("&id=")[1]
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        headers.update({'X-Language': language})
        resp = requests.request(method="get", url=uri, headers=headers, verify=False)
        if not resp.content:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="get", url=uri, headers=headers, verify=False)
            if not resp.content:
                raise Exception(f"Get {service}'s endpoints failed: {language}.")
        endpoints_info = json.loads(resp.content)
        total = endpoints_info['total']
        logger.info(f"Get {service}'s endpoints return {endpoints_info}")
        if total <= 0:
            logger.error(f"Get {service}'s endpoints return total {total} records.")
            raise Exception(f"Get {service}'s endpoints return total {total} records.")
        return endpoints_info["endpoints"]

    @staticmethod
    def _do_update_service_endpoints(uri, iam_token, new_endpoint):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="put", url=uri, headers=headers, json=new_endpoint, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="put", url=uri, headers=headers, json=new_endpoint, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do update endpoint failed.")
                raise Exception("Do update endpoint failed.")
        if "success" not in str(resp.content):
            logger.error(f"Update endpoints request return: {resp.content}.")
            raise Exception(f"Update endpoints request return: {resp.content}.")
        logger.info("Do update endpoint success.")

    def _update_links(self, mo_lvs_ip, mo_port, iam_token, cmdb):
        migrated_console_domain = self._get_migrated_console_domain(cmdb)
        delete_links, update_links = self._get_links_info(mo_lvs_ip, mo_port, iam_token, cmdb)
        for link_d in delete_links:
            seq_id = link_d.get('seqId')
            del link_d['seqId']
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/links/{seq_id}"
            self._do_delete_service_links(uri, iam_token)
        for link_u in update_links:
            seq_id = link_u.get('seqId')
            del link_u['seqId']
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/links/{seq_id}"
            source_uri = link_u.get("href")
            console_domain = source_uri.strip("https://").split("/csdr/")[0]
            new_uri = source_uri.replace(console_domain, migrated_console_domain)
            link_u.update({"endpoint": new_uri})
            self._do_update_service_links(uri, iam_token, link_u)
        logger.info("Links update success.")

    def _get_links_info(self, mo_lvs_ip, mo_port, iam_token, cmdb):
        delete_links = list()
        supported_dr_regions = self._get_all_supported_dr_region_ids(cmdb)
        for region_id in supported_dr_regions:
            links = self._do_get_service_links(mo_lvs_ip, mo_port, iam_token, region_id=region_id)
            delete_links.extend(links)
        update_links = self._do_get_service_links(mo_lvs_ip, mo_port, iam_token)
        return delete_links, update_links

    @staticmethod
    def _do_get_service_links(mo_lvs_ip, mo_port, iam_token, region_id=None):
        links_list = list()
        uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/links"
        if region_id:
            uri += f'?region={region_id}'
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="get", url=uri, headers=headers, verify=False)
        if not resp.content:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="get", url=uri, headers=headers, verify=False)
            if not resp.content:
                raise Exception(f"Get links failed.")
        links_info = json.loads(resp.content)
        total = links_info['total']
        if total <= 0:
            logger.error(f"Get links return total {total} records.")
            raise Exception(f"Get links return total {total} records.")
        for link in links_info['links']:
            if link.get('key').split('_')[-1] not in [Component.CSDR_L, Component.CSHA_L, Component.VHA_L]:
                continue
            if region_id and link.get('region_id') != region_id:
                continue
            links_list.append(link)
        logger.info(f"Get {region_id}'s links return: {links_list}.")
        return links_list

    @staticmethod
    def _do_delete_service_links(uri, iam_token):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="delete", url=uri, headers=headers, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="delete", url=uri, headers=headers, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do delete links failed.")
                raise Exception("Do delete links failed.")
        if "success" not in str(resp.content):
            logger.error(f"Delete links request return: {resp.content}.")
            raise Exception(f"Delete links request return: {resp.content}.")
        logger.info("Do delete links success.")

    @staticmethod
    def _do_update_service_links(uri, iam_token, new_links):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="put", url=uri, headers=headers, json=new_links, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="put", url=uri, headers=headers, json=new_links, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do update links failed.")
                raise Exception("Do update links failed.")
        if "success" not in str(resp.content):
            logger.error(f"Update links request return: {resp.content}.")
            raise Exception(f"Update links request return: {resp.content}.")
        logger.info("Do update links success.")

    def _update_region_endpoints(self, mo_lvs_ip, mo_port, iam_token, cmdb):
        global_domain = self._get_migrated_global_domain(cmdb)
        all_services = self._get_all_installed_service_from_new_cmdb()
        region_endpoints = self._get_region_endpoints_info(mo_lvs_ip, mo_port, iam_token)
        for region_endpoint in region_endpoints:
            del region_endpoint['seqId']
            service = region_endpoint.get("endpointId")
            region_id = region_endpoint.get("regionId")
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints/{service}/regions/{region_id}"
            new_address = f"{service}.{global_domain}"
            region_endpoint.update({"address": new_address})
            self._do_update_service_region_endpoints(uri, iam_token, region_endpoint)
        for service in all_services:
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints/{service}/regions/*"
            if self._judge_service_region_endpoints_exists(uri, service, global_domain, iam_token):
                logger.info(f"Global {service}'s region endpoints already exists.")
                continue
            new_address = f"{service}.{global_domain}"
            global_region_endpoints = {"address": new_address, "port": 9443}
            self._do_register_service_region_endpoints(uri, iam_token, global_region_endpoints)
        logger.info(f"Region endpoints update success.")

    def _get_region_endpoints_info(self, mo_lvs_ip, mo_port, iam_token):
        region_endpoint_lst = list()
        all_services = self._get_all_installed_service_from_new_cmdb()
        all_services += ["cdrs"]
        for service in all_services:
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/endpoints/{service}/regions"
            region_endpoints = self._do_get_service_region_endpoints(uri, iam_token)
            region_endpoint_lst.extend(region_endpoints)
        logger.info(f"Get region endpoints return: {region_endpoint_lst}.")
        return region_endpoint_lst

    @staticmethod
    def _do_get_service_region_endpoints(uri, iam_token, is_filter=False):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="get", url=uri, headers=headers, verify=False)
        if not resp.content:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="get", url=uri, headers=headers, verify=False)
            if not resp.content:
                raise Exception(f"Get region endpoints failed.")
        region_endpoints_info = json.loads(resp.content)
        return region_endpoints_info if is_filter else region_endpoints_info["regionEndpoints"]

    @staticmethod
    def _do_update_service_region_endpoints(uri, iam_token, new_region_endpoints):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="put", url=uri, headers=headers, json=new_region_endpoints, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="put", url=uri, headers=headers, json=new_region_endpoints, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do update region endpoints failed.")
                raise Exception("Do update region endpoints failed.")
        if "success" not in str(resp.content):
            logger.error(f"Update region endpoints request return: {resp.content}.")
            raise Exception(f"Update region endpoints request return: {resp.content}.")
        logger.info("Do update region endpoints success.")

    @staticmethod
    def _do_register_service_region_endpoints(uri, iam_token, new_region_endpoints):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(
            method="post", url=uri, headers=headers, json=new_region_endpoints, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(
                method="post", url=uri, headers=headers, json=new_region_endpoints, verify=False)
            if not resp.content:
                logger.error(f"Register service region endpoints failed.")
                raise Exception("Register service region endpoints failed.")
        if "success" not in str(resp.content):
            logger.error(f"Register region endpoints request return: {resp.content}.")
            raise Exception(f"Register region endpoints request return: {resp.content}.")
        logger.info("Register service region endpoints success.")

    def _judge_service_region_endpoints_exists(self, uri, service, global_domain, iam_token):
        global_region_endpoints = self._do_get_service_region_endpoints(uri, iam_token, is_filter=True)
        return True if f'{service}.{global_domain}' in str(
            global_region_endpoints) and "*" in str(global_region_endpoints) else False

    def _update_html_templates(self, mo_lvs_ip, mo_port, iam_token):
        services_html_templates = self._get_services_html_templates(mo_lvs_ip, mo_port, iam_token)
        for html_template in services_html_templates:
            if not html_template.get("region_id", ""):
                logger.info(f"Get {html_template.get('service')}'s html-templates region_id is null, skip delete")
                continue
            resource_id = html_template.get('resourceId')
            if not resource_id:
                logger.info(f"Get {html_template.get('service')}'s html-templates resourceId is null, skip delete")
                continue
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/html-templates/{resource_id}"
            self._do_delete_service_html_templates(uri, iam_token)
        logger.info("Do delete services html templates success.")

    def _get_services_html_templates(self, mo_lvs_ip, mo_port, iam_token):
        services_html_templates = list()
        for service in [Component.CSDR_L, Component.CSHA_L, Component.VHA_L]:
            uri = f"https://{mo_lvs_ip}:{mo_port}/silvan/rest/v1.0/html-templates?service={service}"
            html_templates = self._do_get_service_html_templates(uri, iam_token)
            services_html_templates.extend(html_templates)
        logger.info(f"Get services html templates return: {services_html_templates}.")
        return services_html_templates

    @staticmethod
    def _do_get_service_html_templates(uri, iam_token):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="get", url=uri, headers=headers, verify=False)
        if not resp.content:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="get", url=uri, headers=headers, verify=False)
            if not resp.content:
                raise Exception(f"Get service html templates failed.")
        html_templates_info = json.loads(resp.content)
        return html_templates_info["htmlTemplates"]

    @staticmethod
    def _do_delete_service_html_templates(uri, iam_token):
        headers = {'Content-Type': 'application/json;', 'Accept': 'application/json;charset=utf8'}
        headers.update({'x-auth-token': iam_token})
        resp = requests.request(method="delete", url=uri, headers=headers, verify=False)
        if resp.status_code not in HTTP_STATUS_OK:
            del headers['x-auth-token']
            headers.update({'X-Auth-Token': iam_token})
            resp = requests.request(method="delete", url=uri, headers=headers, verify=False)
            if resp.status_code not in HTTP_STATUS_OK:
                logger.error("Do delete service html templates failed.")
                raise Exception("Do delete service html templates failed.")
        if "success" not in str(resp.content):
            logger.error(f"Delete service html templates request return: {resp.content}.")
            raise Exception(f"Delete service html templates request return: {resp.content}.")
        logger.info("Do delete service html templates success.")

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(30), reraise=True)
    def update_console(self):
        params = Params(self.project_id, self.pod_id)
        DMK_API.login_dmk(params.target_dmk_float_ip, params.dmk_deploy_user, params.dmk_user_new_pwd)
        region_type = "master" if self.condition.is_primary else "standby"
        migrated_nodes = MOMigratedNodeInfos(self.project_id, region_type=region_type).node_infos
        tenant_nodes = [node for node in migrated_nodes if "MOConsoleFrameworkService" in node.service_names]
        if not tenant_nodes:
            logger.error("Get tenant nodes failed.")
            raise Exception("Get tenant nodes failed.")
        root_pwd = ""
        ssh_user = "ossuser"
        sudo_user = "root"
        ossuser_pwd = ""
        tenant_ips = [node.ip_address for node in tenant_nodes]
        for user_info in tenant_nodes[0].os_users_info:
            if user_info.user_name == sudo_user:
                root_pwd = user_info.user_password
            if user_info.user_name == ssh_user:
                ossuser_pwd = user_info.user_password

        team_id = DMK_API.get_dmk_team_id(Component.REPLICATION)

        DR_API.create_remote_account(
            team_id, tenant_ips, DmkAccountInfo(params.dmk_deploy_user, ssh_user, ossuser_pwd, root_pwd,
                                                params.dmk_deploy_group))
        account_id = DMK_API.get_dmk_account_id(params.dmk_deploy_user, ssh_user)
        # 从服务器上查找软件包
        pkg_path = PKG_API.get_file_abs_path_by_re(Pkg.CONSOLE_PKG_RE)
        logger.info(f"Get console package return {pkg_path}.")
        PKG_API.check_compressed_file(pkg_path, Capacity.MB_S, Pkg.CONSOLE_PKG_SIZE_LIMIT, Pkg.CONSOLE_PKG_FILE_LIMIT)
        # 上传
        DMK_API.upload_pkg_to_dmk(pkg_path)
        pkg_version = PKG_API.get_version(pkg_path)
        host_info = f"method=su\nstatic_hosts={','.join(tenant_ips)}"
        is_standby_region = "no" if self.condition.is_primary else "yes"
        config_info = f"---\nversion: {pkg_version}\nremote_ssh_user: {ssh_user}" \
                      f"\nis_standby_region: '{is_standby_region}'\nConsole:\n  " \
                      f"hosts:\n    - '{tenant_ips[0]}'\n    - '{tenant_ips[1]}'"

        result = DMK_API.execute_dmk_deployment(DmkDeploymentInfo(Component.CSDR_U, True, pkg_version, "Deploy",
                                                                  host_info, config_info, account_id))
        if not result:
            logger.error("Execute console Patch install failed.")
            raise Exception("Execute console Patch install failed.")
