# -*- coding:utf-8 -*-
import json
from collections import defaultdict
from time import sleep

import requests
import utils.common.log as logger

from plugins.eReplication.common.lib.utils import check_host_connection

SERVICE_OK = 0
SERVICE_PWD_ERROR = 1
SERVICE_NOT_OK = 2
SERVICE_INVALID = 3
SERVICE_IP_NOT_CONNECTED = 4


class RequestApi(object):
    def __init__(self, node_ip: str, user: str, pwd: str, port: int,
                 raise_ex: bool = True):
        self.node_ip = node_ip
        self.user = user
        self.pwd = pwd
        self.port = port
        self.raise_ex = raise_ex
        if ":" in self.node_ip:
            self.node_ip = f"[{self.node_ip}]"
        self.url = f"https://{self.node_ip}:{self.port}/"
        self.header = self.build_header()

    def build_header(self):
        _headers = dict()
        _headers.update({"Content-Type": "application/json;charset=utf-8"})
        _headers.update(
            {"Accept": "application/json;version=1.0;charset=UTF-8"})
        _headers.update({"Accept-Language": "zh_CN"})
        _headers.update({"x-auth-user": self.user})
        _headers.update({"x-auth-key": self.pwd})
        return _headers

    @staticmethod
    def get_resp_data(resp):
        if resp is None:
            return defaultdict()
        if resp != 'None' and resp != 'null' and resp:
            return json.loads(resp)
        return defaultdict()

    def send_post_request(self, body, path):
        if isinstance(body, dict) or isinstance(body, list):
            body = json.dumps(body)
        resp = requests.post(
            self.url + path, headers=self.header, data=body, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def send_get_request(self, path):
        resp = requests.get(
            self.url + path, headers=self.header, data=None, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def send_put_request(self, body, path):
        if isinstance(body, dict) or isinstance(body, list):
            body = json.dumps(body)
        resp = requests.put(
            self.url + path, headers=self.header, data=body, verify=False)
        return resp.status_code, self.get_resp_data(resp.content)

    def check_dr_service(self):
        """
        检查容灾服务后台是否可用
        检查方式: 通过服务ip、用户名、密码发送一条rest请求,检查能否正确获取到指定消息
        :return:
        """
        result = None
        if not check_host_connection(self.node_ip):
            result = SERVICE_IP_NOT_CONNECTED, ""
        for i_tmp in range(0, 3):
            logger.info(f"Check dr service, times: {i_tmp + 1}.")
            try:
                state, resp_body = self.send_get_request("ws/drmservers")
            except Exception as err:
                logger.error(f"Check DR service failed on {self.node_ip}: {err}.")
                result = SERVICE_INVALID, str(err)
                continue
            if state == 200:
                result = SERVICE_OK, ""
                break
            logger.error(f"Check service failed: {state},  {resp_body}.")
            if 'errorCode' in resp_body and str(
                    resp_body['errorCode']) == "1073947659":
                # 用户名或密码错误
                result = SERVICE_PWD_ERROR, ""
            else:
                result = \
                    SERVICE_NOT_OK, f"state: {state}, result: {resp_body}"
            sleep(2)

        if self.raise_ex:
            if result[0] != SERVICE_OK:
                logger.error("DR service is unavailable.")
                raise Exception("DR service is unavailable.")
        return result
