# -*- coding:utf-8 -*-

import utils.common.log as logger
from utils.business.project_util import ProjectApi
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException

from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.dmk_client import API as DMK_API
from plugins.eReplication.common.constant import VERSION_MAP
from plugins.eReplication.common.dr_api import API as DR_API
from plugins.eReplication.common.lib.base import BaseSubJob
from plugins.eReplication.common.lib.decorator import handle_task_check_result
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.upgrade_api import API as UPGRADE_API

logger.init("eReplication")


class CheckVersion(BaseSubJob):
    """校验升级过程的所需要的通信矩阵是否放开"""

    @handle_task_check_result
    def execute(self, project_id, pod_id, *args, **kwargs):
        """标准调用接口：执行安装&配置

        :param project_id:
        :param pod_id:
        :return:
        """

        results = list()
        pkg_version = PKG_API.get_pkg_version()
        res, current_version, results = \
            self._check_all_node_version(results)
        if not res:
            return results
        solution, version = ProjectApi.get_solution_and_version(
            self.project_id)
        version_list = version.split('-')
        src_version = VERSION_MAP.get(version_list[0])
        target_version = VERSION_MAP.get(version_list[1])
        check_pkg_res, results = \
            self._check_pkg_version(pkg_version, target_version, results)
        if not check_pkg_res:
            return results
        check_env_res, results = \
            self._check_current_version_support_upgrade(
                current_version, src_version, target_version, results)
        if not check_env_res:
            return results
        check = CheckResult(
            itemname_ch="升级版本号检查",
            itemname_en="Checking the Upgrade Version", status="success")
        results.append(check)
        return results

    def _check_all_node_version(self, results):
        version_list = list()
        current_version = None
        nodes = Nodes(self.project_id, self.pod_id)
        for host in nodes.all_hosts:
            current_version = DR_API(Auth(host, nodes.ssh_user, nodes.ssh_pwd, nodes.sudo_user,
                                          nodes.sudo_pwd)).get_current_version()
            version_list.append(current_version)
        if len(set(version_list)) > 1:
            logger.error(
                "The local and remote service versions are different.")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version", status="failure",
                error_msg_cn=HCCIException(675033))
            results.append(check)
            return False, current_version, results
        return True, current_version, results

    def _check_pkg_version(self, pkg_version, target_version, results):
        params = Params(self.project_id, self.pod_id)
        DMK_API.login_dmk(
            params.dmk_float_ip, params.dmk_deploy_user,
            params.dmk_user_new_pwd)
        if not pkg_version.startswith(target_version):
            logger.error(
                f"The version of the upgrade software package"
                f"[{pkg_version}] does not match the target version"
                f"[{target_version}].")
            check = CheckResult(
                itemname_ch="升级版本号检查",
                itemname_en="Checking the Upgrade Version", status="failure",
                error_msg_cn=HCCIException(
                    675034, pkg_version, target_version))
            results.append(check)
            # 已上传的包跟目标版本不匹配，先删除已上传到DMK上的软件包
            DMK_API.delete_pkg_from_dmk("eReplication", pkg_version)
            return False, results
        # 软件包校验成功，重新触发上传软件包操作
        UPGRADE_API(self.project_id, self.pod_id, self.region_id). \
            upload_server_package()
        logger.info("Upload package successfully.")
        return True, results

    @staticmethod
    def _check_current_version_support_upgrade(
            current_version, src_version, target_version, results):
        if not (current_version.startswith(src_version) or
                current_version.startswith(VERSION_MAP.get("6.5.1"))) and \
                not (current_version == "8.3.RC1" and target_version == "8.3.0"):
            logger.error(
                f"The version of the eReplication software[{current_version}] "
                f"does not match the source version[{src_version}].")
            check = CheckResult(
                itemname_ch="升级版本号检查", itemname_en="Checking the Upgrade Version",
                status="failure", error_msg_cn=HCCIException(675035))
            results.append(check)
            return False, results
        return True, results
