# -*- coding:utf-8 -*-
import os
import shutil
import zipfile

import utils.common.log as logger
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.mo_client import API as MO_API
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.lib.base import BaseSubJob
from plugins.eReplication.common.lib.decorator import handle_task_result

logger.init("InstallConsole")


class InstallConsole(BaseSubJob):
    """
    完成Console软件的安装部署
    """

    def __init__(self, project_id, pod_id, regionid_list=None):
        super(InstallConsole, self).__init__(project_id, pod_id, regionid_list)
        self.pkg_console_path = None
        self.pkg_fcd_console_path = None
        self.fcd_home_pkg_temp_dir = Path.LOCAL_TEMP_PATH

    def unzip(self, file, target_path):
        logger.info(f"Start unzip console file {file} to {target_path}.")
        try:
            self._do_unzip(file, target_path)
        except FileNotFoundError or zipfile.BadZipFile as err:
            logger.error(f"Unzip {file} failed: {err}.")
            raise HCCIException("663605", file)
        logger.info(
            f"Unzip console file {file} to {target_path} success.")

    @staticmethod
    def _do_unzip(file, target_path):
        # 解压前校验文件大小
        PKG_API.check_compressed_file(
            file, unit="MB", size_limit=Pkg.CONSOLE_PKG_SIZE_LIMIT,
            file_num_limit=Pkg.CONSOLE_PKG_FILE_LIMIT)
        PKG_API.extract_file(file, target_path)

    def _find_pkg(self, console_release=False):
        pkg_re = Pkg.CONSOLE_PKG_RE
        pkg_path = None
        if console_release:
            pkg_re = Pkg.CONSOLE_RELEASE_RE
            pkg_path = self.fcd_home_pkg_temp_dir
        find_pkg_path = PKG_API.get_file_abs_path_by_re(
            pkg_re, pkg_path=pkg_path)
        if console_release:
            self.pkg_fcd_console_path = find_pkg_path
        else:
            self.pkg_console_path = find_pkg_path

    def _unzip_pkg(self):
        self._find_pkg()
        if os.path.exists(self.fcd_home_pkg_temp_dir):
            shutil.rmtree(self.fcd_home_pkg_temp_dir)
        os.mkdir(self.fcd_home_pkg_temp_dir)
        self.unzip(self.pkg_console_path, self.fcd_home_pkg_temp_dir)
        self._find_pkg(console_release=True)

    @handle_task_result
    def execute(self, project_id, pod_id, *args, **kwargs):
        """
        标准调用接口：执行安装至MO静态节点
        :param project_id:
        :param pod_id:
        :return:Message类对象
        """
        self._unzip_pkg()
        mo_api = MO_API(self.project_id, self.pod_id)
        mo_api.upload_console_pkg(self.fcd_home_pkg_temp_dir)
        result = mo_api.deploy_console(
            self.pkg_fcd_console_path)
        if result.status_code == 200:
            shutil.rmtree(self.fcd_home_pkg_temp_dir)
            return
        raise Exception(f"Deploy console failed: {result}")
