# -*- coding:utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
import re
import threading
import time

import utils.common.log as logger
from tenacity import retry, stop_after_attempt, wait_fixed
from utils.common.exception import HCCIException

from plugins.eReplication.common.api.pkg_api import API as PKG_API
from plugins.eReplication.common.client.ssh_client import API as SSH_API
from plugins.eReplication.common.constant import Capacity
from plugins.eReplication.common.constant import Config
from plugins.eReplication.common.constant import EulerUpgrade
from plugins.eReplication.common.constant import OS
from plugins.eReplication.common.constant import Path
from plugins.eReplication.common.constant import Pkg
from plugins.eReplication.common.lib.model import Auth
from plugins.eReplication.common.lib.params import Nodes
from plugins.eReplication.common.lib.params import Params
from plugins.eReplication.common.lib.thread import ExcThread
from plugins.eReplication.common.lib.utils import get_param_value
from plugins.eReplication.common.lib.utils import set_param_value


class API(object):

    def __init__(self, project_id, pod_id, region_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.region_id = region_id
        self.nodes = Nodes(self.project_id, self.pod_id)
        self.params = Params(self.project_id, self.pod_id)
        skip_os_upgrade_ips = get_param_value(pod_id, Config.OS_CONFIG, Config.SKIP_OS_UPGRADE_IPS,
                                              Config.SKIP_OS_UPGRADE_IPS)
        self.skip_os_list = skip_os_upgrade_ips.split(",") if skip_os_upgrade_ips else []

    def add_node_ip_to_cfg(self, host, key=Config.SKIP_OS_UPGRADE_IPS):
        """
        :param host: 需要添加的ip
        :param key: skip_os_upgrade_ips: 记录版本已经是新版本或者不用升级Os的ip列表,用于重试跳过，为节约时间已经升级过的OS不再升级
        """
        logger.info(f"Add node ip to cfg, on ({host}) begin.")
        ip_string = get_param_value(self.pod_id, Config.OS_CONFIG, key, key)
        skip_os_list = ip_string.split(",") if ip_string else []
        if skip_os_list and host not in skip_os_list:
            set_param_value(self.project_id, Config.OS_CONFIG, self.region_id, key, f"{ip_string},{host}")
        elif skip_os_list and host in skip_os_list:
            return
        else:
            set_param_value(self.project_id, Config.OS_CONFIG, self.region_id, key, host)
        logger.info(f"Add node ip to cfg, on ({host}) end.")

    @staticmethod
    def assemble_logger_strings(state_key):
        logger_strings = "Current state is [ "
        state = OS.UPGRADE_PRECENT_MAP.get(state_key, '')
        logger_strings += f"{state_key}: {state} "
        logger_strings += "]."
        return logger_strings

    def check_os_image_mounted(self, sudo_client):
        """检查镜像是否已经挂载 已挂载： True 未挂载： False"""
        logger.info("Check OS image mounted.")
        os_arch = self._get_os_arch(sudo_client)
        cmd = f"mount | grep -v deleted | grep -qP " \
              f"'{Path.ISO_PKG_PATH}/OceanStor_BCManager_EulerOS_{os_arch}.iso" \
              f".*{Path.ISO_UPGRADE_PATH}';echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            logger.info("The OS image is not mounted to the OS.")
            return False
        else:
            logger.info(f"The OS image has been mounted to the OS.")
            return True

    def check_os_version(self):
        logger.info(f"Start to check os version.")
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user, self.nodes.sudo_pwd)
            func = (self._check_os_version, thread_name, (auth,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info(f"Check os version end.")

    def check_vg_free_space(self):
        logger.info(f"Start to check vg free space.")
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            if host in self.skip_os_list:
                logger.info(f"Skip check vg free space on {host}.")
                continue
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user, self.nodes.sudo_pwd)
            func = (self._check_vg_free_space, thread_name, (auth,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info(f"Check_vg_free_space end.")

    @staticmethod
    def get_upgrade_os_abs_pkg_path(os_arch):
        pkg_suffix = Pkg.UPGRADE_OS_PKG_SUFFIX_ARM if os_arch == 'ARM' else Pkg.UPGRADE_OS_PKG_SUFFIX_X86
        return PKG_API.get_file_abs_path(Pkg.SERVER_PKG_PREFIX, pkg_suffix)

    def mount_os_image(self, sudo_client):
        os_arch = self._get_os_arch(sudo_client)
        pkg_path = self.get_upgrade_os_abs_pkg_path(os_arch)
        PKG_API.check_compressed_file(pkg_path, Capacity.MB_S, Pkg.UPGRADE_OS_PKG_SIZE_LIMIT,
                                      Pkg.UPGRADE_OS_PKG_FILE_COUNT)
        host = sudo_client.get('ip')
        file_name = os.path.basename(pkg_path)
        logger.info(f"Start to put file {file_name} to node[{host}]")
        SSH_API.put_file(host, self.nodes.ssh_user, self.nodes.ssh_pwd, pkg_path, Path.DR_USER_PATH, 22)
        cmd = f"mv -f {Path.DR_USER_PATH}/'{file_name}' {Path.NODE_ROOT_PATH}/"
        SSH_API.exec_command_return_list(sudo_client, cmd, timeout=1)
        cmd = f"mkdir -p '{Path.ISO_PKG_PATH}' && mkdir -p '{Path.ISO_UPGRADE_PATH}' " \
              f"&& unzip -o {Path.NODE_ROOT_PATH}/'{file_name}' -d {Path.ISO_PKG_PATH}" \
              f"&& mount -t iso9660 -o nosuid,noexec,nodev " \
              f"{Path.ISO_PKG_PATH}/OceanStor_BCManager_EulerOS_{os_arch}.iso {Path.ISO_UPGRADE_PATH};" \
              f"echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            logger.error(f"Fail to mount the OS image to node[{host}].")
            raise HCCIException(663545, Path.ISO_UPGRADE_PATH, str(result))
        logger.info(f"Mount the OS image to node[{host}] successful.")

    def os_upgrade(self):
        logger.info("Os upgrade begin")
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            if host in self.skip_os_list:
                logger.info(f"Skip os upgrade on {host}.")
                continue
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user, self.nodes.sudo_pwd)
            if not self._need_os_upgrade(auth):
                logger.info(f"Check OS Version not need upgrade on {host}.")
                continue
            func = (self._os_upgrade, thread_name, (auth,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Os upgrade end")

    def rollback_os(self):
        """回退操作系统"""
        logger.info(f"Rollback os begin.")
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user, self.nodes.sudo_pwd)
            if host in self.skip_os_list:
                logger.info(f"Skip rollback os on {host}.")
                continue
            func = (self._rollback_node_os, thread_name, (auth,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info(f"Rollback os end.")

    def upgrade_os_submit(self):
        logger.info("Upgrade os submit begin")
        funcs = list()
        thread_name = threading.current_thread().name
        for host in self.nodes.all_hosts:
            if host in self.skip_os_list:
                logger.info(f"Skip os upgrade on {host}.")
                continue
            auth = Auth(host, self.nodes.ssh_user, self.nodes.ssh_pwd,
                        self.nodes.sudo_user, self.nodes.sudo_pwd)
            func = (self._clean_temp_file, thread_name, (auth,), {})
            funcs.append(func)
        ExcThread.exec_func_in_thread(funcs)
        logger.info("Upgrade os submit end")

    def _mount_bak_disk(self, auth):
        logger.info(f"Start to extend disk on {auth.host}.")
        ssh_client = SSH_API.get_sudo_ssh_client(auth)
        if self._check_bak_disk_mounted(ssh_client):
            logger.info(f"The bak disk does not need to be mounted to nodes[{auth.host}].")
            SSH_API.close_ssh(ssh_client)
            return
        mount_cmd = f"lvcreate -y -L {OS.VG_FREE_SIZE - 0.1}g -n osbak euleros && " \
                    f"mkfs.ext4 -m 0 {Path.LOGIC_VOLUME_PATH} && " \
                    f"mkdir -p {Path.BAK_TEMP_PATH} && mount {Path.LOGIC_VOLUME_PATH} {Path.BAK_TEMP_PATH}; " \
                    f"echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, mount_cmd)
        if "CMD_RESULT=0" not in str(result):
            logger.error(f"Fail to mount bak disk to node[{auth.host}].")
            raise HCCIException(675038, auth.host, str(result))
        SSH_API.close_ssh(ssh_client)
        logger.info(f"Success to mount bak disk to node[{auth.host}].")

    @staticmethod
    def _check_bak_disk_mounted(ssh_client):
        host_ip = ssh_client.get('ip')
        logger.info(f"Start to check disk mount to node[{host_ip}]")
        cmd = f"mount | grep -v deleted | grep -qP " \
              f"'euleros-osbak.*{Path.BAK_TEMP_PATH}';echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            logger.info(f"The bak disk is not mounted to node[{host_ip}].")
            return False
        else:
            logger.info(f"The bak disk has been mounted to node[{host_ip}].")
            return True

    @staticmethod
    def get_os_version(sudo_client):
        cmd = r"uname -r | grep -oP '.+(?=(\.x86_64|\.aarch64))'" \
              r"| awk -F'r' '{print $0}'"
        result = SSH_API.exec_command_return_list(sudo_client, cmd)
        if not result:
            raise Exception("Get Euler OS version failed, please check.")
        return result[0]

    def _check_os_version(self, auth):
        logger.info(f"Check os version on {auth.host} begin.")
        try:
            if not self._need_os_upgrade(auth):
                logger.info(f"Check OS Version not need upgrade on {auth.host}.")
                self.add_node_ip_to_cfg(auth.host)
        except Exception as e:
            err_msg = f"Check OS version failed, node:{auth.host}, err:{str(e)}"
            logger.error(err_msg)
            raise HCCIException(663548, auth.host, err_msg) from e
        logger.info(f"Check os version on {auth.host} end.")

    @staticmethod
    def _check_vg_free_space(auth):
        logger.info(f"Start to check vg free space on {auth.host}.")
        ssh_client = SSH_API.get_sudo_ssh_client(auth)
        cmd = "vgdisplay euleros |grep 'Free  PE / Size' | awk '{print $7,$8}'"
        try:
            result = SSH_API.exec_command_return_list(ssh_client, cmd)
            if not result:
                raise Exception("Get volume group free space failed, please check.")
            size = float(str(result[0].split()[0]).replace('<', ''))
            unit = str(result[0].split()[1])
            is_mb_small = unit == 'MiB' and size < OS.VG_FREE_SIZE * Capacity.ONE_THOUSAND
            is_gb_small = unit == 'GiB' and size < OS.VG_FREE_SIZE
            if is_mb_small or is_gb_small:
                logger.error(f"Check volume group free space fail.At least {OS.VG_FREE_SIZE}GB space is required.")
                raise HCCIException(675039, auth.host, OS.VG_FREE_SIZE)
            logger.info(f"Check vg free space on {auth.host} end.")
        except Exception as e:
            err_msg = f"Check OS version failed, node:{auth.host}, err:{str(e)}"
            logger.error(err_msg)
            raise HCCIException(663548, auth.host, err_msg) from e
        finally:
            SSH_API.close_ssh(ssh_client)

    def _check_upgrade_state(self, auth):
        logger.info(f"Check upgrade state on {auth.host} begin.")
        # 超时时间设定为55min
        timeout = 55 * 60
        interval = 120
        _time = 0
        while _time <= timeout:
            ssh_client = None
            try:
                ssh_client = SSH_API.get_sudo_ssh_client(auth)
                result, _ = self._get_current_state(ssh_client)
                logger.info(f"Current upgrade state: {result} on node: {auth.host}, count={_time}.")
                logger.info(self.assemble_logger_strings(result))
                if result == 'SUCCESS':
                    logger.info("Current upgrade state: SUCCESS, break.")
                    break
            except HCCIException as ex:
                logger.error(f"Get current failed state!  {str(ex)}")
            except Exception as ex:
                logger.error(f"Upgrade os failed! {str(ex)}")
            finally:
                SSH_API.close_ssh(ssh_client)
            _time += interval
            time.sleep(interval)
        logger.info(f"Check upgrade state on {auth.host} end.")

    @retry(stop=stop_after_attempt(10), wait=wait_fixed(60), reraise=True)
    def _clean_temp_file(self, auth):
        logger.info(f"Start to clean all os upgrade files on {auth.host}.")
        ssh_client = SSH_API.get_sudo_ssh_client(auth)
        if not self.check_os_image_mounted(ssh_client):
            self.mount_os_image(ssh_client)
        os_arch = self._get_os_arch(ssh_client)
        pkg_path = self.get_upgrade_os_abs_pkg_path(os_arch)
        file_name = os.path.basename(pkg_path)
        cmd = f"sh {Path.ISO_UPGRADE_PATH}/upgrade_iso.sh clean;echo CMD_RESULT=$?"
        result = SSH_API.exec_command_return_list(ssh_client, cmd)
        if "CMD_RESULT=0" not in str(result):
            err_msg = f"Clean all os upgrade files failed {str(result)}"
            logger.info(err_msg)
            raise HCCIException(663550, auth.host, err_msg)
        time.sleep(10)
        clean_cmd = f"umount {Path.ISO_UPGRADE_PATH};" \
                    f"umount {Path.BAK_TEMP_PATH};" \
                    f"rm -rf '{Path.BAK_TEMP_PATH}';" \
                    f"rm -rf /var/log/upgrade_iso.log.*;" \
                    f"rm -f '{Path.NODE_ROOT_PATH}/{file_name}';" \
                    f"rm -rf {Path.EULER_UPGRADE_PATH}"
        SSH_API.exec_command_return_list(ssh_client, clean_cmd)
        cmd = f"sed -i '/euleros-osbak/d' /etc/fstab;" \
              f"lvremove -y {Path.LOGIC_VOLUME_PATH}"
        SSH_API.exec_command_return_list(ssh_client, cmd)
        logger.info("Success to clean all upgrade files.")

    def _exe_rollback_node_os(self, auth):
        logger.info(f"Start to rollback OS version on {auth.host}.")
        sudo_client = SSH_API.get_sudo_ssh_client(auth)
        try:
            cmd = f"echo '/dev/mapper/euleros-osbak {Path.BAK_TEMP_PATH} ext4 defaults 1 0' >> /etc/fstab && " \
                  f"mount {Path.LOGIC_VOLUME_PATH} {Path.BAK_TEMP_PATH}"
            SSH_API.exec_command_return_list(sudo_client, cmd)
            cmd = f"chattr -R -i -a /usr &>/dev/null; sh {Path.ISO_UPGRADE_PATH}/upgrade_iso.sh rollback"
            result = SSH_API.exec_command_return_list(sudo_client, cmd)
            logger.info(f"Execute rollback script, result={str(result)}")
        except Exception as e:
            # 正常回滚，rollback会导致虚拟机重启，断开SSH连接，抛异常
            logger.info(f"Rollback after rebooting {str(e)} on {auth.host}")

        SSH_API.close_ssh(sudo_client)
        logger.info("Wait vm reboot, sleep 60s.")
        # 等待虚拟机重启
        time.sleep(60)
        self._wait_rollback(auth)
        # 再次检查是否回滚成功
        if not self._need_os_upgrade(auth):
            err_msg = "Current version is the latest version, rollback failed."
            logger.error(err_msg)
            raise HCCIException(663549, auth.host, err_msg)
        logger.info(f"Success rollback OS version on {auth.host}.")

    def _exe_upgrade_os(self, auth):
        logger.info(f"Start send upgrade os cmd to node ({auth.host}).")
        ssh_client = SSH_API.get_sudo_ssh_client(auth)
        if not self.check_os_image_mounted(ssh_client):
            self.mount_os_image(ssh_client)
        os_arch = self._get_os_arch(ssh_client)
        upgrade_cmd = f"bash -c 'sh {Path.ISO_UPGRADE_PATH}/upgrade_iso.sh upgrade " \
                      f"{Path.ISO_PKG_PATH}/OceanStor_BCManager_EulerOS_{os_arch}.iso {Path.OS_BAK_PATH} " \
                      f"1>>/var/log/upgrade_iso.log.`date +%s` 2>&1 &'"

        SSH_API.exec_command_return_list(ssh_client, upgrade_cmd)
        SSH_API.close_ssh(ssh_client)
        logger.info(f"Success send upgrade os cmd to node ({auth.host}).")

    def _get_current_state(self, ssh_client):
        logger.info("Get current state begin")
        if not self.check_os_image_mounted(ssh_client):
            self.mount_os_image(ssh_client)
        get_state_cmd = f"sh {Path.ISO_UPGRADE_PATH}/upgrade_iso.sh get_current_state"
        result = SSH_API.exec_command_return_list(ssh_client, get_state_cmd)
        for output in result:
            logger.info(f"Get current state, output:[{output}].")
            if output.strip() in OS.VALID_STATE_LIST:
                return output.strip(), ""
            if "|" in output:
                state = output.split("|")[0].strip()
                description = output.split("|")[1].strip()
                logger.info("Get current state end")
                return state, description
        raise HCCIException(663546, str(result), ssh_client.get('ip'), Path.ISO_UPGRADE_PATH)

    @staticmethod
    def _get_os_arch(ssh_client):
        arch_check_cmd = 'uname -m | grep -q x86_64;echo "x86type: $?"'
        ret_string = SSH_API.exec_command_return_list(ssh_client, arch_check_cmd)
        if str(ret_string).find("x86type: 0") != -1:
            arch = "X86"
        else:
            arch = "ARM"
        return arch

    def _need_os_upgrade(self, auth):
        ssh_client = None
        try:
            ssh_client = SSH_API.get_sudo_ssh_client(auth)
            euler_version = self.get_os_version(ssh_client)
            if euler_version == EulerUpgrade.LATEST_VERSION:
                logger.info(f"Check current version[{euler_version}] is already new version.")
                return False
            match_obj = re.search(r'eulerosv\d+r\d+', euler_version)
            if not match_obj:
                logger.info(f"Check current os[{euler_version}] is not euler os.")
                return False
            release_version = match_obj.group(0)
            if release_version not in EulerUpgrade.SUPPORT_VERSION:
                logger.info(f"Check current version[{release_version}] does not support upgrade.")
                return False
            # 与目标大版本相同，且当前小版本大于目标小版本，则不升级
            current_min_version = re.search(r'h\d+', euler_version)
            if release_version == EulerUpgrade.SUPPORT_VERSION[1]\
                    and int(str(current_min_version[0]).replace('h', '')) > EulerUpgrade.MIN_VERSION:
                logger.info(f"Check current version[{euler_version}] is later than the target version.")
                return False
            logger.info(f"Check version success. current version[{release_version}] can be upgraded.")
            return True
        except Exception as e:
            logger.error(f"Upgrade os failed! {str(e)}")
            raise HCCIException(663547, auth.host, str(e)) from e
        finally:
            SSH_API.close_ssh(ssh_client)

    def _os_prepare(self, auth):
        logger.info(f"Os prepare on {auth.host} begin.")
        ssh_client = None
        try:
            ssh_client = SSH_API.get_sudo_ssh_client(auth)
            os_arch = self._get_os_arch(ssh_client)
            pkg_path = self.get_upgrade_os_abs_pkg_path(os_arch)
            PKG_API.check_compressed_file(pkg_path, Capacity.MB_S, Pkg.UPGRADE_OS_PKG_SIZE_LIMIT,
                                          Pkg.UPGRADE_OS_PKG_FILE_COUNT)
            file_name = os.path.basename(pkg_path)
            logger.info(f"Get file[{file_name}] absolute path[{pkg_path}].")
            # 检查镜像是否挂载，若未挂载则挂载镜像
            if not self.check_os_image_mounted(ssh_client):
                logger.info(f"The image is not mounted to the node[{auth.host}],try to mount the image.")
                self.mount_os_image(ssh_client)
            # 执行镜像中内置的检查脚本，校验升级条件
            cmd = f"sh {Path.ISO_UPGRADE_PATH}/upgrade_iso.sh precheck " \
                  f"{Path.ISO_PKG_PATH}/OceanStor_BCManager_EulerOS_{os_arch}.iso {Path.OS_BAK_PATH}"
            SSH_API.exec_command_return_list(ssh_client, cmd)
            logger.info(f"Run the built-in check script in the image on"
                        f" node({auth.host}) to verify the upgrade conditions.")
            state, description = self._get_current_state(ssh_client)
            if state == 'SUCCESS':
                logger.info(f"Prepare os success on node({auth.host})")
            elif state == 'FAILED' and re.search(r"Current OS version\[.*\] is not compatible.", description):
                logger.info(f"The os version is not compatible with current iso on node({auth.host}). Skip upgrade.")
                self.add_node_ip_to_cfg(auth.host)
            elif state == 'FAILED' and "is already new version" in description:
                logger.info(f"The os version is already latest version on node({auth.host}). Skip upgrade.")
                self.add_node_ip_to_cfg(auth.host)
            else:
                logger.error(f"Prepare os failed. Error reason is {description} on {auth.host}.")
                raise HCCIException(663545, Path.ISO_UPGRADE_PATH, str(description))
        finally:
            SSH_API.close_ssh(ssh_client)

    @staticmethod
    def _remove_self_start_script(auth):
        """移除自动启文件/etc/init.d/lego"""
        logger.info(f"Remove self start script on node {auth.host} begin.")
        ssh_client = None
        try:
            # 移除自启动文件，升级时会重复覆盖
            # 备份数据工步已做备份，回退时可回滚
            ssh_client = SSH_API.get_sudo_ssh_client(auth)
            cmd = "rm -f /etc/init.d/lego;echo CMD_RESULT=$?"
            SSH_API.send_command(ssh_client, cmd, "CMD_RESULT=0")
        finally:
            SSH_API.close_ssh(ssh_client)
        logger.info(f"Remove self start script on node {auth.host} end.")

    def _os_upgrade(self, auth):
        logger.info(f"Os upgrade on node {auth.host} begin.")
        self._remove_self_start_script(auth)
        self._mount_bak_disk(auth)
        self._os_prepare(auth)
        self._exe_upgrade_os(auth)
        # 由于发送OS升级指令后，立马执行升级，防止查状态干扰升级，故等待5分钟后再查状态
        logger.info("Wait vm reboot, sleep 300s.")
        time.sleep(300)
        self._check_upgrade_state(auth)
        # 再次检查版本是否升级上去
        if self._need_os_upgrade(auth):
            err_msg = f"Current version is not the latest version, upgrade failed."
            logger.error(err_msg)
            raise HCCIException(663547, auth.host, err_msg)
        logger.info(f"Os upgrade on node {auth.host} end.")

    def _rollback_node_os(self, auth):
        logger.info(f"Execute rollback os on {auth.host} begin.")
        ssh_client = None
        try:
            sudo_client = SSH_API.get_sudo_ssh_client(auth)
            if not self.check_os_image_mounted(sudo_client):
                self.mount_os_image(sudo_client)
            # OS是最新版，需要回滚
            if not self._need_os_upgrade(auth):
                logger.info(f"Need execute rollback on node {auth.host}.")
                self._exe_rollback_node_os(auth)
            self._clean_temp_file(auth)
        except Exception as e:
            err_msg = f"Execute rollback os on {auth.host} failed. error: {str(e)}"
            logger.error(err_msg)
            raise HCCIException(663549, auth.host, err_msg)
        finally:
            SSH_API.close_ssh(ssh_client)
        logger.info(f"Execute rollback os on {auth.host} end.")

    def _wait_rollback(self, auth):
        # 回退10分钟
        timeout = 10 * 60
        _time = 0
        _interval = 2 * 60
        logger.info("Wait vm reboot, check rollback state.")
        while _time <= timeout:
            _time += _interval
            time.sleep(_interval)
            sudo_client = None
            try:
                sudo_client = SSH_API.get_sudo_ssh_client(auth)
                result = SSH_API.exec_command_return_list(sudo_client, 'uname -r')
                logger.info(f"Current os version: {result} on node: {auth.host}, count={_time}.")
                if self.get_os_version(sudo_client) != EulerUpgrade.LATEST_VERSION:
                    logger.info("OS rollback success.")
                    break
            except Exception as e:
                logger.warning(f"Check rollback state failed! error: {str(e)}")
            finally:
                SSH_API.close_ssh(sudo_client)
