import utils.common.log as logger
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil

from plugins.CSBS.common import model
from plugins.CSBS.common.constant import KARBOR_INSTALL_WORKDIR, COMPONENT
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient

logger.init("CSBS")


class KarborUtil(object):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.service_name = COMPONENT
        self.param_util = ParamUtil()
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.ssh_client = SshClient()

    def _get_karbor_ip_list_from_cloudparam(self):
        karbor_ip_list = []
        for ip_name in ["CSBS_Service1_ip", "CSBS_Service2_ip"]:
            karbor_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, ip_name)
            if not karbor_ip:
                raise Exception(f"Failed to obtain the ip of {ip_name}.")
            karbor_ip_list.append(karbor_ip)

        if not self.param_tool.is_csha_scene(self.project_id):
            karbor3_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Service3_ip")
            if not karbor3_ip:
                raise Exception("Failed to obtain the ip of CSBS_Service3_ip.")
            karbor_ip_list.append(karbor3_ip)
        return karbor_ip_list

    def get_csbs_node_info_from_cmdb(self):
        try:
            region_id = self.param_tool.get_region_id()
            nodes_info = ManageOneCmdbUtil(self.project_id, self.pod_id). \
                get_deploy_node_info(region_code=region_id,
                                     index_name=self.service_name)
            if not nodes_info:
                raise Exception("Failed to obtain csbs node info.")
            logger.info(f"Succeed to obtain {self.service_name} node info.")
            return nodes_info
        except Exception as err:
            logger.error(f"Failed to obtain csbs node info, err_msg:{err}.")
            raise Exception(f"Failed to obtain csbs node info, err_msg:{err}.") from err

    def _get_karbor_ip_list_from_cmdb(self):
        ip_list = []
        for node_info in self.get_csbs_node_info_from_cmdb():
            if not (node_info and node_info.get("name").
                    startswith("Service-CSBS")):
                continue
            for ip_dic in node_info.get("ipAddresses"):
                if ip_dic.get("netPlane") == "external_om":
                    ip_list.append(ip_dic.get("ip"))
        return ip_list

    def _get_karbor_ip_list_in_modification_con_ha_scene(self):
        karbor_ip_list = []
        for ip_name in ["CSBS_Service1", "CSBS_Service2", "CSBS_Service3"]:
            karbor_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, ip_name)
            if not karbor_ip:
                raise Exception(f"Failed to obtain the ip of {ip_name}.")
            karbor_ip_list.append(karbor_ip)
        return karbor_ip_list

    def get_karbor_ip_list(self):
        karbor_ip_list = []
        if self.param_tool.is_modification_region_con_ha(self.project_id):
            karbor_ip_list = \
                self._get_karbor_ip_list_in_modification_con_ha_scene()
        elif self.param_tool.is_install_csbs(self.project_id):
            karbor_ip_list = self._get_karbor_ip_list_from_cloudparam()
        elif self.param_tool.is_installed_csbs(self.project_id):
            karbor_ip_list = self._get_karbor_ip_list_from_cmdb()
        if not karbor_ip_list:
            raise Exception("Failed to obtain the karbor ips.")
        karbor_ip_list.sort()
        return karbor_ip_list

    def get_karbor_float_ip(self):
        float_ip = ""
        if self.param_tool.is_modification_region_con_ha(self.project_id):
            float_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Float_IP1")
        elif self.param_tool.is_install_csbs(self.project_id):
            float_ip = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "CSBS_Float_ip1")
        elif self.param_tool.is_installed_csbs(self.project_id):
            nodes_info = self.get_csbs_node_info_from_cmdb()
            for node_info in nodes_info:
                if node_info.get("name").startswith("Service-CSBS"):
                    float_ip = \
                        node_info.get("floatIpAddresses")[0].get("floatIp")
                    break
        if not float_ip:
            raise Exception("Failed to obtain the karbor float ip.")
        return str(float_ip)

    def get_karbor_internal_float_ip(self):
        float_ip = self.param_util.get_value_from_cloudparam(
            self.pod_id, self.service_name, "CSBS_Float_ip2")
        if not float_ip:
            raise Exception(f"Failed to obtain the karbor internal float ip.")
        return float_ip

    def get_karbor_user_info(self):
        karbor_user_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "karbor_user_name")
        if not karbor_user_name:
            karbor_user_name = "djmanager"

        karbor_root_name = self.param_util.get_param_value(
            self.pod_id, self.service_name, "CBS_root_account")
        if not karbor_root_name:
            karbor_root_name = "root"

        if self.param_tool.is_install_csbs(self.project_id) or \
                self.param_tool.is_modification_region_con_ha(self.project_id):
            user_pwd = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "karbor_user_passwd")
            if not user_pwd:
                raise Exception("Failed to obtain the pwd of djmanager.")

            root_pwd = self.param_util.get_value_from_cloudparam(
                self.pod_id, self.service_name, "karbor_root_passwd")
            if not root_pwd:
                raise Exception("Failed to obtain the pwd of karbor root.")
        else:
            user_pwd = self.param_util.get_param_value(
                self.pod_id, self.service_name, "karbor_user_passwd",
                "CSBS_VBS_server_node_user_pwd")
            if not user_pwd:
                raise Exception("Failed to obtain the pwd of djmanager.")

            root_pwd = self.param_util.get_param_value(
                self.pod_id, self.service_name, "karbor_root_passwd",
                "CSBS_VBS_server_node_root_pwd")
            if not root_pwd:
                raise Exception("Failed to obtain the pwd of karbor root.")
        return model.NodeUser(user_name=karbor_user_name,
                              user_pwd=user_pwd,
                              root_name=karbor_root_name,
                              root_pwd=root_pwd)

    def get_karbor_node_list(self):
        user_info = self.get_karbor_user_info()
        karbor_nodes_list = []
        for karbor_ip in self.get_karbor_ip_list():
            cbs_node = model.Node(
                node_ip=karbor_ip,
                user=user_info.user_name,
                user_pwd=user_info.user_pwd,
                root_name=user_info.root_name,
                root_pwd=user_info.root_pwd,
                extra="")
            karbor_nodes_list.append(cbs_node)
        return karbor_nodes_list

    def get_karbor_version(self):
        karbor_client = self.ssh_client.get_ssh_client(self.get_karbor_node_list()[0])
        karbor_version = dict()
        cmds_dict = {
            "short_version": "grep 'SHORT_VERSION' /opt/huawei/dj/cfg/dj.version | cut -d '=' -f 2",
            "full_version": "grep 'FULL_VERSION' /opt/huawei/dj/cfg/dj.version | cut -d ' ' -f 5"
        }
        for key, cmd in cmds_dict.items():
            result = self.ssh_client.ssh_exec_command_return(karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                raise Exception("Failed to execute cmd cat /opt/huawei/dj/cfg/dj.version.")
            karbor_version[key] = result[0]
        self.ssh_client.ssh_close(karbor_client)
        return karbor_version

    @staticmethod
    def get_installer_workdir():
        return KARBOR_INSTALL_WORKDIR

    def change_meter_switch(self, tar_status="on"):
        map_dict = {"on": "true", "off": "false"}
        if tar_status not in map_dict:
            raise Exception("The tar_status attr should be 'on' or 'off'.")

        logger.info(f"Ssh to karbor node and turn {tar_status} meter switch.")
        for node in self.get_karbor_node_list():
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"source /opt/huawei/dj/inst/utils.sh;set_features --support_sdr_meter {map_dict[tar_status]}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to turn {tar_status} meter switch on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to turn {tar_status} meter switch.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def change_cipher_type(self, cipher_type="generalCipher"):
        if cipher_type not in ("generalCipher", "SMCompatible", "SMOnly"):
            raise Exception("Unsupported cipher type.")

        logger.info("Ssh to karbor node and change cipher type.")
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"source /opt/huawei/dj/inst/utils.sh;switch_cipher_type --cipher_type {cipher_type}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to change cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info("Succeed to change cipher type.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def get_cipher_type(self):
        logger.info("Ssh to karbor node and get cipher type.")
        exec_result = None
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = "source /opt/huawei/dj/inst/utils.sh;get_cipher_type"
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

        result_str = "::".join(exec_result)
        if "generalCipher" in result_str:
            return "generalCipher"
        elif "SMCompatible" in result_str:
            return "SMCompatible"
        elif "SMOnly" in result_str:
            return "SMOnly"
        else:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")

    def set_karbor_endpoint(self, option, endpoint):
        logger.info(f"Ssh to karbor node and config karbor endpoint, "
                    f"option: {option}, endpoint: {endpoint}.")
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            cmd = f"source /opt/huawei/dj/inst/utils.sh;set_karbor_endpoints {option} {endpoint}"
            result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(result):
                logger.error(f"Failed to set_karbor_endpoint on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(f"Succeed to set_karbor_endpoint, "
                        f"option: {option}, endpoint: {endpoint}.")
            self.ssh_client.ssh_close(karbor_client)
            return True
        return False

    def set_karbor_account(self, account_name, account_pwd):
        logger.info(f"Ssh to karbor node and config karbor account, "
                    f"account name: {account_name}.")

        karbor_client = self.ssh_client.get_ssh_client(
            self.get_karbor_node_list()[0])
        self.ssh_client.ssh_send_command_expect(
            karbor_client, "source /opt/huawei/dj/inst/utils.sh;set_karbor_account", "user name", 30)
        self.ssh_client.ssh_send_command_expect(
            karbor_client, account_name, "password", 30)
        result = self.ssh_client.ssh_send_command_expect(
            karbor_client, account_pwd, "successfully", 30)
        logger.info(result)
        if self.ssh_client.failed_to_return(result, "successfully", karbor_client):
            raise Exception("Set karbor account failed.")
        return True

    def get_install_type(self):
        logger.info("Ssh to karbor node and get installtype.")
        exec_result = None
        cmd = "grep 'profile =' /opt/huawei/dj/cfg/sys.ini | cut -d '=' -f 2"
        for node in self.get_karbor_node_list()[:2]:
            karbor_client = self.ssh_client.get_ssh_client(node)
            exec_result = self.ssh_client.ssh_exec_command_return(
                karbor_client, cmd)
            if not self.ssh_client.is_ssh_cmd_executed(exec_result):
                logger.error(f"Failed to get cipher type on "
                             f"karbor node, node ip:{node.node_ip}.")
                self.ssh_client.ssh_close(karbor_client)
                continue
            logger.info(
                f"Succeed to execute command of '{cmd}' on karbor node.")
            self.ssh_client.ssh_close(karbor_client)
            break
        if not exec_result:
            raise Exception(
                "Failed to obtain the cipher type from karbor node.")
        logger.info(f"Succeed to get install type {exec_result[0].strip()} on karbor node.")
        return exec_result[0].strip()

    def get_hostname(self, ip_key):
        karbor_list = self.get_karbor_ip_list()
        # noinspection PyBroadException
        try:
            return f"karbor{(karbor_list.index(ip_key) + 1)}"
        except Exception:
            return f"karbor{(len(karbor_list) + 1)}"

    def get_karbor_ip_netmask(self):
        karbor_ip_netmask = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT,
                                                                      "CSBS_Service_subnet_mask")
        if karbor_ip_netmask:
            logger.info(f"Succeed to get_karbor_ip_netmask, {karbor_ip_netmask}")
            return karbor_ip_netmask
        raise Exception("Get karbor netmask failed.")
