import os
import zipfile

import utils.common.log as logger
from utils.DBAdapter.DBConnector import BaseOps
from utils.business.account_query import AccountQuery
from utils.business.dmk_util import DmkApi
from utils.business.dns_util import DNSApi, DNSNodeType
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.project_condition_utils import get_project_conditions
from utils.business.project_util import ProjectApi
from utils.common.exception import HCCIException
from utils.common.software_package_util import find_software_package_by_name
from utils.common.software_package_util import get_hcci_pkg_path
from utils.constant.path_constant import SOURCES_ROOT_DIR

from plugins.CSBS.common import constant
from plugins.CSBS.common import util
from plugins.CSBS.common.constant import COMPONENT

AUTH_TYPE = "iam"


class ParamTool(object):
    def __init__(self, project_id, pod_id):
        self.project_id = project_id
        self.pod_id = pod_id
        self.param_util = ParamUtil()
        self.mo_util = ManageOneUtil2()

    @staticmethod
    def is_csha_scene(project_id):
        replication_dict = ProjectApi().get_project_ha_dr_status(project_id)
        csha = replication_dict['CSHA']
        if replication_dict['RegionConHA'] == 1:
            csha = 1
        if not isinstance(csha, int):
            raise HCCIException(640067)
        return csha

    @staticmethod
    def get_network_name(project_id):
        network_name = "external_om"
        project_conditions = get_project_conditions(project_id)
        if project_conditions.get('UpgradedToHCS80') == 1:
            network_name = "Public_Service"
        return network_name

    @staticmethod
    def is_cpu_arm(project_id):
        return get_project_condition_boolean(project_id, 'manageARM | is_arm_scene')

    @staticmethod
    def is_expand_az(project_id):
        condition = "ExpansionAZ_KVM | ExpansionAZ_BMS"
        return get_project_condition_boolean(project_id, condition)

    @staticmethod
    def is_expand_ebackup_server(project_id):
        condition = "eBackup_ServerProxy_Driver | (eBackup & !eBackup_Proxy)"
        return get_project_condition_boolean(project_id, condition)

    @staticmethod
    def is_cshastorage(project_id):
        condition = "CSHAStorage"
        return get_project_condition_boolean(project_id, condition)

    @staticmethod
    def get_auth_type():
        return AUTH_TYPE

    @staticmethod
    def get_default_hcci_pkg_path():
        return get_hcci_pkg_path()

    @staticmethod
    def get_pkg_path(pkg_pre_name, pkg_suffix, pkg_version):
        pkg_path, pkg_name = find_software_package_by_name(pkg_pre_name=pkg_pre_name, pkg_post_name=pkg_suffix,
                                                           pkg_version=pkg_version)
        if not pkg_name:
            raise Exception("Can not find {}***{} pkg.".format(pkg_pre_name, pkg_suffix))
        pkg_path = os.path.join(pkg_path, pkg_name)
        return pkg_path

    @staticmethod
    def get_pkg_version(pkg_path):
        file_fs = None
        try:
            file_fs = zipfile.ZipFile(r'{}'.format(pkg_path), 'r')
            for file in file_fs.namelist():
                if file.endswith(".json"):
                    return file.split("-").__getitem__(1)
            raise Exception(f"Get package version failed, package name: {pkg_path}.")
        except Exception as err:
            raise Exception(f"Get package version failed, package name: {pkg_path}, err_msg: {str(err)}.") from err
        finally:
            if file_fs:
                file_fs.close()

    @staticmethod
    def is_install_csbs(project_id):
        project_conditions = get_project_conditions(project_id)
        return bool(project_conditions.get("InstallCSBS-VBS-Server"))

    @staticmethod
    def is_installed_csbs(project_id):
        project_conditions = get_project_conditions(project_id)
        return bool(project_conditions.get("InstalledCSBS-VBS")
                    or project_conditions.get("InstalledCSBSServer")
                    or project_conditions.get("BCInstalled"))

    @staticmethod
    def is_modification_region_con_ha(project_id):
        project_conditions = get_project_conditions(project_id)
        return bool(project_conditions.get("ProjectModifactionRegionConHA"))

    @staticmethod
    def get_project_plugins_path():
        return SOURCES_ROOT_DIR

    def get_installer_config(self):
        path_installer_config = "plugins/CSBS/scripts/deploy/karbor/sys.ini"
        return os.path.join(self.get_project_plugins_path(), path_installer_config)

    def get_install_type(self):
        return "private_recover" if self.is_csha_scene(self.project_id) else "private_kvm"

    def get_region_id(self):
        region_id = self.param_util.get_param_value(self.pod_id, "public", "region_id", "region0_id")
        if not region_id:
            raise Exception("Failed to obtain the current region_id.")
        return region_id

    def get_region_name(self):
        region_name = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "region_name")
        if not region_name:
            raise Exception("Failed to obtain the current region name.")
        return region_name

    def get_console_domain_name(self):
        console_domain_name = self.param_util.get_value_from_cloudparam(
            self.pod_id, COMPONENT, "console_domain_name")
        if not console_domain_name:
            raise Exception("Failed to obtain console_domain_name param.")
        return console_domain_name

    def get_cipher_type(self):
        project_conditions = get_project_conditions(self.project_id)
        if project_conditions.get("generalCipher"):
            return "generalCipher"
        elif project_conditions.get("SMCompatible"):
            return "SMCompatible"
        elif project_conditions.get("SMOnly"):
            return "SMOnly"
        else:
            raise Exception("Failed to obtain the cipher type.")

    def get_sc_url(self):
        _, sc_domain, _ = self.mo_util.get_mo_domain_name_list(self.project_id, self.pod_id)
        if not sc_domain:
            raise Exception("Failed to obtain sc domain.")
        util.check_url_param(sc_domain)
        logger.info("Succeed to obtain sc domain, "
                    f"the sc domain is: [{sc_domain}].")
        return f"https://{sc_domain}:26335"

    def get_oc_url(self):
        oc_domain, _, _ = self.mo_util.get_mo_domain_name_list(self.project_id, self.pod_id)
        if not oc_domain:
            raise Exception("Failed to obtain oc domain.")
        util.check_url_param(oc_domain)
        logger.info("Succeed to obtain oc domain, "
                    f"the oc domain is: [{oc_domain}].")
        return f"https://{oc_domain}:26335"

    def get_iam_url(self):
        _, _, iam_domain = self.mo_util.get_mo_domain_name_list(self.project_id, self.pod_id)
        if not iam_domain:
            raise Exception("Failed to obtain iam domain.")
        util.check_url_param(iam_domain)
        logger.info("Succeed to obtain iam domain, "
                    f"the iam domain is: [{iam_domain}].")
        return f"https://{iam_domain}:26335"

    def get_ip_version(self):
        if ProjectApi().is_ipv6_project(self.project_id):
            ip_version = 6
        else:
            ip_version = 4
        return ip_version

    def get_account_pwd_from_unify_pwd(self, region_id, account_list,
                                       used_scene=None, component_name=constant.CSBS_SERVICE_NAME,
                                       sub_component_name="CSBS-VBS", vm_ip=""):
        account_query = AccountQuery(self.project_id, self.pod_id)
        return account_query.get_account_info(
            region_id, account_list, used_scene, component_name, sub_component_name, vm_ip
        )

    def get_all_region_code(self):
        region_info_list = ManageOneCmdbUtil(self.project_id, self.pod_id).get_region_info()
        return [region_info.get('regionCode') for region_info in region_info_list if region_info.get('regionCode')]

    def get_dns_ip_list(self):
        logger.info('Use public interface DNSApi() to get DNS info')
        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        if dns_ips and len(dns_ips) == 2:
            return dns_ips
        raise Exception("Get dns ip failed.")

    def get_ntp_server_ip(self):
        return self.get_dns_ip_list()

    def get_syslog_server_ip(self):
        # syslog_server_ip
        logger.info("Use the public interface ManageOneUtil2() to get tanent_info")
        syslog_port = self.mo_util.getUDPPort(self.pod_id)
        syslog_server_ip = self.mo_util.getlvsIp(self.pod_id)
        if not syslog_server_ip:
            raise Exception("Get syslog server ip failed.")
        syslog_server_ip = syslog_server_ip + ":" + syslog_port
        if syslog_server_ip:
            return syslog_server_ip
        return ''

    def check_mo_om_and_tanent_info(self):
        mo_om_info = self.mo_util.getMoOmInfo(self.pod_id)
        # 该接口不同工程场景获取参数个数不同
        if len(mo_om_info) not in [4, 5] or not mo_om_info:
            logger.error("Get mo_om_info info failed, the length of mo_om_info is not 4 or 5,"
                         f" mo_om_info[{mo_om_info}]")
            return False
        tanent_info = self.mo_util.getMoTanentInfoDN(self.pod_id)
        if len(tanent_info) != 5 or not tanent_info:
            logger.error("Get tanent_info info failed, the length of tanent_info is not 5, "
                         f"tanent_info[{tanent_info}]")
            return False
        return True

    def get_karbor_timezone(self):
        timezone = self.param_util.get_value_from_cloudparam(self.pod_id, COMPONENT, "timezone")
        if timezone:
            logger.info(f"Succeed to get_karbor_timezone, {timezone}")
            return timezone
        raise Exception("Get karbor timezone failed.")

    def get_auth_url(self, cps_client):
        if self.get_auth_type() == 'keystone':
            auth_url = cps_client.get_keystone_endpoint()
        elif self.get_auth_type() == 'iam':
            auth_url = self.get_iam_url() + "/v3/"
        else:
            raise Exception("The auth type is not keystone or iam.")
        if not auth_url:
            raise Exception("Failed to obtain the auth url.")
        return auth_url

    def get_os_arch(self):
        return 'ARM' if self.is_cpu_arm(self.project_id) else 'X86'


class VmsName(object):
    KARBOR1 = 'Service-CSBS'
    KARBOR2 = 'Service-CSBS2'
    KARBOR3 = 'Service-CSBS3'
    KARBOR_HA1 = 'Service-CSBS'
    KARBOR_HA2 = 'Service-CSBSdr'


class Params(object):
    def __init__(self, project_id, pod_id, service_name="CSBS-VBS"):
        # step exec info
        self.project_id = project_id
        self.pod_id = pod_id
        # common setting when install
        self.db_api = BaseOps()
        self.cascad_type = self.db_api.get_pod_cascad_type(pod_id)
        self.project_info = self.db_api.get_project_by_id(project_id)

        # servcie info
        self.service_name = service_name
        self.dmk_service_name = "DMK"
        self.__db_param_dict = {}

    def get_params(self):
        if not self.__db_param_dict:
            self.init_params()
        return self.__db_param_dict

    def init_params(self):
        service_items = self.db_api.get_service_cloud_param(self.pod_id, self.service_name)

        self.__db_param_dict.update(service_items)

        dmk_items = DmkApi().get_dmk_info(self.pod_id)

        self.__db_param_dict.update(dmk_items)

        self.__db_param_dict['manage_CPU_arch'] = 'x86'
        is_cpu_arm = get_project_condition_boolean(self.project_id,
                                                   'manageARM')
        if is_cpu_arm:
            self.__db_param_dict['manage_CPU_arch'] = 'arm'

    @staticmethod
    def is_expansion_kvm_pod_or_az(project_id):
        db_api = BaseOps()
        project_info = db_api.get_project_by_id(project_id)
        return project_info.get('expansion_type') in ("ExpansionAZ_KVM",)
