import os
import shutil
import threading

import utils.common.log as logger
from utils.business.manageone_console_util import ManageOneConsoleUtil
from utils.business.manageone_util import ManageOneUtil
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message
from utils.common.software_package_util import find_software_package_by_name

from plugins.CSBS.common.params_tool import ParamTool, Params
from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.common.util import unzip_file

logger.init("CSBS")


class DeployCBSConsole(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(DeployCBSConsole, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.mo_util = ManageOneUtil()
        self.params = ParamUtil()
        self.region_id = \
            self.params.get_param_value(self.pod_id, "public",
                                        "region_id", "region0_id")
        self.param_tool = ParamTool(project_id, pod_id)
        self.db_param_dict = Params(project_id, pod_id).get_params()
        self.fcd_pkg_temp_dir = os.path.join(os.path.dirname(os.path.dirname(__file__)), "cbs_console")
        self.r_lock = threading.RLock()

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self._deploy_console()
        except HCCIException as err:
            logger.error("Execute error:{}.".format(str(err)))
            return Message(500, err)
        except Exception as err:
            logger.error("Execute error:{}.".format(str(err)))
            return Message(500, HCCIException(640082))
        finally:
            if os.path.exists(self.fcd_pkg_temp_dir):
                shutil.rmtree(self.fcd_pkg_temp_dir)

    def _deploy_console(self):
        pkg_name = self.upload_console_pkg()
        # 安装CBS Console
        logger.info('Begin to deploy CSBS-VBS console package.')
        # Parameter console_version is used to specify the prefix
        # of the static file on the frontend and distinguish
        # different region file.
        self.mo_util.deploy_console(self.pod_id, "", "", "", "", cloud_service_type="cbs", region_id=self.region_id,
                                    console_version=self.region_id, file_name=pkg_name)
        logger.info("Deploy CSBS-VBS console successfully.")
        return Message(200)

    def upload_console_pkg(self):
        # 上传console包
        logger.info('Begin to upload CSBS-VBS console, package.')
        pkg_path, file_name = find_software_package_by_name("OceanStor BCManager", "_CSBS-OceanProtect_Console.zip")
        if not file_name:
            raise HCCIException("640010")
        pkg = os.path.join(pkg_path, file_name)
        console_version = self.param_tool.get_pkg_version(pkg)
        if not console_version:
            raise Exception("Failed getting version from console's package.")
        if os.path.exists(self.fcd_pkg_temp_dir):
            shutil.rmtree(self.fcd_pkg_temp_dir)
        unzip_file(pkg, self.fcd_pkg_temp_dir, unzip_size_limit_mb=100, unzip_file_count_limit_kilo=10)
        pkg_name = "cbs-{version}.tar.gz".format(version=console_version)
        package_dir = "{}/{}".format(self.fcd_pkg_temp_dir, pkg_name)
        pkg_info = {"pkg_path": package_dir}
        mo_console_util = ManageOneConsoleUtil()
        mo_console_util.upload_static_pkg_to_static_node(self.pod_id, is_region=True, pkg_info=pkg_info,
                                                         check_pkg=False)
        # 上传完成后需要删除临时目录。
        if os.path.exists(self.fcd_pkg_temp_dir):
            shutil.rmtree(self.fcd_pkg_temp_dir)
        return pkg_name
