import utils.common.log as logger
from utils.business.install_csp_agent import CSPAgent
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.param_util import ParamUtil
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import COMPONENT
from plugins.CSBS.common.step_base import DeployBaseSubJob

logger.init("CSBS")


class InstallCspAgent(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(InstallCspAgent, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.cmdb_util = ManageOneCmdbUtil(project_id, pod_id)
        self.service_name = COMPONENT
        self.params = ParamUtil()
        self.region_id = \
            self.params.get_param_value(self.pod_id, "public",
                                        "region_id", "region0_id")
        self.user_name = "djmanager"
        self.user_pwd = \
            self.params.get_param_value(pod_id, self.service_name,
                                        "karbor_user_passwd",
                                        "CSBS_VBS_server_node_user_pwd")
        self.root_pwd = \
            self.params.get_param_value(pod_id, self.service_name,
                                        "karbor_root_passwd",
                                        "CSBS_VBS_server_node_root_pwd")

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            self._install_csp_agent()
        except HCCIException as err:
            logger.error("Failed to install csp agent, err_msg:{}.".format(str(err)))
            return Message(500, err)
        except Exception as err:
            logger.error("Failed to execute InstallCSPAgent, err_msg:{}.".format(str(err)))
            return Message(500, HCCIException(640086, str(err)))

        return Message(200)

    def _install_csp_agent(self):
        logger.info("Start to install csp agent for {} service.".format(self.service_name))
        karbor_node_list = self._get_karbor_node_list()
        ret_dic = CSPAgent(self.pod_id).install_csp_agent(service_name=self.service_name,
                                                          node_list=karbor_node_list)

        if ret_dic["result_code"] != "0":
            raise Exception("Install csp agent failed, result code:[{}].".format(ret_dic["result_code"]))
        logger.info("Succeed to install csp agent for {} service.".format(self.service_name))

    def _get_karbor_node_list(self):
        karbor_ip_list = self.get_service_ip_list_from_cmdb(self.region_id,
                                                            self.service_name)
        logger.info("Succeed to obtain karbor ips, "
                    "karbor ip list is:{}.".format(karbor_ip_list))
        node_list = []
        for karbor_ip in karbor_ip_list:
            node_info = {
                "node_ip": karbor_ip,
                "process_username": self.user_name,
                "user_password": self.user_pwd,
                "sudo_cmd": "sudo su",
                "root_password": self.root_pwd
            }
            node_list.append(node_info)
        return node_list

    def get_service_ip_list_from_cmdb(self, region_id, service_name):
        nodes_info = \
            self.cmdb_util.get_deploy_node_info(region_code=region_id,
                                                index_name=service_name)
        logger.info("Get {} service nodes info: {}.".format(service_name,
                                                            nodes_info))
        ip_list = []
        for node_info in nodes_info:
            if not (node_info and node_info.get("name").
                    startswith("Service-CSBS")):
                continue
            for ip_dic in node_info.get("ipAddresses"):
                if ip_dic.get("netPlane") == "external_om":
                    ip_list.append(ip_dic.get("ip"))
        return ip_list
