import datetime
import re

import utils.common.log as logger
from utils.common.ssh_util2 import Ssh

# 命令返回的标准长度
SSH_RET_LENGTH = 7
EXCEPT_CODE = -3
ERROR_CODE = -2
COMMON_CODE = 0
# 最大天数
MAX_DAY_NUM = 999
EXPIRING_DAY = 7


class SShCmd:
    """
    proxy
    """

    def __init__(self):
        self.ssh_obj = Ssh()

    def ssh_cmds(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        result = self.ssh_obj.ssh_cmds(
            obj.node_ip,
            cmd,
            obj.user,
            obj.user_pwd,
            obj.root_pwd,
            "",
            ""
        )
        self.parsing_result(expect_value, result, expect)

    def ssh_cmds_to_proxy(self, node, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        result = self.ssh_obj.ssh_cmds(
            node.node_ip,
            cmd,
            node.user,
            node.user_pwd,
            node.root_pwd,
            "",
            ""
        )
        self.parsing_result(expect_value, result, expect)

    @staticmethod
    def get_exe_cmd(cmd):
        result_str = None
        try:
            if ';' in cmd:
                exe_result_cmd = cmd.split(';')[-1]
                match_obj = re.search(
                    r'echo(.*)\$\?', exe_result_cmd, re.M | re.I)
                if match_obj:
                    result_str = exe_result_cmd.replace('echo ', '')
                    result_str = result_str.replace('$?', '0')
        except Exception as err:
            logger.info(str(err))
        if not result_str:
            cmd = cmd + '; echo exe cmd result: $?'
            result_str = 'exe cmd result: 0'
        return cmd, result_str

    @staticmethod
    def exe_user_to_root(node):
        ssh_client = Ssh.ssh_create_client(
            node.node_ip,
            node.user,
            node.user_pwd
        )
        Ssh.ssh_send_command(ssh_client, "su -", 'Password', 100)
        Ssh.ssh_send_command(ssh_client, node.root_pwd, '#', 100)
        return ssh_client

    def ssh_exe_config_ha_cmd(self, agentproxy, cmd, passwd):
        for proxy_node in [agentproxy.get_proxy_node_list()[0]]:
            logger.info(f'Starting to connect to {proxy_node.node_ip} .')
            ssh_client = self.exe_user_to_root(proxy_node)
            Ssh.ssh_send_command(ssh_client, cmd, 'password', 100)
            Ssh.ssh_send_command(ssh_client, passwd, 'again', 100)
            Ssh.ssh_send_command(ssh_client, passwd, 'success', 100)
            Ssh.ssh_close(ssh_client)
            logger.info(
                f'The {proxy_node.node_ip} congfig ha account information is configured '
                'successfully.')

    def parsing_result(self, expect_value, result, expect=None):
        if expect:
            if isinstance(expect, str):
                expect_value = expect

            if isinstance(expect, list):
                expect = [expect_value.replace('0', x).strip() for x in expect]
                if set(expect) & set(result):
                    logger.info(str(set(expect) & set(result)))
                    self.success_result(expect_value)
                    return
                else:
                    self.fail_result(expect_value)

        if expect_value in result:
            self.success_result(expect_value)
        else:
            self.fail_result(expect_value)

    @staticmethod
    def success_result(expect_value):
        result_str = expect_value.replace('0', '')
        result_str = result_str + ': success'.strip()
        logger.info(result_str)

    @staticmethod
    def fail_result(expect_value):
        result_str = expect_value.replace('0', '')
        result_str = result_str + ': fail'.strip()
        logger.error(result_str)
        raise Exception(result_str)

    def ssh_cmds_to_all_proxys(self, obj, cmd, expect=None):
        cmd, expect_value = self.get_exe_cmd(cmd)
        for node in obj.get_proxy_node_list():
            result = self.ssh_obj.ssh_cmds(
                node.node_ip,
                cmd,
                node.user,
                node.user_pwd,
                node.root_pwd,
                "",
                ""
            )
            self.parsing_result(expect_value, result, expect)

    def check_node_password_expired(self, node):
        try:
            self._is_node_alive(node)
        except Exception as err:
            logger.error(err)
            raise Exception('Failed to log in to the upper node. '
                            'Check whether the account has expired'
                            ' or the password is incorrect.', node.node_ip) from err
        ssh_client = self.get_server_ssh_client(
            node.node_ip, ssh_user=node.user, ssh_pwd=node.user_pwd,
            sudo_user=node.root_name, sudo_pwd=node.root_pwd)

        return_data = self.check_os_password_expired_day(
            ssh_client, [node.user, node.root_name])
        hcp_expire_day = return_data.get(node.user)
        root_expire_day = return_data.get(node.root_name)
        if hcp_expire_day <= 30 or root_expire_day <= 30:
            raise Exception('The node password is about to expire or has'
                            ' expired. Please change it.', node.node_ip)
        logger.info(f'Node {node.node_ip} Passes the Password Expiration Check ')

    def _is_node_alive(self, node):
        ssh_client = self.ssh_obj.ssh_create_client(node.node_ip, node.user, node.user_pwd)
        self.ssh_obj.ssh_close(ssh_client)

    @staticmethod
    def get_server_ssh_client(host, ssh_user=None, ssh_pwd=None,
                              sudo_user=None, sudo_pwd=None):
        ssh_ins = Ssh()
        ssh_client = None
        if ssh_user and ssh_pwd:
            ssh_client = ssh_ins.ssh_create_client(
                host, ssh_user, ssh_pwd, port=22, timeout=60)
            if sudo_user and sudo_pwd:
                ssh_ins.ssh_send_command(
                    ssh_client, "sudo su - {}".format(
                        sudo_user), "password", 20)
                ssh_ins.ssh_send_command(ssh_client, sudo_pwd, "#", 20)
                ssh_ins.ssh_send_command(ssh_client, "TMOUT=0", "#", 20)
            return ssh_client
        logger.error("The user name or password required for "
                     "login is missing.")
        return ssh_client

    def check_os_password_expired_day(self, ssh_client, user_lst):
        return_data = {}
        for user in user_lst:
            cmd = "chage -l {}".format(user)
            output = Ssh.ssh_send_command(ssh_client, cmd, "#", 20)
            output.pop(0)
            output.pop(-1)
            logger.info(
                "Execute command success, result: {}.".format(output))
            if not output or len(output) < SSH_RET_LENGTH:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            code, expires, inactive = self.get_password_expires(
                "".join(output))
            logger.info(
                "Ended to check {} user, result: {}, {}, {}.".format(
                    user, code, expires, inactive))
            if code == ERROR_CODE:
                err_msg = "Execution command did not get the expected " \
                          "output, please check."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == EXCEPT_CODE:
                err_msg = \
                    "Analysis command output error, please try again."
                logger.error(err_msg)
                raise Exception(err_msg)
            elif code == 0:
                expire_days = (expires - datetime.datetime.utcnow()).days
                logger.info("The password of the {} user expires "
                            "after {} days.".format(user, expire_days))
                if not expire_days:
                    raise Exception(
                        "Get user: {} expire days failed.".format(user))
                return_data[user] = expire_days
        Ssh.ssh_close(ssh_client)
        return return_data

    @staticmethod
    def get_password_expires(output_list):
        """
        获取密码的过期时长和失效时间
        :param output_list: SSH命令返回结果
        :return:
        """
        # 对输出进行分行处理
        expired_time, inactive_time, expired_str = "", "", ""
        outputs = output_list.splitlines(False)
        if len(output_list) < SSH_RET_LENGTH:
            # 执行命令结果长度不够，错误
            raise Exception(f"Insufficient length of ssh cmd return val, error code:{ERROR_CODE}")
        try:
            for out_put_line in outputs:
                temp_name = out_put_line.split(":")[0].strip()
                temp_value = out_put_line.split(":")[1].strip()
                if temp_name == 'Password expires':
                    expired_str = temp_value
                    expired_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if expired_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
                elif temp_name == 'Password inactive':
                    inactive_str = temp_value
                    inactive_time = datetime.datetime.utcnow() + datetime.timedelta(days=MAX_DAY_NUM) \
                        if inactive_str == 'never' else datetime.datetime.strptime(expired_str, '%b %d, %Y')
        except Exception as error:
            # 执行命令结果解析，错误
            logger.error(f"Failed to parse ssh command results. Error Message: {error}")
            raise Exception(
                f"Failed to parse ssh command results, except code: {EXCEPT_CODE}. Error Message: {error}") from error
        ret_code = COMMON_CODE
        if (not expired_time) or (not inactive_time):
            ret_code = ERROR_CODE
        return ret_code, expired_time, inactive_time
