# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import os
import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.software_package_util import find_software_package_by_name

from plugins.CSBS.common import model
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.util import check_zipfile, ConfigIniFile, unzip_file

VALID_STATE_LIST = ["PRE_UPGRADING", "UPGRADING", "POST_UPGRADING", "PRE_CHECKING", "ROLLBACKING", "SUCCESS"]
UPGRADE_CONFIG_KEY = "UpgradeOs"
SKIP_NODE_IP = "skip_ip_list"

logger.init("OS Util")


class OSUtil:
    def __init__(self, project_id, pod_id, node):
        self.ssh_util = SshClient()
        self.project_id = project_id
        self.node: model.Node = node
        self.params_tools = ParamsTools(project_id)
        self.param_tool = ParamTool(project_id, pod_id)
        self.os_arch = self.param_tool.get_os_arch()
        self.iso_file_name = f"OceanStor_BCManager_EulerOS_{self.os_arch}.iso"

    @staticmethod
    def get_kvm_image_file(project_id):
        is_cpu_arm = ParamTool.is_cpu_arm(project_id)
        if is_cpu_arm:
            os_template = OSUtil.get_os_file("OceanStor_BCManager_Euler_ARM", "KVMtemplate_Euler_ARM.zip")
        else:
            os_template = OSUtil.get_os_file("OceanStor_BCManager_Euler_X86", "KVMtemplate_Euler_X86.zip")

        file_name = os_template.replace(".qcow2", "")
        return file_name

    @staticmethod
    def get_os_file(os_file_name, os_zip_file_name):
        default_hcci_pkg_path = ParamTool.get_default_hcci_pkg_path()
        pkg_path, pkg_name = find_software_package_by_name(os_file_name, ".qcow2")

        if (pkg_path != default_hcci_pkg_path and pkg_name) or (not pkg_name):
            zip_file_path, zip_file_name = find_software_package_by_name("OceanStor BCManager", os_zip_file_name)
            if not zip_file_name or not zip_file_path:
                raise HCCIException(640002, f"OceanStor BCManager***{os_zip_file_name}")
            file_path = os.path.join(zip_file_path, zip_file_name)
            logger.info(f'Current os image zip:{file_path}.')

            unzip_file(file_path, default_hcci_pkg_path, unzip_size_limit_mb=5120, unzip_file_count_limit_kilo=10)
            _, pkg_name = find_software_package_by_name(os_file_name, ".qcow2")
            if not pkg_name or not os.path.exists(os.path.join(default_hcci_pkg_path, pkg_name)):
                raise HCCIException(640002, f"{os_file_name}.qcow2")
            logger.info(f'Current os image qcow2:{pkg_name}.')

        # 有可能文件已经生成，但是还未完全解压，做个延时
        time.sleep(10)

        return pkg_name

    def check_iso_mounted(self, ssh_client) -> bool:
        """
        检查镜像是否已经挂载 已挂载： True 未挂载： False
        """
        check_mount_cmd = f"mount | grep -v deleted | grep -qP '/root/{self.iso_file_name}" \
                          f".*/mnt/upgrade '; echo \"mounted: $?\""
        ret_string = SshClient.ssh_exec_command_return(ssh_client, check_mount_cmd)
        return str(ret_string).find("mounted: 0") > -1

    def get_os_package_name(self, arch):
        if arch == "X86":
            package_post_name = "EulerOS_X86.zip"
        else:
            package_post_name = "EulerOS_ARM.zip"

        path, name = find_software_package_by_name(
            pkg_pre_name="OceanStor",
            pkg_post_name=package_post_name,
            project_id=self.project_id)
        file_path = os.path.join(path, name)
        check_zipfile([file_path])
        return file_path, name

    def upload_iso(self):
        file_path, file_name = self.get_os_package_name(self.os_arch)
        self.ssh_util.put(self.node.node_ip, self.node.user, self.node.user_pwd, file_path,
                          f"/home/{self.node.user}/")

    def mount_iso(self, ssh_client):
        self.upload_iso()
        file_path, file_name = self.get_os_package_name(self.os_arch)
        result = self.ssh_util.ssh_exec_command_return(ssh_client,
                                                       f"unzip -o /home/{self.node.user}/'{file_name}' -d /root")
        if not self.ssh_util.is_ssh_cmd_executed(result):
            raise HCCIException(650076, f"Unzip failed! {str(result)}")
        result = self.ssh_util.ssh_exec_command_return(ssh_client, "test -e /mnt/upgrade")
        if not self.ssh_util.is_ssh_cmd_executed(result):
            self.ssh_util.ssh_exec_command_return(ssh_client, "mkdir -p /mnt/upgrade")
        mount_cmd = f"mount -t iso9660 -o nosuid,noexec,nodev /root/{self.iso_file_name} /mnt/upgrade"
        ret_string = self.ssh_util.ssh_exec_command_return(ssh_client, mount_cmd)
        if not self.ssh_util.is_ssh_cmd_executed(ret_string):
            raise HCCIException(650076, str(ret_string))

    def clean_all(self, ssh_client):
        logger.info(f"Start to clean file on node ({self.node.node_ip}).")
        if not self.check_iso_mounted(ssh_client):
            self.mount_iso(ssh_client)
        file_path, file_name = self.get_os_package_name(self.os_arch)
        clean_cmd = f"sh /mnt/upgrade/upgrade_iso.sh clean; " \
                    f"umount /mnt/upgrade;rm -rf /var/log/upgrade_iso.log.*;" \
                    f"rm -rf '/home/{self.node.user}/{file_name}';rm -rf /root/{self.iso_file_name}"
        self.ssh_util.ssh_exec_command_return(ssh_client, clean_cmd)
        rm_cmd = f"rm -rf /mnt/upgrade /mnt/upgrade_iso /root/OS_UPGRADE_STATE /root/OS_ROLLBACK_STATE"
        self.ssh_util.ssh_exec_command_return(ssh_client, rm_cmd)

    def get_current_state(self, ssh_client):
        if not self.check_iso_mounted(ssh_client):
            self.mount_iso(ssh_client)
        get_state_cmd = "sh /mnt/upgrade/upgrade_iso.sh get_current_state"
        ret_string = SshClient.ssh_exec_command_return(ssh_client, get_state_cmd, timeout=30)
        for output in ret_string:
            if output.strip() in VALID_STATE_LIST:
                return output.strip(), ""
            if "|" in output:
                description = output.split("|")[1].strip()
                state = output.split("|")[0].strip()
                return state, description
        raise HCCIException(650077, str(ret_string), ssh_client.get('ip'))

    def add_node_ip_to_cfg(self, node_ip):
        env_config_path = self.params_tools.get_env_config_path()
        conf_env_file = ConfigIniFile(env_config_path)
        skip_ip = conf_env_file.get_value_by_key_and_sub_key(UPGRADE_CONFIG_KEY, SKIP_NODE_IP) or ""
        ip_list = skip_ip.split(',')
        if node_ip not in ip_list:
            ip_list.append(node_ip)
        ip_list = list(filter(None, ip_list))
        conf_env_file.set_value_by_key_and_sub_key(UPGRADE_CONFIG_KEY, SKIP_NODE_IP, ",".join(ip_list))

    def need_to_upgrade(self):
        env_config_path = self.params_tools.get_env_config_path()
        conf_env_file = ConfigIniFile(env_config_path)
        skip_ip = conf_env_file.get_value_by_key_and_sub_key(UPGRADE_CONFIG_KEY, SKIP_NODE_IP) or ""
        ip_list = skip_ip.split(',')
        return self.node.node_ip not in ip_list
