import utils.common.log as logger
from utils.common.code2Msg import code2Msg

from plugins.CSBS.common import model
from plugins.CSBS.common.param_check.os_param_check import check_pwd_will_expired
from plugins.CSBS.common.dpa_login_check import DPALoginCheck
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karborproxy import KarborProxyOperation


class ProxyParamsCheck(object):
    def __init__(self, project_id):
        self.project_id = project_id
        self.ssh_client = SshClient()
        self.proxy_operation = KarborProxyOperation(self.project_id)
        self.proxy_ip_list = self.proxy_operation.all_ip_list
        self.proxy_user = self.proxy_operation.root_name
        self.proxy_pwd = self.proxy_operation.root_pwd
        self.proxy_root_pwd = self.proxy_operation.get_karborproxy_user_info().root_pwd
        self.dpa_login_check = DPALoginCheck(project_id)

    def check_proxy_login_ability(self, check_result):
        check_result = self.check_proxy_dpamanager_login_ability(check_result)

        check_result = self.check_proxy_root_login_ability(check_result)
        return check_result

    def check_proxy_dpamanager_login_ability(self, check_result):
        dpamanager_pwd_will_expired = []
        dpamanager_pwd_no_ssh_ips = []

        check_list = ["csbs_karborproxy_dpamanager_password"]

        def _get_dpamanager_login_check_result():
            success_count = 0
            for proxy_ip in self.proxy_ip_list:
                ssh_client = None
                pwd_expired_flag = None
                try:
                    ssh_client = self.ssh_client.create_ssh_client(proxy_ip, self.proxy_user, self.proxy_pwd)
                except Exception as err:
                    logger.error(f"Failed to login to Proxy node: IP {proxy_ip}, User dpamanager, error:{str(err)}")
                    dpamanager_pwd_no_ssh_ips.append(proxy_ip)
                finally:
                    if ssh_client:
                        pwd_expired_flag = check_pwd_will_expired(self.ssh_client, ssh_client,
                                                                  "dpamanager")
                        self.ssh_client.ssh_close(ssh_client)
                if pwd_expired_flag:
                    logger.error(f"Node {proxy_ip}, User {self.proxy_user} password will expire, "
                                 f"please change and retry.")
                    dpamanager_pwd_will_expired.append(proxy_ip)
                else:
                    success_count += 1
            return success_count

        check_success_count = _get_dpamanager_login_check_result()
        if check_success_count == len(self.proxy_ip_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            if dpamanager_pwd_will_expired:
                logger.error(f"Node IP {','.join(dpamanager_pwd_will_expired)}, "
                             "User dpamanager password will expire.")
                error_msg = code2Msg(645056)
            else:
                logger.error(f"Failed to login to Proxy node: IP {','.join(dpamanager_pwd_no_ssh_ips)}, "
                             "User dpamanager.")
                error_msg = code2Msg(642001)
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=error_msg)
        return check_result

    def check_proxy_root_login_ability(self, check_result):
        check_list = ["csbs_karborproxy_root_password"]
        check_success_count = 0
        proxy_root_no_ssh_ips = []
        for proxy_ip in self.proxy_ip_list:
            ssh_client = None
            try:
                ssh_client = self._check_root_login(proxy_ip)
            except Exception as err:
                logger.error(
                    "Failed to login to Proxy node: IP {}, User {}, "
                    "error: {}".format(proxy_ip, 'root', str(err))
                )
                proxy_root_no_ssh_ips.append(proxy_ip)
            finally:
                if ssh_client:
                    self.ssh_client.ssh_close(ssh_client)
            check_success_count += 1

        if check_success_count == len(self.proxy_ip_list):
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        else:
            logger.error(
                "Failed to login to Proxy node: IP {}, User {}".format(
                    ','.join(proxy_root_no_ssh_ips), "root")
            )
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(642001))
        return check_result

    def _check_root_login(self, _ip):
        proxy_node = model.Node(
            node_ip=_ip,
            user_pwd=self.proxy_pwd,
            user=self.proxy_user,
            root_name='root',
            root_pwd=self.proxy_root_pwd
        )
        return self.ssh_client.get_ssh_client_user_su_root(proxy_node)

    def check_dpa_login_ability(self, check_result):
        check_list = ['DPA_Business_Address', 'DPA_Cluster_Address', 'DPA_Admin', 'DPA_Admin_Password', 'DPA_Sec_Admin',
                      'DPA_Sec_Admin_Password', 'DPA_General_User', 'DPA_General_User_Password']
        try:
            self.dpa_login_check.check_account_info()
            check_result.set_check_result(param_keys=check_list, status=200, error_msg='')
        except Exception as exception:
            logger.error(f"Failed to login to dpa, exception: {str(exception)}")
            check_result.set_check_result(param_keys=check_list, status=500, error_msg=code2Msg(645060))
        return check_result
