import ipaddress
import re

import IPy
import utils.common.log as logger
from utils.business.project_condition_utils import get_project_conditions, get_project_condition_boolean,\
    is_expansion_kvm_pod_or_az
from utils.business.project_util import ProjectApi
from utils.common.error.hcci_error_code import get_code_msg
from utils.common.param_check import check_param_string, check_param_integer, check_param_ip, is_netmask2, \
    check_param_netmask, check_param_email, check_param_password, ping

from plugins.CSBS.common.util import check_openstack_region
from plugins.CSBS.scripts.deploy.karborproxy.common.funtions import check_url_list

logger.init("CSBS")

MANAGE_DATA_PARAMS = {
    'system_backup_sftp_dir': {'type': 'super_path', 'suggest': 640076},
    'system_backup_sftp_ip': {'type': 'ip', 'suggest': 640060},
    'system_backup_sftp_username': {'type': 'super_string', 'suggest': 640057},
    'system_backup_sftp_password': {'type': 'super_string', 'suggest': 640057},
    'system_backup_sftp_port': {'type': 'integer', 'suggest': 640064}
}

SERVERLIST_PARAMS_EXPANSION_AZ = {
    'az_related_to_datamover': {'type': 'super_string', 'suggest': 640057}
}

SERVERLIST_LLD_IP_DICT = {
    'CSBS_Float_IP1': {'type': "ip", 'suggest': 640060},
    'CSBS_Float_IP2': {'type': "ip", 'suggest': 640060},
    'CSBS_Service1': {'type': "ip", 'suggest': 640060},
    'CSBS_Service2': {'type': "ip", 'suggest': 640060},
    'CSBS_Service3': {'type': "ip", 'suggest': 640060}
}

# 跨az高可用场景使用参数
SERVERLIST_PARAMS_CROSS_AZ_HA = {
    'Arbitration_DC1_01_API': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC1_01_OM': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC1_02_API': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC1_02_OM': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC2_01_API': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC2_01_OM': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC2_02_API': {'type': "ip", 'suggest': 640060},
    'Arbitration_DC2_02_OM': {'type': "ip", 'suggest': 640060},
    'Third_site_IP_of_Arbitration_Service': {'type': "ip", 'suggest': 640060},
    'arb_admin_password': {'type': 'super_string', 'suggest': 640057},
    'arb_business_password': {'type': 'super_string', 'suggest': 640057},
}

SERVERLIST_PARAMS_DICT_ALL = {
    # add in FCD650
    'oc_external_global_domain_name': {'type': 'super_string',
                                       'suggest': 640057},
    'moTenantconsoleIp': {'type': "ip", 'suggest': 640060},
    'moTenant01Ip': {'type': "ip", 'suggest': 640060},
    'moTenant02Ip': {'type': "ip", 'suggest': 640060},
    'iam_nginx_domain_name_prefix': {'type': 'super_string',
                                     'suggest': 640057},
    'shubao_lb_float_ip': {'type': "ip", 'suggest': 640060},
    'apigateway_dmk_password': {'type': 'super_string', 'suggest': 640057},
    'ossusrPsd': {'type': 'super_string', 'suggest': 640057},
    'primary_region_dmk_dns_password': {'type': 'super_string',
                                        'suggest': 640057},  # sub_region

    'time_zone': {'type': 'super_string', 'suggest': 640057},
    'region0_id': {'type': 'super_string', 'suggest': 640057},
    'region_display_name': {'type': 'super_string', 'suggest': 640057},
    'openstack_region': {'type': 'super_string', 'suggest': 640057},
    'openstack_domain': {'type': 'super_string', 'suggest': 640057},
    'openstack_reverse_proxy_ip': {'type': "ip", 'suggest': 640060},
    'openstack_root_password': {'type': 'super_string', 'suggest': 640057},
    'openstack_fsp_password': {'type': 'super_string', 'suggest': 640057},
    'global_domain_name': {'type': 'super_string', 'suggest': 640057},
    'external_global_domain_name': {'type': 'super_string', 'suggest': 640057},
    'tenant_domain_name': {'type': 'super_string', 'suggest': 640057},
    'dmk_floatIp': {'type': "ip", 'suggest': 640060},
    'dmk_ui_password': {'type': 'super_string', 'suggest': 640057},
    'dmk_os_business_user_password': {'type': 'super_string',
                                      'suggest': 640057},
    'current_region_dmk_dns_password': {'type': 'super_string',
                                        'suggest': 640057},
    'moDeployFloatIp': {'type': "ip", 'suggest': 640060},
    'mo.webdeploypassword': {'type': 'super_string', 'suggest': 640057},
    'moOMFloatIp': {'type': "ip", 'suggest': 640060},
    'MoOMWebNewPassword': {'type': 'super_string', 'suggest': 640057},
    'moTenantFloatIp': {'type': "ip", 'suggest': 640060},
    'tanent.webDefaultPassword': {'type': 'super_string', 'suggest': 640057},
    'moService01Ip': {'type': "ip", 'suggest': 640060},
    'lvsIp': {'type': "ip", 'suggest': 640060},
    'sopuserPasswd': {'type': 'super_string', 'suggest': 640057},
    'rootPassword': {'type': 'super_string', 'suggest': 640057},
    'ossadmPsd': {'type': 'super_string', 'suggest': 640057},
    'global_LVS_dmk_password': {'type': 'super_string', 'suggest': 640057},
    'az_related_to_datamover': {'type': 'super_string', 'suggest': 640057},
    'global_nginx_dmk_password': {'type': 'super_string', 'suggest': 640057},
    'dmk_primaryIp': {'type': "ip", 'suggest': 640060},
    'dmk_standbyIp': {'type': "ip", 'suggest': 640060}
}

# B2B scene param
SERVERLIST_PARAMS_B2B_SCENE = {
    'iam_manage_nginx_domain_name_prefix': {'type': 'super_string',
                                            'suggest': 640057},
}


def params_check(**kwargs):
    """插件参数校验接口函数
    :param kwargs:
    :return: 成功：True, ''， 失败：False，错误信息，格式：{key1:err_msg1, key2:err_msg2}
    """
    project_id = kwargs.get("params").get("project_id")
    # 判断是否为ipv6场景
    ip_version = 6 if ProjectApi().is_ipv6_project(project_id) else 4
    try:
        is_expansion_az_or_pod = is_expansion_kvm_pod_or_az(project_id)
    except Exception:
        logger.info("fail to get projectid")
        return False, 'fail to get is_expansion_kvm_pod_or_az'

    serverlist_params_dict, error_msg_dict1 = _get_serverlist_params_dict(project_id, kwargs, is_expansion_az_or_pod)

    error_msg_dict2 = _check_serverlist_params(serverlist_params_dict, kwargs, ip_version)

    error_msg_dict3, csbs_float_ip_dict = _check_lld_ip_list(kwargs, ip_version)

    # Check whether the entered IP address conflicts with other IP addresses in the LLD.
    error_msg_dict = check_ip_conflict(kwargs.get("params"), SERVERLIST_LLD_IP_DICT,
                                       error_msg_dict1 | error_msg_dict2 | error_msg_dict3)

    if not error_msg_dict:
        # lld表格支持用户添加删除表格行, 无相应数据行, 参数不校验
        if list(csbs_float_ip_dict.values()).count(None) == 0:
            if not check_float_ip_start_end(
                    csbs_float_ip_dict.get('CSBS_Float_IP1'),
                    csbs_float_ip_dict.get('CSBS_Float_IP2')):
                msg = get_code_msg(640079)
                error_msg_dict.update({'CSBS_Float_IP1': msg})
                logger.info(f"check params result: {error_msg_dict}")
                return False, error_msg_dict
        logger.info("check params result: PASS")
        return True, ''
    else:
        logger.info(f"check params result: {error_msg_dict}")
        return False, error_msg_dict


def _get_serverlist_params_dict(project_id, kwargs, is_expansion_az_or_pod):
    error_msg_dict = {}
    # 这里得到的是一个列表，列表里存的是字典。
    got_params = kwargs["params"]["param_list"]
    if get_project_condition_boolean(project_id, "InstallCSBS-VBS-Proxy"):
        all_dpa_info = get_dpa_business_cluster_account_info(got_params)
        dpa_params_check = DpaParamCheck(all_dpa_info, project_id)
        if dpa_params_check.all_err_msg:
            error_msg_dict.update(dpa_params_check.all_err_msg)
    if is_expansion_az_or_pod == 1:
        serverlist_params_dict = SERVERLIST_PARAMS_EXPANSION_AZ
    else:
        serverlist_params_dict = SERVERLIST_PARAMS_DICT_ALL

    # 判断是否为跨az高可用场景
    project_ha_dr_status_dict = ProjectApi().get_project_ha_dr_status(
        project_id)
    is_ha = project_ha_dr_status_dict.get('RegionConHA', 0)
    if is_ha:
        serverlist_params_dict.update(SERVERLIST_PARAMS_CROSS_AZ_HA)

    # verify system_backup params when ManageSysBackup is true
    project_conditions = get_project_conditions(project_id)
    is_msb = project_conditions.get('ManageSysBackup', 0)
    if is_msb:
        serverlist_params_dict.update(MANAGE_DATA_PARAMS)

    # 检查是否包含b2b必填参数
    if check_content_b2b_param(kwargs["params"]["param_list"]):
        serverlist_params_dict.update(SERVERLIST_PARAMS_B2B_SCENE)

    return serverlist_params_dict, error_msg_dict


def _check_serverlist_params(serverlist_params_dict, kwargs, ip_version):
    error_msg_dict = {}
    for key in serverlist_params_dict:
        found = False
        for item in kwargs["params"]["param_list"]:
            if item["key"] == key:
                found = True
                value = item["value"]
                break
        if not found:
            continue

        param_type = serverlist_params_dict[key]['type']
        if param_type in ("ip", 'splited_ips'):
            handle_func = handle_param_type_func_dict.get(param_type)
            check_result = handle_func(value, ip_version)
        else:
            handle_func = handle_param_type_func_dict.get(param_type)
            check_result = handle_func(value)

        if not check_result:
            logger.error("input param's type is wrong,info:"
                         "key is %s,value is %s" % (key, value))
            msg = f"param key {key} check failed"
            if 'suggest' in list(serverlist_params_dict[key].keys()):
                msg = get_code_msg(serverlist_params_dict[key]['suggest'])
            error_msg_dict.update({key: msg})
    return error_msg_dict


def _check_lld_ip_list(kwargs, ip_version):
    error_msg_dict = {}
    serverlist_lld_dict = SERVERLIST_LLD_IP_DICT
    kwargs_key_lld_ip_list = 'lld_ip_list'
    csbs_float_ip_dict = dict.fromkeys(['CSBS_Float_IP1', 'CSBS_Float_IP2'])
    if kwargs_key_lld_ip_list not in list(kwargs["params"].keys()):
        return error_msg_dict, csbs_float_ip_dict

    for key in list(serverlist_lld_dict.keys()):
        found = False
        ip_address = None
        for item in kwargs['params'][kwargs_key_lld_ip_list]:
            if item["param_name"] != key:
                continue
            found = True
            ip_address = item["ip_address"]
            if key in list(csbs_float_ip_dict.keys()):
                csbs_float_ip_dict[key] = ip_address
            break
        if not found:
            continue
        param_type = serverlist_lld_dict.get(key).get('type')
        if param_type == "ip":
            check_result = check_param_ip(ip_address, ip_version=ip_version)
            if check_result:
                continue
            msg = f"param key {key} check failed"
            if 'suggest' in list(serverlist_lld_dict.get(key).keys()):
                msg = get_code_msg(serverlist_lld_dict.get(key).get('suggest'))
            error_msg_dict.update({key: msg})
    return error_msg_dict, csbs_float_ip_dict


def check_special_string(param_name):
    pattern = "^[^\d\-\ :~!@#$%^&*()_+-=,.]+"
    match_obj = re.search(pattern, param_name)
    if match_obj:
        return True
    else:
        if not param_name:
            return False
        else:
            return True


def check_param_domain_name(param_name):
    pattern = \
        r"^[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+$"
    match_obj = re.match(pattern, param_name, re.I)
    if match_obj:
        return True
    else:
        return False


def check_ip_none(value):
    if value:
        return check_param_ip(value) or check_param_ip(value, ip_version=6)
    return True


def check_integer_none(value):
    if value:
        return check_param_integer(value)
    return True


def check_str_none(value):
    if value:
        return check_special_string(value)
    return True


def check_special_path(value):
    result = check_special_string(value)
    if result and not str(value).__contains__('.'):
        return True
    return False


def check_splited_ips(value, ip_version):
    if not value:
        return False
    if value.__contains__(',') and value.split(',').__len__() == 2:
        for _ip in value.split(','):
            if not check_param_ip(_ip.strip(), ip_version):
                return False
        return True
    return False


def check_float_ip_start_end(float_ip1, float_ip2):
    return IPy.IP(float_ip1) < IPy.IP(float_ip2)


def check_content_b2b_param(b2b_param_list):
    for item in b2b_param_list:
        if item["key"] in list(SERVERLIST_PARAMS_B2B_SCENE.keys()):
            return True
    return False


def _ip_match_gateway_and_subnet_mask(param_name, error_msg_dict, ip_address, gateway, subnet_mask):
    try:
        if ip_address not in IPy.IP(gateway).make_net(subnet_mask):
            error_msg_dict.update({param_name: get_code_msg(
                640087) % (ip_address, gateway, subnet_mask)})
    except ValueError:
        pass


def check_ip_conflict(check_params, check_params_name, error_msg_dict):
    param_list = []
    check_ip_dict = {}

    def _get_check_ip_dict():
        for _item in check_params["lld_ip_list"]:
            if _item["param_name"] in check_params_name:
                # Determine whether to check the parameter. If yes, obtain the parameter from the list.
                param_name = _item.get("param_name", "").strip()
                ip_address = _item.get("ip_address", "").strip()
                subnet_mask = _item.get("subnet_mask").strip()
                gateway = _item.get("gateway").strip()
                # Check whether the parameter is empty.
                if not ip_address or not subnet_mask or not gateway:
                    error_msg_dict.update({param_name: get_code_msg(640084)})
                    continue
                # Check whether the configured IP address is a compliant IP address.
                if not is_valid_ip(ip_address) \
                        or not is_valid_ip(gateway):
                    error_msg_dict.update({param_name: get_code_msg(640085)})
                    continue
                # Check whether the configured IP parameters conflict with each other.
                if ip_address in check_ip_dict:
                    error_msg_dict.update(
                        {param_name: get_code_msg(640083) % ip_address})
                    continue
                # Check whether the entered IP address matches the gateway and subnet mask.
                _ip_match_gateway_and_subnet_mask(param_name, error_msg_dict, ip_address, gateway, subnet_mask)
                # Put the parameter to be checked into the dictionary:{ip:param_name,...}
                check_ip_dict[ip_address] = param_name
            else:
                param_list.append(_item)

    def _check_ips_str(ips_str):
        for _ip in ips_str.strip().split(","):
            if _ip.strip() in check_ip_dict:
                error_msg_dict.update({check_ip_dict.get(_ip): get_code_msg(640083) % _ip})
                # Duplicate IP addresses are detected and delete the IP key from the check dictionary.
                check_ip_dict.pop(_ip)

    # Obtain the IP parameters to be checked.
    _get_check_ip_dict()

    # Check whether the parameter values in the parameter list conflict with the target IP address dictionary.
    param_list.extend(check_params["param_list"])
    for item in param_list:
        temp_value = item.get("value", "") if item.get("value") else item.get("ip_address", "")
        if not temp_value:
            continue
        temp_value = temp_value.strip()
        if "," in temp_value:
            # Separately split and verify the IP_list parameter value.
            _check_ips_str(temp_value)
            continue
        if temp_value in check_ip_dict:
            error_msg_dict.update(
                {check_ip_dict[temp_value]: get_code_msg(640083) % temp_value})
            check_ip_dict.pop(temp_value)

    return error_msg_dict


def is_valid_ip(vm_ip):
    try:
        ipaddress.ip_address(vm_ip)
        return True
    except Exception as err:
        logger.warn(err)
        return False


def get_dpa_business_cluster_account_info(params: list):
    """
    获得所有的DPA信息
    return [dict]
    """
    dpa_info_dict = {}
    dpa_account_key = ['DPA_Admin',
                       'DPA_Admin_Password',
                       'DPA_Sec_Admin',
                       'DPA_Sec_Admin_Password',
                       'DPA_General_User',
                       'DPA_General_User_Password',
                       'DPA_Cluster_Address',
                       'DPA_Business_Address',
                       "Az_Related_To_Zone"
                       ]
    for row_info in params:
        if row_info.get('service') == "CSBS-VBS":
            if row_info.get('key') in dpa_account_key:
                dpa_info_dict.update(
                    {row_info.get('key'): row_info.get('value')})
        if row_info.get('key') == 'openstack_region':
            dpa_info_dict.update(
                {row_info.get('key'): row_info.get('value')})
    return dpa_info_dict


handle_param_type_func_dict = {
    "string": check_param_string,
    "super_string": check_special_string,
    "integer": check_param_integer,
    "ip": check_param_ip,
    "domain_name": check_param_domain_name,
    "netmask2": is_netmask2,
    "netmask": check_param_netmask,
    "email": check_param_email,
    "password": check_param_password,
    "str_none": check_str_none,
    "ip_none": check_ip_none,
    "integer_none": check_integer_none,
    "super_path": check_special_path,
    "splited_ips": check_splited_ips
}


class DpaParamCheck:

    def __init__(self, params, project_id):
        """
        dpa参数检查
        检查规则第一条：
        所有的值不能为空
            DPA_Business_Address,DPA_Cluster_Address,DPA_Admin,DPA_Admin_Password
            DPA_Sec_Admin,DPA_Sec_Admin_Password,DPA_General_User,
            DPA_General_User_Password,
            Az_Related_To_DPA
        """
        self.all_err_msg = {}
        self.project_id = project_id
        self.dpa_params = params
        self.business_address = self.dpa_params.get('DPA_Business_Address')
        self.cluster_address = self.dpa_params.get('DPA_Cluster_Address')
        self.az_related_to_dpa = self.dpa_params.get('Az_Related_To_Zone')
        self.pre_model = "normal"
        self.check_items()

    def check_items(self):
        self.business_address_list = re.split('[,#]', self.dpa_params.get('DPA_Business_Address'))
        self.business_address_num = len(self.business_address_list)
        self.cluster_address_list = re.split('&&|,', self.cluster_address)
        self.cluster_address_num = self.cluster_address.split("&&")

        # 检查 DPA_Business_Address的格式
        self.check_dpa_business_address()

        # DPA_Cluster_Address填写是否正确.
        self.dpa_check_cluster_address()

        # 检查AZ信息
        self.check_az_related_to_dpa()

        # 检查账号数量是否一致，并与dpa业务面的IP数量是否一致
        self.dpa_check_info_number()

    def get_ip_version(self):
        if ProjectApi().is_ipv6_project(self.project_id):
            ip_version = '6'
        else:
            ip_version = '4'
        return ip_version

    def dpa_check_info_number(self):
        """
        检视DPA 业务面的IP，账号信息数量是否一致。

        """
        dpa_account_key = ['DPA_Admin',
                           'DPA_Admin_Password'
                           'DPA_Sec_Admin',
                           'DPA_Sec_Admin_Password',
                           'DPA_General_User',
                           'DPA_General_User_Password']
        for key, value in self.dpa_params.items():
            if key in dpa_account_key:
                if self.is_include_space(value):
                    self.all_err_msg.update({key: get_code_msg(640127) % key})
                    return

                account_value_len = len(value.split(','))
                if account_value_len != self.business_address_num:
                    # 错误信息：err_msg.update({key: f'{key}个数不对，请与DPA IP一致'})
                    self.all_err_msg.update({key: get_code_msg(640115) % key})

    def dpa_check_cluster_address(self):
        """
        检查business_address和cluster_address地址是否匹配。
        """
        cluster_address = self.dpa_params.get('DPA_Cluster_Address')
        if not cluster_address:
            return

        if self.is_include_space(cluster_address):
            self.all_err_msg.update(
                {"DPA_Cluster_Address": get_code_msg(640127) % 'DPA_Cluster_Address'})
            return

        for ip_port in self.cluster_address_list:
            result = check_url_list(ip_port, self.get_ip_version())
            if not result:
                self.all_err_msg.update({"DPA_Cluster_Address": get_code_msg(640117)})
                return

            cluster_ip = ip_port.split(':')[0]
            if not self.is_ping(cluster_ip):
                self.all_err_msg.update({"DPA_Cluster_Address": get_code_msg(640123) % cluster_ip})
                return

        cluster_address_len = len(self.cluster_address_num)
        if cluster_address_len != self.business_address_num:
            self.all_err_msg.update({"DPA_Cluster_Address": get_code_msg(640116)})
            return

        if len(set(self.cluster_address_list)) != len(self.cluster_address_list):
            self.all_err_msg.update({
                "DPA_Cluster_Address": get_code_msg(640125) % "DPA_Cluster_Address"})

    def _iter_business_address_when_sharp_exist(self):
        for zone_ip_port in self.business_address.split('#'):
            for ip_port in zone_ip_port.split(','):
                if not check_url_list(ip_port, self.get_ip_version()):
                    self.all_err_msg.update({"DPA_Business_Address": get_code_msg(640117)})

    def check_dpa_business_address(self):
        """
        检查dpa_cluster_address 的格式，例:192.168.1.1:8088
        :return:

        """

        if not self.business_address:
            return

        if self.is_include_space(self.business_address):
            self.all_err_msg.update(
                {"DPA_Business_Address": get_code_msg(640127) % 'DPA_Business_Address'})
            return
        if '#' in self.business_address:
            self.pre_model = 'csha'
            if self.business_address.count('#') != 1:
                self.all_err_msg.update({"DPA_Business_Address": get_code_msg(640117)})
            if self.business_address[-1] == '#' or self.business_address[0] == '#':
                self.all_err_msg.update({"DPA_Business_Address": get_code_msg(640117)})
            self._iter_business_address_when_sharp_exist()
        else:
            for ip_port in self.business_address.split(','):
                if not check_url_list(ip_port, self.get_ip_version()):
                    self.all_err_msg.update({"DPA_Business_Address": get_code_msg(640117)})

        if len(set(self.business_address_list)) != len(self.business_address_list):
            self.all_err_msg.update(
                {"DPA_Business_Address": get_code_msg(
                    640125) % "DPA_Business_Address"})

    @staticmethod
    def is_ping(tar_ip):
        return ping(tar_ip)

    def check_az_related_to_dpa(self):

        if not self.az_related_to_dpa:
            return

        if self.is_include_space(self.az_related_to_dpa):
            self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640127) % "Az_Related_To_Zone"})
            return
        if self.az_related_to_dpa[0] in [',', '#'] or self.az_related_to_dpa[-1] in [',', '#']:
            self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640120) % "Az_Related_To_Zone"})
            return
        if self.az_related_to_dpa.count("#") > 1:
            self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640120) % "Az_Related_To_Zone"})
            return
        if self.pre_model == 'normal' and self.az_related_to_dpa.count("#") >= 1:
            self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640120) % "Az_Related_To_Zone"})
            return

        azs_s = re.split('[,#]', self.az_related_to_dpa)
        for available_zone in azs_s:
            if not available_zone:
                self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640120) % "Az_Related_To_Zone"})
                return
            if not check_openstack_region(available_zone):
                self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640120) % "Az_Related_To_Zone"})
                return
        if len(list(set(azs_s))) != len(azs_s):
            # 判断AZ名称是否有重复
            self.all_err_msg.update({"Az_Related_To_Zone": get_code_msg(640121) % "Az_Related_To_Zone"})
            return
        project_conditions = get_project_conditions(self.project_id)
        is_project_deploy = project_conditions.get('ProjectDeploy')
        if is_project_deploy:
            openstack_region = self.dpa_params.get('openstack_region')
            if self.az_related_to_dpa != openstack_region:
                self.all_err_msg.update(
                    {"Az_Related_To_DPA": get_code_msg(640122)})

    @staticmethod
    def check_network_segment(net_work):
        pattern = r'^(?=(\b|\D))(((\d{1,2})|(1\d{1,2})|(2[0-4]\d)|(25[0-5]))' \
                  r'\.){3}((\d{1,2})|(1\d{1,2})|(2[0-4]\d)|(25[0-5]))(?=(\b|\
                  D))/([1-2][0-9]|3[0-2]|[1-9])$'
        match_obj = re.match(pattern, net_work)
        return match_obj

    @staticmethod
    def is_include_space(info_str):
        """
        判断是否包含空格
        """
        pattern = r'\s'
        search_result = re.search(pattern, info_str)
        return search_result
