import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.step_base import DeployBaseSubJob
from plugins.CSBS.scripts.deploy.karbor.cbs_installer import cbs_karborhealth

logger.init("KarborCheck")
RETRY = 15


class KarborCheck(DeployBaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list=None):
        super(KarborCheck, self).__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id

    def execute(self, project_id, pod_id, regionid_list=None):
        retry = 1
        error_service = ""
        logger.info('Start to check the services after the CSBS-VBS component is installed.')
        while retry <= RETRY:
            try:
                if not self._config_karbor(project_id, pod_id, retry):
                    retry += 1
                    continue
                else:
                    logger.info('After the CSBS-VBS component is installed, the service check is successful.')
                    return Message(200)
            except Exception as err:
                logger.error(f"Execute error:{str(err)}")
                return Message(500, HCCIException(640001, str(err)))
        return Message(500, HCCIException(640056, error_service))

    def _config_karbor(self, project_id, pod_id, retry):
        health_checker = cbs_karborhealth.KarborHealth(project_id, pod_id)
        if health_checker.install():
            return True
        error_service_list = list(set(health_checker.error_service))
        error_object_list = list(set(health_checker.error_object))
        error_object = None
        error_service = None
        if error_service_list:
            error_service = self._handle_error_info(error_service_list, "fault")
        if error_object_list:
            error_object = self._handle_error_info(error_object_list, "error")
        logger.error(f"Check service result: {error_service}, check karbor connect result: {error_object}.")
        time.sleep(60)
        logger.info(f"Start retry:{retry}.")
        return False

    @staticmethod
    def _handle_error_info(error_list, err_flag):
        error_result = ""
        for err_info in error_list:
            error_result = f"{error_result}, {err_info} is {err_flag}"
        return error_result[2:]

    def rollback(self, project_id, pod_id, regionid_list=None):
        try:
            if not cbs_karborhealth.KarborHealth(project_id, pod_id).rollback():
                return Message(500, HCCIException(640056, " "))
        except Exception as err:
            logger.error(f"Execute error:{err}")
            return Message(500, HCCIException(640001, str(err)))
        return Message(200)
