import os
import re

import utils.common.log as logger

from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.OpenStack.cps_util import HostsInfoByPod
from utils.business.arb_util import ArbApi
from utils.business.dns_util import DNSApi
from utils.business.dns_util import DNSNodeType
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi
from utils.common.param_check import check_param_ip
from utils.common.software_package_util import find_software_package_by_name

from plugins.CSBS.common import model
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.util import check_openstack_region
from plugins.CSBS.scripts.deploy.karborproxy.common.funtions import check_param_string
from plugins.CSBS.scripts.deploy.karborproxy.common.funtions import check_url_list
from plugins.CSBS.scripts.deploy.karborproxy.libs.error_msgs import GetDNSIPError
from plugins.CSBS.scripts.deploy.karborproxy.libs.error_msgs import ParamFormatError
from plugins.CSBS.scripts.deploy.karborproxy.libs.error_msgs import ParamIsNullWrong
from plugins.CSBS.scripts.deploy.karborproxy.libs.error_msgs import ParamNumUnequalError


class Node:
    def __init__(self, node_ip, password, account):
        self.node_ip = node_ip
        self.password = password
        self.account = account


class Base:
    def __init__(self, project_id, pod_id):
        self.pod_id = pod_id
        self.project_id = project_id
        self.param_util = ParamUtil()
        self.project_api = ProjectApi()
        self.service_name = "CSBS-VBS"
        self.proxy_name = "KarborProxy"
        self.data = self.param_util.get_service_cloud_param(
            self.pod_id, self.service_name)
        self.manageone = ManageOneUtil2()

    def get_ip_version(self):
        if ProjectApi().is_ipv6_project(self.project_id):
            ip_version = 'ipV6'
        else:
            ip_version = 'ipV4'
        logger.info('The IP version is %s.' % ip_version)
        return ip_version


class AgentProxy(Base):
    """
    Obtaining KarborProxy Node Information
    """
    network_name = "DMZ_Service"
    root_name = "root"
    dpamanager_name = "dpamanager"
    dpamanager_path = "/home/dpamanager"
    root_path = "/root"
    port = 22
    web_port = 8088

    def __init__(self, project_id, pod_id):
        super(AgentProxy, self).__init__(project_id, pod_id)
        self.kms_address = self.data.get('kms_address')
        self.openstack_region = self.param_util.get_param_value(
            self.pod_id, "OpenStack", "openstack_region", "openstack_region")
        self.dpamanager_pwd = self.data.get("KarborProxy_dpamanager_password")
        self.root_pwd = self.data.get("KarborProxy_root_password")
        self.karbor_proxy_ip0 = self._get_check_ip(self.data.get("karbor_proxy_ip0"))
        self.karbor_proxy_ip1 = self._get_check_ip(self.data.get("karbor_proxy_ip1"))
        self.management_float_ip = self._get_check_ip(
            self.data.get("karbor_proxy_management_ip"))
        self.timezone = self._get_check_string(self.data.get("timezone"))
        self.agent_dmk_user_name = self.data.get('CBS_dmk_groupadminname')
        self.agent_dmk_pwd = self.data.get('CBS_dmk_grouppasswd')
        self.karbor_proxy_ip0_gateway = \
            self.data.get('karbor_proxy_ip0_gateway')
        self.karbor_proxy_ip0_subnet_mask = \
            self.data.get('karbor_proxy_ip0_subnet_mask')
        self.region = self.data.get('region_name')
        self.agent_dmk_ip = self.param_util.get_param_value(
            pod_id, 'DMK', 'dmk_floatIp', 'dmk_floatIp')
        self._ip = self.karbor_proxy_ip0
        self.node_ip = self.karbor_proxy_ip0
        self.region_id = self._get_check_string(self.param_util.get_param_value(
            self.pod_id, "public", "region_id", "region0_id"))
        self.cipher_type = ParamTool(self.project_id, pod_id).get_cipher_type()

        self._get_fsp_params()
        self._init_params()
        self.parameter_cannot_empty()

    def _get_check_ip(self, check_ip):
        if not check_ip:
            return check_ip
        if not check_param_ip(check_ip, int(self.get_ip_version()[-1])):
            raise Exception('The IP address format does '
                            'not comply with the specifications.')
        return check_ip

    def _get_fsp_params(self):
        if get_project_condition_boolean(
                self.project_id, 'ExpansionAdCloudService'):
            self.fsp_root_pwd = self.param_util.get_param_value(
                self.pod_id, "OpenStack", 'root_password',
                'openstack_root_password')
            self.fsp_user_pwd = self.param_util.get_param_value(
                self.pod_id, "OpenStack", 'fsp_password',
                'openstack_fsp_password')
            self.fsp_env_pwd = self.param_util.get_param_value(
                self.pod_id, "OpenStack", 'env_os_password',
                'openstack_env_os_password')

    @staticmethod
    def _get_check_string(check_str):
        if not check_str:
            return check_str
        if not check_param_string(check_str):
            raise Exception("Command injection risks exist in character "
                            "strings.")
        return check_str

    def _init_params(self):
        if get_project_condition_boolean(self.project_id,
                                         "InstallCSBS-VBS-Proxy"):
            self.alarm_user, self.alarm_pwd = \
                self.manageone.getThirdpartyInfor(self.pod_id)
            self._get_check_string(self.alarm_user)
            self.param_tool = ParamTool(self.project_id, self.pod_id)
            self.iam_karbor_address = \
                self.param_tool.get_iam_url().replace('https://', '')
            self.om_url = \
                self.param_tool.get_oc_url().replace('https://', '')

    def parameter_cannot_empty(self):
        not_check_empty_params = ['az_related_to_dpa']
        if get_project_condition_boolean(self.project_id, "InstallCSBS-VBS-Proxy"):
            for key, value in self.__dict__.items():
                if (key not in not_check_empty_params) and (not value):
                    ParamIsNullWrong(key).to_raise()

    def is_arm(self):
        return get_project_condition_boolean(self.project_id, "manageARM")

    @property
    def all_ip_list(self):
        return [self.karbor_proxy_ip0, self.karbor_proxy_ip1]

    def get_proxy_user_info(self):
        if get_project_condition_boolean(self.project_id,
                                         "InstallCSBS-VBS-Proxy"):
            user_pwd = self.dpamanager_pwd
            if not user_pwd:
                raise Exception(f"Failed to obtain the {self.dpamanager_name} "
                                f"pwd of the {self.proxy_name} node.")

            root_pwd = self.root_pwd
            if not root_pwd:
                raise Exception(f"Failed to obtain the root pwd of the "
                                f"{self.proxy_name} node.")
        elif get_project_condition_boolean(self.project_id,
                                           "InstalledCSBS-VBS-Proxy"):
            user_pwd = self.param_util.get_param_value(
                self.pod_id, self.service_name,
                "KarborProxy_dpamanager_password",
                "KarborProxy_node_dpamanager_pwd")
            if not user_pwd:
                raise Exception(f"Failed to obtain the {self.dpamanager_name} "
                                f"pwd of the {self.proxy_name} node.")

            root_pwd = self.param_util.get_param_value(
                self.pod_id, self.service_name, "KarborProxy_root_password",
                "KarborProxy_node_root_pwd")
            if not root_pwd:
                raise Exception(f"Failed to obtain the root pwd of the "
                                f"{self.proxy_name} node.")
        else:
            raise Exception(f"Failed to obtain the pwd of the "
                            f"{self.proxy_name} node, check whether the HCSD "
                            f"project scene is normal.")
        return model.NodeUser(user_name=self.dpamanager_name,
                              user_pwd=user_pwd,
                              root_name=self.root_name,
                              root_pwd=root_pwd)

    def get_proxy_node_info_from_cmdb(self):
        nodes_info = ManageOneCmdbUtil(self.project_id, self.pod_id). \
            get_deploy_node_info(
            region_code=self.region_id,
            index_name=self.service_name)
        if not nodes_info:
            raise Exception("Failed to obtain karbor proxy node info.")
        logger.info(f"Succeed to obtain {self.service_name} node info.")
        return nodes_info

    def _get_proxy_ip_list_from_cmdb(self):
        ip_list = []
        for node_info in self.get_proxy_node_info_from_cmdb():
            if not (node_info and self.proxy_name in node_info.get("name")):
                continue
            for ip_dic in node_info.get("ipAddresses"):
                if ip_dic.get("netPlane") == self.network_name:
                    ip_list.append(ip_dic.get("ip"))
        return ip_list

    def get_proxy_ip_list(self):
        proxy_ip_list = []
        if get_project_condition_boolean(self.project_id,
                                         "InstallCSBS-VBS-Proxy"):
            proxy_ip_list = [self.karbor_proxy_ip0, self.karbor_proxy_ip1]
        elif get_project_condition_boolean(self.project_id,
                                           "InstalledCSBS-VBS-Proxy"):
            proxy_ip_list = self._get_proxy_ip_list_from_cmdb()
        if not proxy_ip_list:
            raise Exception("Failed to obtain the karbor proxy ips.")
        proxy_ip_list = list(set(proxy_ip_list))
        proxy_ip_list.sort()
        return proxy_ip_list

    def get_proxy_node_list(self):
        user_info = self.get_proxy_user_info()
        proxy_nodes_list = []
        for proxy_ip in self.get_proxy_ip_list():
            proxy_node = model.Node(
                node_ip=proxy_ip,
                user=user_info.user_name,
                user_pwd=user_info.user_pwd,
                root_name=user_info.root_name,
                root_pwd=user_info.root_pwd,
                extra="")
            proxy_nodes_list.append(proxy_node)
        return proxy_nodes_list

    @property
    def all_ips(self):
        ips = ','.join([self.karbor_proxy_ip0, self.karbor_proxy_ip1])
        logger.info('The IP address of karborproxy is %s .' % ips)
        return ips

    @property
    def proxy_ip(self):
        """Obtain the IP address of the first node by default."""
        return self._ip

    @proxy_ip.setter
    def proxy_ip(self, value):
        self._ip = value

    @property
    def is_ha_scene(self):
        ha_info = self.project_api.get_project_ha_dr_status(self.project_id)
        return ha_info.get("RegionConHA")

    @property
    def is_csha(self):
        ha_info = self.project_api.get_project_ha_dr_status(self.project_id)
        return ha_info.get("CSHAStorage")

    def get_dns_ips(self):
        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        logger.info('This dns ip is %s .' % ','.join(dns_ips))
        if not dns_ips:
            GetDNSIPError().to_raise()
        for dns_ip in dns_ips:
            if not check_param_ip(dns_ip, int(self.get_ip_version()[-1])):
                raise Exception('The DNS IP address format does '
                                'not comply with the specifications.')

        return dns_ips

    @staticmethod
    def get_agent_config_ini_path():
        cur_path = os.path.dirname(os.path.realpath(__file__))
        file_name = os.path.join(cur_path, "sys.ini")
        return file_name


class AgentProxyDPAInfo(AgentProxy):
    param_nums = 0
    dpa_params = [
        'dpa_sec_admin',
        'dpa_sec_admin_password',
        'dpa_general_user',
        'dpa_general_user_password',
        'dpa_admin',
        'dpa_admin_password',
    ]

    def __init__(self, project_id, pod_id):
        super(AgentProxyDPAInfo, self).__init__(project_id, pod_id)
        self.dpa_business_address = self.data.get('dpa_business_address')
        self.dpa_cluster_address = self.data.get('dpa_cluster_address')
        # 校验
        self.dpa_admin = self.data.get('dpa_admin')
        self.dpa_admin_pwd = self.data.get('dpa_admin_password')
        self.dpa_sec_admin = self.data.get('dpa_sec_admin')
        self.dpa_sec_admin_pwd = self.data.get('dpa_sec_admin_password')
        self.dpa_general_user = self.data.get('dpa_general_user')
        self.dpa_general_user_pwd = self.data.get('dpa_general_user_password')
        self.az_related_to_dpa = self.data.get('az_related_to_zone')
        self.openstack_expansion_az = self.param_util.get_param_value(
            self.pod_id, "OpenStack", "openstack_region", "openstack_expansion_az")
        self.data_protection_app_mode = 'normal'
        self.pre_data_protection_app_mode = 'normal'
        self._get_dpa_params()
        self.check_dpa_info()

    def _get_dpa_params(self):
        if self._is_install_proxy():
            self.dpa_business_address_list = self.get_dpa_business_address_list()
            self.agent_assist_version = self.get_agent_assist_version()
            self.agent_version = self.get_agent_version()
            self.dpa_cluster_address_list = self.data.get(
                'dpa_cluster_address').split('&&')
            self.dpa_info_list = self.get_dpa_info()
            self.get_dpa_az_zone()

    def get_az_related_to_dpa(self):
        return re.split('[,#]', self.az_related_to_dpa)

    def _get_dpa_business_address_list(self, dpa_business_address, dpa_business_address_list):
        zone = 'business_zone'
        for zone_ip_port in re.split('(#)', dpa_business_address):
            for ip_port in zone_ip_port.split(','):
                if ip_port != '#' and not check_url_list(ip_port, self.get_ip_version()):
                    ParamFormatError('dpa_business_address').to_raise()
                if ip_port != '#' and zone == 'business_zone':
                    dpa_business_address_list.append((ip_port, zone))
                elif ip_port == '#':
                    zone = 'ha_zone'
                else:
                    dpa_business_address_list.append((ip_port, zone))

    def get_dpa_business_address_list(self):
        dpa_business_address_list = []
        if '#' in self.dpa_business_address:
            if self.dpa_business_address.count('#') != 1:
                ParamFormatError('dpa_business_address').to_raise()
            if self.dpa_business_address[-1] == '#' or self.dpa_business_address[0] == '#':
                ParamFormatError('dpa_business_address').to_raise()
            self.pre_data_protection_app_mode = 'csha'
            self._get_dpa_business_address_list(self.dpa_business_address, dpa_business_address_list)
        else:
            for ip_port in self.dpa_business_address.split(','):
                if not check_url_list(ip_port, self.get_ip_version()):
                    ParamFormatError('dpa_business_address').to_raise()
                dpa_business_address_list.append(ip_port)
        return dpa_business_address_list

    def get_fsp_node(self):
        fsp_node = ''
        if get_project_condition_boolean(
                self.project_id, 'ExpansionAdCloudService') or \
                get_project_condition_boolean(self.project_id, 'BCInstalled'):
            fsp_node_info = HostsInfoByPod(self.project_id, self.pod_id)
            fsp_node_list = fsp_node_info.get_hosts_info()
            fsp_node = fsp_node_list[0].get('external_omip', '')
        if not fsp_node:
            logger.error("get fsp node info is empty.")
            raise Exception("get fsp node info is empty.")
        return fsp_node

    def get_agent_assist_version(self):
        _, package_name = find_software_package_by_name(
            pkg_pre_name='OceanStor_BCManager',
            pkg_post_name='AgentAssist_linux_x86_64.tar.gz')
        version = package_name.split('_')[2]
        # 校验
        self._get_check_string(version)
        logger.info('This assistant package version is %s .' % version)
        return version

    def get_agent_version(self):
        _, package_name = find_software_package_by_name(
            pkg_pre_name='CDM',
            pkg_post_name='Client_Linux_el7_x64.tar.gz')
        version = package_name.split('_')[1]
        # 校验
        self._get_check_string(version)
        logger.info('This agent package version is %s .' % version)
        return version

    def check_dpa_info(self):
        if self._is_install_proxy():
            self.check_dpa_info_num()
            self.check_dpa_cluster_address()

    def get_az_list(self):
        az_list = []
        if not self.az_related_to_dpa:
            ParamFormatError('az_related_to_zone').to_raise()
        for zone_az in self.az_related_to_dpa.split('#'):
            if not zone_az:
                ParamFormatError('az_related_to_zone').to_raise()
            for available_zone in zone_az.split(','):
                if not check_openstack_region(available_zone):
                    ParamFormatError('az_related_to_zone').to_raise()
        return az_list

    def get_dpa_az_zone(self):
        """
        az_related_to_dpa 参数存在且大于等于2时自动配置高可用 csha，其他场景均为 normal
        :return:
        """
        az_list = self.get_az_list()
        b_az, ha_az = None, None
        if self.pre_data_protection_app_mode == 'csha':
            if len(az_list) == 1:
                return az_list[0], ha_az
            if self.az_related_to_dpa.count('#') not in [1, 0]:
                ParamFormatError('az_related_to_zone').to_raise()
            if self.az_related_to_dpa.count('#'):
                b_az, ha_az = self.az_related_to_dpa.split('#')
                self.data_protection_app_mode = 'csha'
            else:
                b_az = self.az_related_to_dpa
        else:
            if '#' in self.az_related_to_dpa:
                ParamFormatError('az_related_to_zone').to_raise()
            b_az = self.az_related_to_dpa
        return b_az, ha_az

    def get_dpa_info(self):
        return [self.data.get(item).split(',') for item in self.dpa_params]

    def check_dpa_info_num(self):
        """
        检查账户数量和集群数量是否对等。
        """
        check_out = []
        param_nums = len(self.dpa_business_address_list)
        for item in self.dpa_params:
            item_value = self.data.get(item).split(',')
            if param_nums != len(item_value):
                ParamNumUnequalError(item).to_raise()
            check_out.extend(item_value)
        return check_out

    def check_dpa_cluster_address(self):
        param_nums = len(self.dpa_business_address_list)
        dpa_cluster_address_list = \
            self.data.get('dpa_cluster_address').split('&&')
        if param_nums != len(dpa_cluster_address_list):
            ParamNumUnequalError('dpa_cluster_address').to_raise()
        for item in dpa_cluster_address_list:
            for ip_port in item.split(','):
                if not check_url_list(ip_port, self.get_ip_version()):
                    ParamFormatError('dpa_cluster_address').to_raise()

    def _is_install_proxy(self):
        return get_project_condition_boolean(self.project_id,
                                             "InstallCSBS-VBS-Proxy") \
               or get_project_condition_boolean(self.project_id,
                                                "InstalledCSBSProxy")


class AgentProxyVMINFO(AgentProxy):
    """
    Obtaining KarborProxy VM Node Information
    """
    management_floatIP_name = "Management-KarborProxy-Float"
    internal_floatIP_name = "Internal-KarborProxy-Float"
    vm_image_name = "KarborProxy-image"
    az_name = "manage-az"
    volume_type_name = "manager.fsa.csg"
    sys_disk = 10
    server_name = "CSBS"
    vm_group_name = "KarborProxy-01"
    vm_group_workflow_name = "KarborProxy-02"
    disk = 110

    def __init__(self, project_id, pod_id):
        super(AgentProxyVMINFO, self).__init__(project_id, pod_id)
        self.vm_cpu = self.data.get("CSBS_proxy_cpu")
        self.vm_mem = self.data.get("CSBS_proxy_mem")
        logger.info(
            'The VM CPU is %s, MEM is %s .' % (self.vm_cpu, self.vm_mem))
        self.set_volume_info()

    def set_volume_info(self):
        if self.is_ha_scene:
            logger.info('This is a DR HA scenario.')
            self.az_name = "dr-manage-az"
            self.volume_type_name = "dr.manager.fsa.csg"

    def get_arb_api_info(self):
        arb_api = ArbApi()
        arb_info = arb_api.get_arb_info(self.pod_id)
        ips = ','.join(arb_info['ips'])
        password = arb_info['user_info']['password']
        dc1_name = arb_info['dc_name']['dc1_name']
        dc2_name = arb_info['dc_name']['dc2_name']
        logger.info('arb info | ips:%s,'
                    'dc1_name:%s, dc2_name:%s .', ips, dc1_name, dc2_name)
        return [(ips, dc1_name, dc2_name, password),
                (ips, dc2_name, dc1_name, password)]


class AgentProxyDPAInfos(AgentProxyDPAInfo):
    def get_agent_assist_version(self):
        return True

    def get_agent_version(self):
        return True
