import os

import utils.common.log as logger
from utils.business.project_condition_utils import get_project_condition_boolean
from utils.business.arb_util import ArbApi
from utils.business.dns_util import DNSApi
from utils.business.dns_util import DNSNodeType
from utils.business.manageone_cmdb_util import ManageOneCmdbUtil
from utils.business.manageone_util2 import ManageOneUtil2
from utils.business.param_util import ParamUtil
from utils.business.project_util import ProjectApi

from plugins.CSBS.common import model
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.scripts.modification.region_con_ha.karborproxy.libs.error_msgs import ParamIsNullWrong, GetDNSIPError


class Base:
    def __init__(self, project_id, pod_id):
        self.pod_id = pod_id
        self.project_id = project_id
        self.param_util = ParamUtil()
        self.project_api = ProjectApi()
        self.service_name = "CSBS-VBS"
        self.proxy_name = "KarborProxy"
        self.data = self.param_util.get_service_cloud_param(
            self.pod_id, self.service_name)
        self.manageone = ManageOneUtil2()

    def get_ip_version(self):
        if ProjectApi().is_ipv6_project(self.project_id):
            ip_version = 'ipV6'
        else:
            ip_version = 'ipV4'
        logger.info('The IP version is %s.' % ip_version)
        return ip_version


class AgentProxy(Base):
    """
    Obtaining KarborProxy Node Information
    """
    network_name = "DMZ_Service"
    root_name = "root"
    dpamanager_name = "dpamanager"
    dpamanager_path = "/home/dpamanager"
    root_path = "/root"
    port = 22
    web_port = 8088

    def __init__(self, project_id, pod_id):
        super(AgentProxy, self).__init__(project_id, pod_id)
        self.dpamanager_pwd = self.data.get("KarborProxy_dpamanager_password")
        self.root_pwd = self.data.get("KarborProxy_root_password")
        self.karbor_proxy_ip0 = self.data.get("karbor_proxy_ip0")
        self.karbor_proxy_ip1 = self.data.get("karbor_proxy_ip1")
        self.region_id = \
            self.param_util.get_param_value(self.pod_id, "public",
                                            "region_id", "region0_id")

    def parameter_cannot_empty(self):
        not_check_empty_params = ['az_related_to_dpa']
        for key, value in self.__dict__.items():
            if key not in not_check_empty_params:
                if not value:
                    ParamIsNullWrong(key).to_raise()

    def is_arm(self):
        return get_project_condition_boolean(self.project_id, "manageARM")

    def get_account_for_unified_pwd(self):
        param_tool = ParamTool(self.project_id, self.pod_id)
        account_list = [('dpamanager', 1), ('root', 1)]
        account_info = param_tool.get_account_pwd_from_unify_pwd(param_tool.get_region_id(), account_list,
                                                                 sub_component_name='KarborProxy')
        return {item.get('accountName'): item.get('newPasswd') for item in account_info if item.get('accountName')}

    def get_proxy_user_info(self):

        if get_project_condition_boolean(self.project_id,
                                         "InstalledCSBS-VBS-Proxy"):
            karbor_account = self.get_account_for_unified_pwd()
            user_pwd = karbor_account.get('dpamanager')
            if not karbor_account.get('dpamanager'):
                raise Exception(f"Failed to obtain the {self.dpamanager_name} "
                                f"pwd of the {self.proxy_name} node.")

            root_pwd = karbor_account.get('root')
            if not root_pwd:
                raise Exception(f"Failed to obtain the root pwd of the "
                                f"{self.proxy_name} node.")
        else:
            raise Exception(f"Failed to obtain the pwd of the "
                            f"{self.proxy_name} node, check whether the HCSD "
                            f"project scene is normal.")
        return model.NodeUser(user_name=self.dpamanager_name,
                              user_pwd=user_pwd,
                              root_name=self.root_name,
                              root_pwd=root_pwd)

    def get_proxy_node_info_from_cmdb(self):
        nodes_info = ManageOneCmdbUtil(self.project_id, self.pod_id). \
            get_deploy_node_info(
            region_code=self.region_id,
            index_name=self.service_name)
        if not nodes_info:
            raise Exception("Failed to obtain karbor proxy node info.")
        logger.info(f"Succeed to obtain {self.service_name} node info.")
        return nodes_info

    def _get_proxy_ip_list_from_cmdb(self):
        ip_list = []
        for node_info in self.get_proxy_node_info_from_cmdb():
            if not (node_info and self.proxy_name in node_info.get("name")):
                continue
            for ip_dic in node_info.get("ipAddresses"):
                if ip_dic.get("netPlane") == self.network_name:
                    ip_list.append(ip_dic.get("ip"))
        return ip_list

    def get_proxy_ip_list(self):
        proxy_ip_list = []
        if get_project_condition_boolean(self.project_id,
                                         "InstalledCSBS-VBS-Proxy"):
            proxy_ip_list = self._get_proxy_ip_list_from_cmdb()
        if not proxy_ip_list:
            raise Exception("Failed to obtain the karbor proxy ips.")
        logger.info('Karbor-Proxy-ip is %s' % ','.join(proxy_ip_list))
        proxy_ip_list = list(set(proxy_ip_list))
        proxy_ip_list.sort()
        return proxy_ip_list

    def get_proxy_node_list(self):
        user_info = self.get_proxy_user_info()
        proxy_nodes_list = []
        for proxy_ip in self.get_proxy_ip_list():
            proxy_node = model.Node(
                node_ip=proxy_ip,
                user=user_info.user_name,
                user_pwd=user_info.user_pwd,
                root_name=user_info.root_name,
                root_pwd=user_info.root_pwd,
                extra="")
            proxy_nodes_list.append(proxy_node)
        return proxy_nodes_list

    def get_dns_ips(self):
        dns = DNSApi()
        dns_type = DNSNodeType()
        dns_ips = dns.get_dns_ips(dns_type.OM, self.pod_id)
        logger.info('This dns ip is %s .' % ','.join(dns_ips))
        if not dns_ips:
            GetDNSIPError().to_raise()
        return dns_ips

    @staticmethod
    def get_agent_config_ini_path():
        cur_path = os.path.dirname(os.path.realpath(__file__))
        file_name = os.path.join(cur_path, "sys.ini")
        return file_name


class AgentProxyVMINFO(AgentProxy):
    """
    Obtaining KarborProxy VM Node Information
    """
    management_floatIP_name = "Management-KarborProxy-Float"
    internal_floatIP_name = "Internal-KarborProxy-Float"
    vm_image_name = "KarborProxy-image"
    az_name = "manage-az"
    volume_type_name = "manager.fsa.csg"
    sys_disk = 10
    server_name = "CSBS"
    vm_group_name = "KarborProxy-01"
    vm_group_workflow_name = "KarborProxy-02"
    disk = 110

    def __init__(self, project_id, pod_id):
        super(AgentProxyVMINFO, self).__init__(project_id, pod_id)
        self.vm_cpu = self.data.get("CSBS_proxy_cpu")
        self.vm_mem = self.data.get("CSBS_proxy_mem")
        logger.info(
            'The VM CPU is %s, MEM is %s .' % (self.vm_cpu, self.vm_mem))
        self.set_volume_info()

    @property
    def is_ha_scene(self):
        ha_info = self.project_api.get_project_ha_dr_status(self.project_id)
        return ha_info.get("RegionConHA")

    def set_volume_info(self):
        if self.is_ha_scene:
            logger.info('This is a DR HA scenario.')
            self.az_name = "dr-manage-az"
            self.volume_type_name = "dr.manager.fsa.csg"

    def get_arb_api_info(self):
        arb_api = ArbApi()
        arb_info = arb_api.get_arb_info(self.pod_id)
        ips = ','.join(arb_info['ips'])
        password = arb_info['user_info']['password']
        dc1_name = arb_info['dc_name']['dc1_name']
        dc2_name = arb_info['dc_name']['dc2_name']
        logger.info('arb info | ips:%s,'
                    'dc1_name:%s, dc2_name:%s .', ips, dc1_name, dc2_name)
        return [(ips, dc1_name, dc2_name, password),
                (ips, dc2_name, dc1_name, password)]
