import collections
import json
import re

import requests
import utils.common.log as logger
from utils.business.param_util import ParamUtil
from utils.common.message import Message

from plugins.CSBS.common.ebackup_util import EBackupUtil
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.util import check_ip, auto_retry, set_ebackup_server
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class ConfigEbackupAZMapToKarbor(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.karbor_operation = KarborOperation(project_id)
        self.node = self.karbor_operation.karbor_node_list[0]
        self.ssh_client = SshClient()
        self.param_util = ParamUtil()
        self.ebackup_util = EBackupUtil(project_id, pod_id, regionid_list)
        self.base_url = ""
        self.headers = dict()
        self.token = ""
        self.session_content = ""

    @staticmethod
    def _get_workflow_ip(workflow_ip_list):
        workflow_node_ip = None
        for workflow_ip in workflow_ip_list:
            workflow_login_url = f"https://{workflow_ip}:8088"
            logger.info(f"workflow_login_url is:{workflow_login_url}")
            try:
                rsp = requests.post(workflow_login_url, headers={'Accept': 'application/json'}, verify=False, timeout=5)
            except Exception as exception:
                logger.info(f"Request {workflow_login_url} failed:{exception}.")
                continue
            redirect_history_list = rsp.history
            if not redirect_history_list:
                continue
            redirect_url = redirect_history_list[len(redirect_history_list) - 1].headers['Location']
            workflow_node_ip = re.findall(r'[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+', redirect_url)[0]
            if workflow_node_ip:
                return workflow_node_ip
        return workflow_node_ip

    @staticmethod
    def _make_base_url(workflow_mg_float_ip):
        if not check_ip(workflow_mg_float_ip):
            raise Exception(f"Illegal workflow management float ip:{workflow_mg_float_ip}.")
        return f"https://{workflow_mg_float_ip}:8088/rest/dev"

    @staticmethod
    def _check_rsp_when_req_url(rsp):
        if not isinstance(rsp, requests.Response):
            raise Exception("Request url failure.")
        if rsp.status_code != 200:
            raise Exception("Request url success, but http status code is not 200.")

    def login(self, login_url, workflow_user, workflow_user_pwd):
        rsp = requests.post(login_url,
                            headers={'Accept': 'application/json'},
                            data=f'{{"scope":0,"username":"{workflow_user}","password":"{workflow_user_pwd}"}}',
                            verify=False,
                            timeout=5)

        self._check_rsp_when_req_url(rsp)

        errcode = rsp.json()["error"]["code"]
        if errcode is None:
            raise Exception("Error code is None.")
        if errcode == 0:
            self.token = rsp.json()["data"]["iBaseToken"]
            cookie = rsp.headers.get('Set-Cookie')
            self.session_content = cookie.split(';')[0]
            self.headers = {'Accept': 'application/json;version=2.2;charset=UTF-8',
                            'iBaseToken': self.token,
                            'Cookie': f'language=en;{self.session_content};DEVICE_ID=dev;sessionIdleTime=60000;'
                                      f'MACHINEROLE=2;CSRF_IBASE_TOKEN={self.token}'}
            logger.info("Login eBackup success.")
        else:
            description = rsp.json()["error"]["description"]
            raise Exception(f"Login eBackup failed:{description}")

    def logout(self):
        if self.token == "" or self.session_content == "":
            return
        url = self.base_url + '/sessions'
        requests.delete(url, headers=self.headers, verify=False, timeout=5)
        logger.info("Logout success.")

    def get_az_ebackup_info(self, azebackup_settings_url):
        rsp = requests.get(azebackup_settings_url, headers=self.headers, verify=False, timeout=5)

        self._check_rsp_when_req_url(rsp)

        errcode = rsp.json()["error"]["code"]
        description = rsp.json()["error"]["description"]
        if errcode is None or description is None:
            raise Exception("Error code or description is None.")
        if errcode != 0:
            raise Exception(f"Get azebackup_settings failed:{description}.")

        az_ebackup_mapinfo = rsp.json()["data"][0]["EBACKUPINFO"]
        if not az_ebackup_mapinfo:
            raise Exception("az_ebackup_mapinfo is None.")

        az_ebackup_map_list = json.loads(az_ebackup_mapinfo)["data"]
        if not isinstance(az_ebackup_map_list, list):
            raise Exception("az_ebackup_map_list is not list.")
        return az_ebackup_map_list

    def get_ebackup_account(self):
        user = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS', 'karbor_config_eBackup_az_map_user')
        if not user:
            raise Exception(f"Illegal ebackup user:{user}.")
        pwd = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS', 'karbor_config_eBackup_az_map_pwd')
        if not pwd:
            raise Exception("Illegal ebackup user pwd.")
        ebackup_account = collections.namedtuple('Account', ['user', 'pwd'])(user, pwd)
        return ebackup_account

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.config_ebackup_az_map()
        except Exception as err:
            logger.error(f"Execute error:{err}.")
            return Message(500, f"Execute error:{err}.")

    def config_ebackup_az_map(self):
        # 从workflow上获取ebackup信息：
        workflow_ip_list = self.ebackup_util.get_workflow_ip_list()
        workflow_node_ip = self._get_workflow_ip(workflow_ip_list)
        if not check_ip(workflow_node_ip):
            logger.error("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
            raise Exception("Unable to find the workflow_node_ip in workflow_ip_list that can log in workflow.")
        logger.info(f"workflow_node_ip is:{workflow_node_ip}.")
        self.base_url = self._make_base_url(workflow_node_ip)

        # 登陆workflow
        login_url = self.base_url + "/login"
        logger.info(f"Start to login,url is {login_url}.")
        ebackup_workflow_user = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                          'karbor_eBackup_admin')
        if not ebackup_workflow_user:
            raise Exception("Get eBackup workflow user admin failed.")
        ebackup_workflow_user_pwd = self.param_util.get_value_from_cloudparam(self.pod_id, 'CSBS-VBS',
                                                                              'karbor_eBackup_admin_pwd')
        if not ebackup_workflow_user_pwd:
            raise Exception("Get eBackup workflow user password failed.")
        self.login(login_url, ebackup_workflow_user, ebackup_workflow_user_pwd)

        # 获取ebackup和az对应关系
        azebackup_settings_url = self.base_url + '/azebackup_settings'
        logger.info(f"Start to get azebackup_settings,url is {azebackup_settings_url}.")
        az_ebackup_map_list = self.get_az_ebackup_info(azebackup_settings_url)

        # 登出workflow
        self.logout()

        # 获取ebackup account
        ebackup_account = self.get_ebackup_account()

        # 配置ebackup和az对应关系到karbor上
        logger.info(f"Start to set ebackup server.")
        config_failure_list = []
        for az_ebackup_map in az_ebackup_map_list:
            ebackup_mg_ip = az_ebackup_map["eBackupIp"]
            ebackup_az = az_ebackup_map["AZName"]
            if ebackup_az in ["manage-az", "dr-manage-az"]:
                continue
            try:
                result = self.exec_ebackup_cmd(ebackup_mg_ip, ebackup_az, ebackup_account.pwd)
            except Exception as error:
                raise Exception(f"Failed to config ebackup server.{str(error)}") from error
            logger.info(f"set eBackup server success: {result}.")
            if not result:
                config_failure_list.append({ebackup_mg_ip: ebackup_az})
        if len(config_failure_list) > 0:
            raise Exception(f"Failed to config ebackup server:{config_failure_list}.")
        logger.info('Config ebackup server success.')
        return Message(200)

    @auto_retry(max_retry_times=3, delay_time=60)
    def exec_ebackup_cmd(self, ebackup_mg_ip, ebackup_az, pwd):
        return set_ebackup_server(self.node, ebackup_mg_ip, ebackup_az, pwd)
