#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.constant import KARBOR_IMAGE_NAME
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.upgrade.params import ParamsTools
from plugins.CSBS.common.upgrade.upgrade_os import UpgradeOS
from plugins.CSBS.common.util import MSG_500, MSG_200, auto_retry
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob
from plugins.CSBS.scripts.upgrade.karbor.sub_job_pre_check_karbor import PreCheckKarbor

logger.init("karbor UpgradeOS")


class UpgradeOSStep(BaseSubJob):
    def __init__(self, project_id=None, pod_id=None, regionid_list=None):
        super(UpgradeOSStep, self).__init__(project_id, pod_id, regionid_list)
        self.params_tools = ParamsTools(project_id)
        self.node_list = self.params_tools.get_karbor_node_list()
        self.upgrade_util = UpgradeOS(self.project_id, self.pod_id, self.node_list)
        self.karbor_operation = KarborOperation(self.project_id)
        self.karbor_pre_check = PreCheckKarbor(project_id=project_id, pod_id=pod_id, regionid_list=regionid_list)

    def execute(self, project_id, pod_id, regionid_list=None):
        result = self._check_karbor_service()
        if not result:
            return Message(MSG_500, HCCIException("645067"))
        logger.info("Start to upgrade os.")
        if not self.upgrade_util.install():
            logger.error(f"Filed to upgrade os.")
            return Message(MSG_500, HCCIException("645062"))
        logger.info("Upgrade os success.")
        self.upgrade_util.upload_kvm(KARBOR_IMAGE_NAME)
        return Message(MSG_200)

    @auto_retry()
    def _check_karbor_service(self):
        logger.info("Start to check service status.")
        check_results = []
        check_config_dic = self.params_store.get_check_config("PreCheckKarbor")
        try:
            check_results = self.karbor_operation.check_services(check_results, check_config_dic["KarborService"])
        except Exception as err:
            logger.error("Check Karbor service failed, reason: {}.".format(str(err)))
            raise HCCIException(645067) from err
        for check_result in check_results:
            logger.info(f"Check {check_result.itemname_en} component, status is: {check_result.status}.")
            if check_result.status != 'success' and self._check_component(check_result):
                logger.info(f"Check {check_result.itemname_en} component failed.")
                raise HCCIException(645067)
        logger.info("Check all service component success.")
        return True

    def _check_component(self, check_result):
        for component in ["karbor-operation", "rabbitmq"]:
            if check_result.itemname_en.startswith(component):
                return False
        return True

