#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import time

import utils.common.log as logger
from utils.business.vm_util import PublicServiceOM
from utils.common.exception import HCCIException

from plugins.CSBS.common import node_installer
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.os_util import OSUtil

logger.init("UpgradeOS")
CHECK_IS_REBOOT = '[ "$(uname -r)" == "$(grep G_ISO_VERSION= /mnt/upgrade/upgrade_iso.sh | awk -F "=" \'{print $2}\'' \
                  ' | sed \'s/"//g\')" ]; echo "reboot_status:$?"'


class UpgradeOS(node_installer.ThreadInstaller):
    def __init__(self, project_id, pod_id, node_list):
        super(UpgradeOS, self).__init__(node_list=node_list)
        self.project_id = project_id
        self.pod_id = pod_id
        self.pkg_pre_name = "OceanStor BCManager"
        self.param_tool = ParamTool(project_id, pod_id)
        self.vm_util = PublicServiceOM()
        self._ssh = SshClient()
        self._iso_file_name = f"OceanStor_BCManager_EulerOS_{self.param_tool.get_os_arch()}.iso"

    def install_thread(self, node):
        os_util = OSUtil(self.project_id, self.pod_id, node)
        if not os_util.need_to_upgrade():
            logger.info("Current node doesn't need to upgrade os.")
            self.install_result[node.node_ip] = True
            return
        ssh_client = self._ssh.get_ssh_client_user_su_root(node)
        try:
            if not os_util.check_iso_mounted(ssh_client):
                os_util.mount_iso(ssh_client)
            # 执行pre_check
            if self._pre_check(ssh_client, node):
                self.install_result[node.node_ip] = True
                return
            # 执行升级
            self._upgrade(ssh_client, node)
            # 轮训查询升级状态
            self._check_upgrade_state(node)
        except Exception as e:
            self.install_result[node.node_ip] = False
            logger.error(f"Upgrade failed! error: {e}")
            return
        finally:
            if ssh_client:
                self._ssh.ssh_close(ssh_client)
        self.install_result[node.node_ip] = True

    def rollback_thread(self, node):
        logger.info("Start to rollback")
        try:
            self._rollback(node)
            # 挂在镜像文件
            self.clean_upgrade_file(node)
        except Exception as e:
            logger.error(f"Rollback failed! error: {e}")
            self.rollback_result[node.node_ip] = False
            return
        self.rollback_result[node.node_ip] = True

    def clean_upgrade_file(self, node):
        logger.info("Start to clean all upgrade files.")
        ssh_client = self._ssh.get_ssh_client_user_su_root(node)
        os_util = OSUtil(self.project_id, self.pod_id, node)
        os_util.clean_all(ssh_client)
        self._ssh.ssh_close(ssh_client)
        logger.info("Success to clean all upgrade files.")

    def upload_kvm(self, vm_image_name):
        back_image = self.vm_util.get_image_info_by_name_id(pod_id=self.pod_id, image_name=f'{vm_image_name}_backup')
        if back_image:
            return
        image_file_name = OSUtil.get_kvm_image_file(self.project_id)
        self.vm_util.update_image_attributes(pod_id=self.pod_id, image_name=vm_image_name,
                                             name=f'{vm_image_name}_backup')
        self.vm_util.create_image(pod_id=self.pod_id, name=vm_image_name, image_file_name=image_file_name,
                                  protected=False, hw_disk_bus=None,
                                  virtual_env_type="KVM", __support_kvm='true')

    def _check_upgrade(self, ssh_client, os_util):
        """
        升级后检查
        如果当前节点os版本已经是最新的了，直接返回True。否则抛出异常
        """
        if not os_util.check_iso_mounted(ssh_client):
            os_util.mount_iso(ssh_client)
        logger.info(f'Start to precheck on node: {ssh_client["ip"]}.')
        try:
            result = self._ssh.ssh_exec_command_return(ssh_client, CHECK_IS_REBOOT)
        except Exception as e:
            raise HCCIException(645062, f"Upgrade os failed! current error msg: {e}.") from e
        finally:
            if ssh_client:
                self._ssh.ssh_close(ssh_client)
        if self._ssh.success_to_return(result, 'reboot_status:0'):
            logger.info(f'Upgrade os success on node: {ssh_client["ip"]}')
            return True
        raise HCCIException(645062, "Upgrade os failed! pls check upgrade log.")

    def _pre_check(self, ssh_client, node):
        """
        升级前检查
        如果当前节点os版本已经是最新的了，直接返回True。否则返回False.
        """
        logger.info(f'Start to precheck on node: {node.node_ip}.')
        cmd = f"sh /mnt/upgrade/upgrade_iso.sh precheck /root/{self._iso_file_name}"
        result = self._ssh.ssh_exec_command_return(ssh_client, cmd)
        if self._ssh.success_to_return(result, 'is already new version'):
            logger.info(f'The os version is already new version on node: {node.node_ip}')
            return True
        if not self._ssh.is_ssh_cmd_executed(result):
            logger.error(f"Execute precheck result: {result}.")
            raise Exception(str(result))
        logger.info("Success to precheck.")
        return False

    def _upgrade(self, ssh_client, node):
        logger.info(f'Start to upgrade on node: {node.node_ip}.')
        upgrade_cmd = f"bash -c 'sh /mnt/upgrade/upgrade_iso.sh upgrade " \
                      f"/root/{self._iso_file_name} " \
                      f"1>>/var/log/upgrade_iso.log.`date +%s` 2>&1 &'"
        result = self._ssh.ssh_send_command_expect(ssh_client, upgrade_cmd)
        logger.info(f"Success to send upgrade cmd. result: {result}")

    def _check_upgrade_state(self, node):
        # 超时时间设定为60min
        timeout = 60 * 60
        interval = 120
        _time = 0
        os_util = OSUtil(self.project_id, self.pod_id, node)
        while _time <= timeout:
            ssh_client = None
            _time += interval
            time.sleep(interval)
            try:
                ssh_client = self._ssh.get_ssh_client_user_su_root(node)
                result = os_util.get_current_state(ssh_client)
                logger.info(f"Current upgrade state: {result} on node: {node.node_ip}.")
                if self._ssh.success_to_return(result, 'SUCCESS') and self._check_upgrade(ssh_client, os_util):
                    # 返回SUCCESS时，仍然需要检查节点的os版本是否正确，避免在重启前就下发了查询请求。
                    break
                if self._ssh.success_to_return(result, 'FAILED'):
                    raise Exception("Upgrade os failed! Current state is FAILED")
            except HCCIException as ex:
                logger.error(f"Get current failed state! {str(ex)}")
            except Exception as e:
                logger.error(f"Get current state has error! {str(e)}")
                raise e
            finally:
                if ssh_client:
                    self._ssh.ssh_close(ssh_client)

    def _rollback(self, node):
        """检查挂载路径是否还存在，如果不存在，当前节点升级失败或未升级"""
        ssh_client = None
        try:
            ssh_client = self._ssh.get_ssh_client_user_su_root(node)
            result = self._ssh.ssh_exec_command_return(ssh_client, "test -e /opt/.osbak")
            # 检查当前备份文件是否还存在，如果不存在就已经完成回退
            if not self._ssh.is_ssh_cmd_executed(result):
                return
            os_util = OSUtil(self.project_id, self.pod_id, node)
            logger.info("Check upgrade_iso file exists.")
            if not os_util.check_iso_mounted(ssh_client):
                os_util.mount_iso(ssh_client)
            self._rollback_os(node)
        except Exception as e:
            err_msg = f"Rollback os failed. error: {str(e)}"
            logger.error(err_msg)
            raise Exception(err_msg) from e
        finally:
            if ssh_client:
                self._ssh.ssh_close(ssh_client)

    def _rollback_os(self, node):
        logger.info(f"Start to rollback OS version on node: {node.node_ip}")
        ssh_client = self._ssh.get_ssh_client_user_su_root(node)
        # rollback会导致虚拟机重启, ssh必定会抛出异常
        try:
            self._ssh.ssh_exec_command_return(ssh_client, "sh /mnt/upgrade/upgrade_iso.sh rollback", timeout=3)
        except Exception as ex:
            logger.info(f'Rollback start, wait for vm restart. msg:{ex}')
        # 等待虚拟机重启
        time.sleep(60)
        self._wait_rollback(node)
        logger.info("Success rollback OS version.")

    def _wait_rollback(self, node):
        # 回退10分钟
        timeout = 10 * 60
        _time = 0
        _interval = 2 * 60
        logger.info("Wait vm reboot, check rollback state.")
        while _time <= timeout:
            _time += _interval
            time.sleep(_interval)
            ssh_client = None
            try:
                ssh_client = self._ssh.get_ssh_client_user_su_root(node)
                result = self._ssh.ssh_exec_command_return(ssh_client, 'uname -r')
                logger.info(f"Current os version: {result} on node: {node.node_ip}.")
                if self._ssh.is_ssh_cmd_executed(result):
                    logger.info("OS rollback success.")
                    break
            except Exception as e:
                logger.warn(f"Check rollback state failed! error: {str(e)}")
            finally:
                if ssh_client:
                    self._ssh.ssh_close(ssh_client)
