import yaml

import utils.common.log as logger
from utils.business.dmk_util import DmkApi
from utils.common.message import Message

from plugins.CSBS.common.karbor_cpshelper import CPSInfo, CpsHelper
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.common.util import check_url_param
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")


class ConfigNeutronEndpointsToKarbor(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super().__init__(project_id, pod_id, regionid_list)
        self.ssh_client = SshClient()
        cps_node = CPSInfo(pod_id).get_cps_node()
        self.cps_helper = CpsHelper(cps_node)
        self.karbor_operation = KarborOperation(project_id)
        self.karbor_node = self.karbor_operation.karbor_node_list[0]

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            return self.config_neutron_endpoint()
        except Exception as err:
            logger.error(f"Execute error:{err}.")
            return Message(500, f"Execute error:{err}.")

    def config_neutron_endpoint(self):
        apicom_endpoint = self._get_apicom_url()
        logger.info(f"The apicom_endpoint is {apicom_endpoint}.")
        check_url_param(apicom_endpoint)
        neutron_endpoint = self.cps_helper.get_neutron_endpoint()
        if not neutron_endpoint:
            raise Exception("Param:neutron_endpoint is None.")
        logger.info(f'Get param neutron_endpoint: {neutron_endpoint}.')
        check_url_param(neutron_endpoint)

        logger.info(f"Start to set neutron endpoints.")
        cmd = f"source /opt/huawei/dj/inst/utils.sh;" \
              f"set_karbor_endpoints --neutron_endpoint {neutron_endpoint} --apicom_endpoint {apicom_endpoint}"
        ssh_client = self.ssh_client.get_ssh_client(self.karbor_node)
        result = self.ssh_client.ssh_send_command_expect(ssh_client, cmd, "successful", 30)
        self.ssh_client.ssh_close(ssh_client)
        if self.ssh_client.failed_to_return(result, "successful", ssh_client):
            raise Exception("Failed to config neutron endpoints.")
        logger.info('Config neutron endpoints success.')
        return Message(200)

    def _get_apicom_url(self):
        dmk_params = self.params_store.get_dmk_params()
        dmk_util = DmkApi()
        dmk_util.login_dmk(dmk_params.float_ip, dmk_params.ui_username, dmk_params.ui_password)
        public_config = dmk_util.get_dmk_public_config()
        logger.info("Get dmk public config successfully.")
        public_var = yaml.safe_load(public_config)
        api_com_domain = public_var['g_current_region']['apicom']['service']
        if not api_com_domain:
            raise Exception("Failed to obtain the apicom domain.")
        return f"https://{api_com_domain}"
