# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re

import utils.common.log as logger
from utils.common.check_result import CheckResult
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.os_util import OSUtil
from plugins.CSBS.common.params_tool import ParamTool
from plugins.CSBS.common.ssh_client import SshClient
from plugins.CSBS.common.upgrade.karbor import KarborOperation
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("Karbor PreCheckVol")

check_item = {
    "os_upgrade_precheck_status": {"name_cn": "OS可升级检查", "name_en": "Check whether the OS can be upgraded.",
                                   "error_code": '645064'}
}


class PreCheckVol(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(PreCheckVol, self).__init__(project_id, pod_id, regionid_list)
        self.karbor_operation = KarborOperation(self.project_id)
        self.param_tool = ParamTool(self.project_id, self.pod_id)
        self.node_list = self.karbor_operation.karbor_node_list
        self.ssh_util = SshClient()
        self.check_results = []

    def execute(self, project_id, pod_id, regionid_list=None):
        try:
            check_results = self.os_precheck()
        except Exception as upgrade_err:
            logger.error(f"Precheck os failed: {upgrade_err}")
            return Message(500,
                           error_msg_cn=f"OS升级前检查出现异常，请查看日志并重试，异常信息:{upgrade_err}",
                           error_msg_en=f"Exception occurs while prechecking,"
                                        f"please check the log and try again, "
                                        f"error info:{upgrade_err}.")
        return Message(200, check_results=check_results)

    def os_precheck(self):
        for node in self.node_list:
            self._check(node)
        if not self.check_results:
            check_result = CheckResult(itemname_ch=check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                                       itemname_en=check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                                       status="success")
            self.check_results.append(check_result)
        return self.check_results

    def _check(self, node):
        os_util = OSUtil(node=node, project_id=self.project_id, pod_id=self.pod_id)
        logger.info(f"Start precheck iso on node({node.node_ip})")
        ssh_client = None
        try:
            ssh_client = self.ssh_util.get_ssh_client_user_su_root(node)
            os_arch = self.param_tool.get_os_arch()
            if not os_util.check_iso_mounted(ssh_client):
                os_util.mount_iso(ssh_client)
            precheck_cmd = f"sh /mnt/upgrade/upgrade_iso.sh precheck " \
                           f"/root/OceanStor_BCManager_EulerOS_{os_arch}.iso"
            self.ssh_util.ssh_exec_command_return(ssh_client, precheck_cmd)
            state, description = os_util.get_current_state(ssh_client)
        except HCCIException as ex:
            logger.error(f"Precheck iso failed! {str(ex)}")
            check_result = CheckResult(
                itemname_ch=check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                itemname_en=check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                status="failure",
                error_msg_cn=HCCIException(
                    check_item.get('os_upgrade_precheck_status', dict()).get('error_code'),
                    node.node_ip, str(ex)))
            self.check_results.append(check_result)
            return
        finally:
            if ssh_client:
                self.ssh_util.ssh_close(ssh_client)
        if state == 'SUCCESS':
            logger.info(f"Precheck iso success on node({node.node_ip})")
        elif state == 'FAILED' and re.search("Current OS version\[.*\] is not compatible.", description):
            logger.info(f"The os version is not compatible with current iso on node({node.node_ip}). Skip upgrade.")
            os_util.add_node_ip_to_cfg(node.node_ip)
        elif state == 'FAILED' and "is already new version" in description:
            logger.info(f"The os version is already latest version on node({node.node_ip}). Skip upgrade.")
            os_util.add_node_ip_to_cfg(node.node_ip)
        else:
            logger.error(f"Precheck os failed. Error reason is {description}.")
            check_result = CheckResult(
                itemname_ch=check_item.get('os_upgrade_precheck_status', dict()).get('name_cn'),
                itemname_en=check_item.get('os_upgrade_precheck_status', dict()).get('name_en'),
                status="failure",
                error_msg_cn=HCCIException(
                    check_item.get('os_upgrade_precheck_status', dict()).get('error_code'),
                    node.node_ip, description))
            self.check_results.append(check_result)
