import os
import shutil
import time

import utils.common.log as logger
from utils.common.exception import HCCIException
from utils.common.message import Message

from plugins.CSBS.common.upgrade.console_upgrade import ConsoleUpgrade
from plugins.CSBS.common.util import unzip_file
from plugins.CSBS.scripts.upgrade.karbor.base import BaseSubJob

logger.init("CSBS-VBS")
RETRY_TIME_LIMIT = 3


class UpgradeConsole(BaseSubJob):
    def __init__(self, project_id, pod_id, regionid_list):
        super(UpgradeConsole, self).__init__(project_id, pod_id, regionid_list)
        self.console_upgrade = ConsoleUpgrade(self.params_store)
        self.unzip_pkg_path = os.path.realpath(self.upgrade_path.pkg_path + "/csbs_console_temp")
        self.console_pkg_name = f"cbs-{self.upgrade_path.console_pkg_version}.tar.gz"
        self.console_pkg_path = os.path.join(self.unzip_pkg_path, self.console_pkg_name)

    def execute(self, project_id, pod_id, regionid_list=None):
        if self.upgrade_path.target_version != self.upgrade_path.console_pkg_version:
            err_msg = "The console package is not the expected one." \
                      "Please check and upload console package of the correct version."
            logger.error(err_msg)
            raise Exception(err_msg)
        try:
            self._unzip_console_pkg()
        except Exception as ex:
            logger.error(ex)
            return Message(500, HCCIException(645000))

        # upload cf2 pkg into primary region ManageOne-Tenant01/02 nodes
        # For the standby region, the value of region_flag is True.
        # For other regions, the value is False.
        region_flag = False
        if self.project_scene.is_global_dr_scene and not self.project_scene.is_primary_region:
            region_flag = True
        if not self.console_upgrade.upload_console_pkg(
                project_id=self.project_id,
                pod_id=self.pod_id,
                region_id=self.region_id,
                console_file=self.console_pkg_path,
                region_flag=region_flag):
            return Message(500, HCCIException(645005))
        try:
            self._upgrade_console(region_flag)
        except Exception as err:
            logger.error(f"Failed to upgrade console, error:{str(err)}.")
            return Message(500, HCCIException(645015))
        self._remove_tmp_console_pkg()
        return Message(200)

    def _remove_tmp_console_pkg(self):
        logger.info("Start to delete the temp unzip console package.")
        if os.path.exists(self.unzip_pkg_path):
            shutil.rmtree(self.unzip_pkg_path)
        logger.info("Succeeded to delete the temp unzip console package.")

    def _unzip_console_pkg(self):
        if os.path.exists(self.unzip_pkg_path):
            shutil.rmtree(self.unzip_pkg_path)
        pkg = os.path.join(self.upgrade_path.pkg_path,
                           self.upgrade_path.console_pkg_name)
        unzip_file(pkg, self.unzip_pkg_path,
                   unzip_size_limit_mb=100,
                   unzip_file_count_limit_kilo=10)

    def _upgrade_console(self, region_flag):
        logger.info(f"Start to upgrade current region console, package name is:{self.console_pkg_name}.")
        retry_times = 0
        while True:
            result = \
                self.console_upgrade.upgrade_cbs_console(
                    pod_id=self.pod_id,
                    console_pkg_name=self.console_pkg_name,
                    region_flag=region_flag,
                    region_id=self.region_id,
                    version=self.region_id,
                    is_region=True
                )
            if not result and retry_times < RETRY_TIME_LIMIT:
                retry_times += 1
                logger.info("Failed to upgrade console, starting to retry,"
                            " and retry_times is {}.".format(retry_times))
                time.sleep(10)
            else:
                break
        if not result:
            err_msg = f"Failed to upgrade current console, after retrying {retry_times} times."
            logger.error(err_msg)
            raise Exception(err_msg)
        logger.info("Upgrade current region console successfully.")
