#!/bin/bash
if [[ ${UID} -eq 0 ]];then
    sudo -u openstack bash "$(readlink -f "$0")" "$@";exit $?
elif [[ ${UID} -ne 51001 ]];then
    echo "Please switch to user(openstack) for run." && exit 1
fi

source /opt/huawei/dj/inst/utils.sh
export LOG_TAG="mo_update_cert"
###############################################################
# 功能说明: MO统一证书替换公共方法
# 错误码：
# 2：备份证书文件失败
# 3：回退证书文件失败
# 4：替换CA证书失败
# 5：替换证书PWD失败
# 6：替换CERT证书失败
# 7：替换证书KEY失败
# 8：校验CA和CERT失败
# 9：校验KEY和PWD失败
# 10： 加密证书私钥失败
# 11： 更新cms证书状态失败
# 12:  备份旧CMS-CERT到证书数据目录失败
# 13:  备份旧CMS-KEY到证书目录失败
# 14:  解密旧KEY到证书目录失败
# 15:  重新生成数据库证书失败
# 16:  重新生成OMM-HA证书失败
##############################################################
CURRENT_TOOL=$(readlink -f "$0")
BACKUP_PATH='/opt/huawei/dj/etc/digital_certificate/digital_certificate.tar.gz'
SERVICE_CA_FILE="/opt/huawei/dj/DJSecurity/server-ca/ca-cert.pem"
SERVICE_PWD_FILE="/opt/huawei/dj/DJSecurity/privkey/privkey.conf"
REPLACE_SERVER="/opt/huawei/dj/DJSecurity/mo_certs/replace_server"
CURRENT_NODE=$(get_info manage_ip)
IFS="," read -r -a NODE_LIST <<< "$(get_info manage_ip list)"
OLD_CER="/opt/huawei/dj/bin/digital_certificate/old.cert"
OLD_KEY="/opt/huawei/dj/bin/digital_certificate/old.key"
OLD_PWD=$(decrypt_pwd "$(awk '{print $NF}' ${SERVICE_PWD_FILE})")

# 实现本节点证书目录备份
function backup_digital_certificate()
{
    log_info "begin backup digital certificate."
    mkdir -p "$(dirname ${BACKUP_PATH})" && rm -f ${BACKUP_PATH}
    if ! (tar -P -czf ${BACKUP_PATH} /opt/huawei/dj/DJSecurity >/dev/null 2>&1);then
        exit 2
    fi
    return 0
}

# 实现本节点证书回退
function rollback_digital_certificate()
{
    log_info "begin rollback digital certificate."
    check_tar_file_before_unzip ${BACKUP_PATH} "/opt/huawei/dj/DJSecurity"
    CHECK_RESULT $? "check digital_certificate.tar.gz failed."
    if ! (tar -P -xzf ${BACKUP_PATH} >/dev/null 2>&1);then
        exit 3
    fi
    return 0
}

# 替换本节点CA证书和密码，失败则回退所有节点证书
function replace_ca_and_password()
{
    log_info "begin update ${SERVICE_CA_FILE} and ${SERVICE_PWD_FILE} certificate."
    if ! ( echo "${CA_CONTENT}" | base64 -d > ${SERVICE_CA_FILE});then
        log_error "Replace ${SERVICE_CA_FILE} failed."
        rollback_digital_certificate_ex && exit 4
    fi

    if ! (check_cert "${SERVICE_CA_FILE}");then
        log_error "Check ${SERVICE_CA_FILE} failed."
        rollback_digital_certificate_ex && exit 5
    fi

    if ! (echo "encrypt_password ${PWD_CONTENT}" > ${SERVICE_PWD_FILE});then
        log_error "Replace privkey failed."
        rollback_digital_certificate_ex && exit 6
    fi
    return 0
}

# 替换本节点组件CERT和KEY，失败则回退所有节点证书
function replace_cert_and_key()
{
    log_info "begin update $1 and $2."
    if ! (echo "${CERT_CONTENT}" | base64 -d > "$1");then
        log_error "Replace $1 failed."
        rollback_digital_certificate_ex && exit 6
    fi
    if ! (echo "${KEY_CONTENT}" | base64 -d > "$2");then
        log_error "Replace $2 failed."
        rollback_digital_certificate_ex && exit 7
    fi
    if ! (openssl verify -CAfile ${SERVICE_CA_FILE} "$1" >/dev/null 2>&1);then
        log_error "Check CA and cert file failed, the replaced certs may be incorrect."
        rollback_digital_certificate_ex && exit 8
    fi

    if ! (echo "$3" | openssl rsa -in "${KEY_FILE}" -passin stdin >/dev/null 2>&1);then
        log_error "Check key failed, the replaced certs may be incorrect."
        rollback_digital_certificate_ex  && exit 9
    fi

    if ! (check_cert "${SERVICE_CA_FILE}");then
        log_error "Check ${SERVICE_CA_FILE} failed."
        rollback_digital_certificate_ex && exit 10
    fi
    return 0
}

# 替换本节点所有组件证书入口
function update_digital_certificate()
{
    log_info "begin certificate on node ${CURRENT_NODE}."
    CA_CONTENT=$(awk '{print $1}' ${REPLACE_SERVER})
    CERT_CONTENT=$(awk '{print $2}' ${REPLACE_SERVER})
    KEY_CONTENT=$(awk '{print $3}' ${REPLACE_SERVER})
    PWD_CONTENT=$(awk '{print $4}' ${REPLACE_SERVER})
    replace_ca_and_password
    KEY_PWD=$(decrypt_pwd "$(awk '{print $NF}' ${SERVICE_PWD_FILE})")
    COMPONENTS="scagent cli-client cps-monitor omm-ha gaussdb haproxy rabbitmq cms zookeeper resource_manager karbor alarm"
    for COMPONENT in ${COMPONENTS};do
        log_info "Begin to replace ${COMPONENT} certs."
        CERT_FILE=/opt/huawei/dj/DJSecurity/server-cert/${COMPONENT}/${COMPONENT}-cert.pem
        KEY_FILE=/opt/huawei/dj/DJSecurity/server-cert/${COMPONENT}/${COMPONENT}-key.pem
        replace_cert_and_key "${CERT_FILE}" "${KEY_FILE}" "${KEY_PWD}"
        if [[ "${COMPONENT}" == "haproxy" ]]; then
            cat "${SERVICE_CA_FILE}" >> "${CERT_FILE}"
            cat "${KEY_FILE}" >> "${CERT_FILE}"
        fi
    done
    [[ "$(get_info node_index)" -eq  3 ]] && return 0
    if ! bash /opt/huawei/dj/tools/gaussdb/generate_database_cert.sh;then
        log_error "generate database cert failed, the replaced certs maybe error."
        rollback_digital_certificate_ex  && exit 15
    fi
    if ! bash /opt/huawei/dj/tools/omm-ha/generate_ommha_cert.sh;then
        log_error "generate omm-ha cert failed, the replaced certs maybe error."
        rollback_digital_certificate_ex  && exit 16
    fi
    return 0
}

# 回退本节点证书成功后，回退其他节点证书
function rollback_digital_certificate_ex()
{
    rollback_digital_certificate
    log_info "rollback certificate success on node ${CURRENT_NODE}"
    for node in "${NODE_LIST[@]}";do
        [[ "${node}" == "${CURRENT_NODE}" ]] && continue
        cmd_manager --node_ip "${node}" --cmd "${CURRENT_TOOL}" --parameters "rollback"
        CHECK_RESULT $? "Rollback certificate failed on node ${node}."
        log_info "rollback certificate success on node ${node}"
    done
    return 0
}

# 备份本节点证书成功后，备份其他节点证书
function backup_digital_certificate_ex()
{
    backup_digital_certificate
    log_info "Backup certificate success on node ${CURRENT_NODE}"
    for node in "${NODE_LIST[@]}";do
        [[ "${node}" == "${CURRENT_NODE}" ]] && continue
        cmd_manager --node_ip "${node}" --cmd "${CURRENT_TOOL}" --parameters "backup"
        CHECK_RESULT $? "Backup certificate failed on node ${node}."
        log_info "Backup certificate success on node ${node}"
    done
    return 0
}

# 替换其他节点证书成功后，替换本节点证书
function update_digital_certificate_ex()
{
    for node in "${NODE_LIST[@]}";do
        [[ "${node}" == "${CURRENT_NODE}" ]] && continue
        cmd_manager --node_ip "${node}" --cmd "${CURRENT_TOOL}" --parameters "update"
        CHECK_RESULT $? "Update certificate failed on node ${node}."
        log_info "Update certificate success on node ${node}"
    done
    update_digital_certificate
    log_info "Update certificate success on node ${CURRENT_NODE}"
    return 0
}

# 更新CMS配置中证书任务状态为成功，更新失败则回退所有节点证书
function update_task_status()
{
    export CSBS_PYTHON_DATA1="${OLD_CER}"
    export CSBS_PYTHON_DATA2="${OLD_KEY}"
    export CSBS_PYTHON_DATA3="${OLD_PWD}"
    export CSBS_PYTHON_DATA4="${CURRENT_NODE}"
    (
        echo "import os"
        echo "import ast"
        echo "import json"
        echo "import ssl"
        echo "from http import client"
        echo "try:"
        echo "    old_cer = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    old_key = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    old_pwd = os.getenv('CSBS_PYTHON_DATA3')"
        echo "    node_ip = os.getenv('CSBS_PYTHON_DATA4')"
        echo "    headers = {'Content-Type': 'application/json'}"
        echo "    get_url = '/cms/v1/get/iteminfo?key=cert'"
        echo "    context = ssl.SSLContext(ssl.PROTOCOL_SSLv23)"
        echo "    context.load_cert_chain(certfile=old_cer, keyfile=old_key, password=old_pwd)"
        echo "    connection = client.HTTPSConnection(node_ip, port=28098, context=context)"
        echo "    connection.request(method='GET', url=get_url, headers=headers)"
        echo "    resp = connection.getresponse()"
        echo "    if resp.status > 300:"
        echo "        exit(0 if resp.status == 404 else resp.status)"
        echo "    resp_data = ast.literal_eval(resp.read().decode())"
        echo "    cert_tasks = ast.literal_eval(resp_data.get('item_value'))"
        echo "    for cert_task in cert_tasks:"
        echo "        if cert_task.get('cert_name') == 'CSBS_VBS-internal':"
        echo "            cert_task['data']['updateStatus'] = 'success'"
        echo "            break"
        echo "    resp_data['item_value'] = cert_tasks"
        echo "    put_url = '/cms/v1/put/iteminfo'"
        echo "    connection.request(method='PUT', url=put_url, headers=headers, body=json.dumps(resp_data))"
        echo "    resp = connection.getresponse()"
        echo "    exit(resp.status if resp.status > 300 else 0)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}

function update_task_status_ex()
{
    if ! update_task_status;then
        log_error "update task status failed, the replaced certs maybe error."
        rollback_digital_certificate_ex && exit 11
    fi
    return 0
}

# 提前备份CMS-CERT和CMS-KEY，用于在所有节点证书替换完成之后，重启之前更新Cms配置和跨节点触发服务重启
function backup_old_cert_key()
{
    mkdir -p "$(dirname ${OLD_CER})"
    if ! (/usr/bin/cp -f /opt/huawei/dj/DJSecurity/server-cert/cms/cms-cert.pem ${OLD_CER});then
        exit 13
    fi
    if ! (/usr/bin/cp -f /opt/huawei/dj/DJSecurity/server-cert/cms/cms-key.pem ${OLD_KEY});then
        exit 13
    fi
    if ! (echo "${OLD_PWD}" | openssl rsa -in ${OLD_KEY} -passin stdin >/dev/null 2>&1);then
        exit 14
    fi
    return 0
}

# 触发其他节点90s延时重启服务后，触发本节点90s延时重启服务
function restart_service()
{
    export CSBS_PYTHON_DATA1="${OLD_CER}"
    export CSBS_PYTHON_DATA2="${OLD_KEY}"
    export CSBS_PYTHON_DATA3="${OLD_PWD}"
    export CSBS_PYTHON_DATA4="$1"
    (
        echo "import os"
        echo "import json"
        echo "import ssl"
        echo "from http import client"
        echo "try:"
        echo "    old_cer = os.getenv('CSBS_PYTHON_DATA1')"
        echo "    old_key = os.getenv('CSBS_PYTHON_DATA2')"
        echo "    old_pwd = os.getenv('CSBS_PYTHON_DATA3')"
        echo "    node_ip = os.getenv('CSBS_PYTHON_DATA4')"
        echo "    headers = {'Content-Type': 'application/json'}"
        echo "    put_url = '/scagent/v1/command'"
        echo "    put_body = {'command': {'name':'/opt/huawei/dj/tools/digital_certificate/restart_backup_service.sh'}}"
        echo "    context = ssl.SSLContext(ssl.PROTOCOL_SSLv23)"
        echo "    context.load_cert_chain(certfile=old_cer, keyfile=old_key, password=old_pwd)"
        echo "    connection = client.HTTPSConnection(node_ip, port=28688, context=context)"
        echo "    connection.request(method='PUT', url=put_url, headers=headers, body=json.dumps(put_body))"
        echo "    resp = connection.getresponse()"
        echo "    exit(resp.status if resp.status > 300 else 0)"
        echo "except Exception:"
        echo "    exit(1)"
    ) | csbs_python
}

function restart_service_ex()
{
    for node in "${NODE_LIST[@]}";do
        [[ "${node}" == "${CURRENT_NODE}" ]] && continue
        restart_service "${node}"
        log_info "Set 90 seconds later to restart service  on node ${node} ret_code: $?."
    done
    bash /opt/huawei/dj/tools/digital_certificate/restart_backup_service.sh
    log_info "Set 90 seconds later to restart service on node ${CURRENT_NODE} ret_code: $?."
    return 0
}

function get_cert_time()
{
    cert_path=$1
    cer_date=$(openssl x509 -subject_hash -fingerprint -in "${cert_path}" -noout -text | grep -w 'Not After')
    IFS=" " read -r -a cer_date_arr <<< "${cer_date}"
    cer_date_time=$(date --date="${cer_date_arr[3]} ${cer_date_arr[4]} ${cer_date_arr[6]} ${cer_date_arr[5]}" +%s)
    echo "${cer_date_time}"
}

function get_cert_diff_time()
{
    cer_date_time=$(get_cert_time "$1")
    cur_date_time=$(date +%s)
    diff_time=$((cer_date_time-cur_date_time))
    echo "${diff_time}"
}

# 校验证书
function check_cert()
{
    cert_file=$1

    if ! openssl x509 -subject_hash -fingerprint -in "${cert_file}" -noout -text &>/dev/null;then
        log_error "${cert_file} can not use openssl to check."
        return 1
    fi

    # 校验证书版本
    version=$(openssl x509 -noout -in "${cert_file}" -text | grep 'Version' | awk '{print $2}')
    if [[ ${version} -lt 3 ]]; then
        log_error "Check CA and cert file failed, the version error."
        return 1
    fi

    # 校验证书签名
    signature=$(openssl x509 -noout -in "${cert_file}" -text | grep 'Signature Algorithm' | awk '{print $3}'| head -n 1)
    if [[ "${signature}" != "sha256WithRSAEncryption" ]]; then
        log_error "Check CA and cert file failed, the signature error."
        return 1
    fi

    # 校验证书公钥长度
    key_len=$(openssl x509 -noout -in "${cert_file}" -text | grep 'Public-Key' | awk '{print $3}'| awk -v FS='(' '{print $2}')
    if [[ ${key_len} -lt 2048 ]]; then
        log_error "Check CA and cert file failed, the key_len error."
        return 1
    fi

    # 校验证书有效期
    diff_time=$(get_cert_diff_time "${cert_file}")
    if [[ ${diff_time} -lt 0 ]];then
        log_error "Check CA and cert file failed, the cert expired."
        return 1
    fi
    return 0
}

# 防止并发
if [[ "$(pgrep -U 51001 -f "$0" -d ' ')" != "$$" ]];then
     CHECK_RESULT 1 "The script is running, no need start again."
fi

# 证书更新脚本入口
if [[ $# -eq 1 ]] && [[ "$1" == "rollback" ]];then
    rollback_digital_certificate
elif [[ $# -eq 1 ]] && [[ "$1" == "update" ]];then
    update_digital_certificate
elif [[ $# -eq 1 ]] && [[ "$1" == "backup" ]];then
    backup_digital_certificate
else
    backup_old_cert_key
    backup_digital_certificate_ex
    update_digital_certificate_ex
    update_task_status_ex
    restart_service_ex
fi
