#!/bin/bash
set +x
CURRENT_PATH=$(dirname "$(readlink -f "$0")")
LINE_ROOT=$(dirname "${CURRENT_PATH}")
source "${LINE_ROOT}"/bin/utils.sh
CERT_PATH="${LINE_ROOT}/server-cert"
mapfile -t -d ',' NODE_LIST < <(get_info.py agentproxy_nodes | xargs echo -n)
LOCAL_IP=$(get_info.py manage_ip)
export LOG_TAG="proxy_install"


function create_root_ca()
{
    mkdir -p "${CERT_PATH}" "${LINE_ROOT}"/privkey "${CERT_PATH}"/demoCA/newcerts
    key_pwd=$(mkpassword 30)
    export PLAIN_TEXT=${key_pwd}
    encrypt_password=$(proxython -c "import kmc.kmc;A=kmc.kmc.API();print(A.encrypt(0))")
    unset PLAIN_TEXT
    echo "encrypt_password ${encrypt_password}" > "${LINE_ROOT}"/privkey/privkey.conf
    touch "${CERT_PATH}"/demoCA/index.txt
    rand_serial=$(openssl rand -base64 8 | sha256sum | cut -c1-8)
    echo "${rand_serial}" > "${CERT_PATH}"/demoCA/serial
    cd "${CERT_PATH}" || exit 1
    echo "${key_pwd}" | openssl genrsa -passout stdin -out "${CERT_PATH}"/ca-key.pem -aes256 2048
    CHECK_RESULT $? "create internal ca ${CERT_PATH} key failed"
    log_info "create root ca ${CERT_PATH} key successfully"
    echo "${key_pwd}" | openssl req -new -x509 -sha256 -key "${CERT_PATH}"/ca-key.pem -out "${CERT_PATH}"/ca-cert.pem -days 3650 \
    -subj "/C=CN/ST=SiChuan/O=Huawei/CN=CBS" -extensions v3_ca -passin stdin
    CHECK_RESULT $? "create internal ca ${CERT_PATH} cert failed"
    log_info "generate root ca certificate finished"
}

function create_key_and_csr()
{
    csr_cn=$1
    cert_passwd=$2
    echo "" > "${CERT_PATH}"/demoCA/index.txt.attr
    log_info "modify unique_subject value"
    echo "unique_subject = no" > "${CERT_PATH}"/demoCA/index.txt.attr
    echo "${cert_passwd}" | openssl genrsa -passout stdin -aes256 -out "${CERT_PATH}/work-key.pem" 2048
    CHECK_RESULT $? "there is no cert key for ${csr_cn}"
    log_info "ready to create work certification request"
    echo "${cert_passwd}" | openssl req -new -sha256 -key "${CERT_PATH}/work-key.pem" -out "${CERT_PATH}/work-csr.pem" -days 3650 \
    -subj "/C=CN/ST=SiChuan/O=Huawei/CN=${csr_cn}" -passin stdin
    CHECK_RESULT $? "Create certification request for work failed"
    log_info "Create certification request for work successfully."
}

function create_work_cert()
{
    cert_passwd=$1
    cd "${CERT_PATH}" || exit 1
    log_info "ready to sign certification for work"
    echo "${cert_passwd}" | openssl ca -batch -in "${CERT_PATH}/work-csr.pem" -cert "${CERT_PATH}"/ca-cert.pem -keyfile "${CERT_PATH}"/ca-key.pem \
    -out "${CERT_PATH}/work-cert.pem" -md sha256 -days 3650 -passin stdin
    CHECK_RESULT $? "sign the certification failed"
    log_info "sign the certification successfully"
}

function generate_certificate()
{
    conf_path="${LINE_ROOT}/conf"
    component=("http" "rpc" "tcp" "haproxy" "omm-ha")
    comp_path=("agentproxy/https_cert" "agentproxy/rpc_cert" "agentproxy/tcp_cert" "haproxy" "omm-ha")
    for i in "${!component[@]}"; do
        mkdir -p "${conf_path}/${comp_path[i]}"
        if [ "${component[i]}" == "haproxy" ];then
            cat "${CERT_PATH}/work-key.pem" >> "${CERT_PATH}/work-cert.pem"
            /usr/bin/cp -f "${CERT_PATH}/ca-cert.pem" "${conf_path}/${comp_path[i]}/${component[i]}_ca.pem"
            /usr/bin/cp -f "${CERT_PATH}/work-cert.pem" "${conf_path}/${comp_path[i]}/${component[i]}_ca-cert.pem"
        elif [ "${component[i]}" == "omm-ha" ]; then
            /usr/bin/cp -f "${CERT_PATH}/ca-cert.pem" "${conf_path}/${comp_path[i]}/ha_ca-cert.pem"
            /usr/bin/cp -f "${CERT_PATH}/work-cert.pem" "${conf_path}/${comp_path[i]}/ha_cert.pem"
            /usr/bin/cp -f "${CERT_PATH}/work-key.pem" "${conf_path}/${comp_path[i]}/ha_key.pem"
        else
            /usr/bin/cp -f "${CERT_PATH}/ca-cert.pem" "${conf_path}/${comp_path[i]}/${component[i]}_ca.pem"
            /usr/bin/cp -f "${CERT_PATH}/work-cert.pem" "${conf_path}/${comp_path[i]}/${component[i]}_ca-cert.pem"
            /usr/bin/cp -f "${CERT_PATH}/work-key.pem" "${conf_path}/${comp_path[i]}/${component[i]}_ca-key.pem"
        fi
    done
    log_info "generate certificate successfully"
}

function scp_cmd()
{
    pwd_en=$(get_pwd default_user_pwd "${LINE_ROOT}/conf/util.conf")
    export ENCRYPT_DATA=${pwd_en}
    password=$(proxython -c "import kmc.kmc;A=kmc.kmc.API();print(A.decrypt(0))")
    unset ENCRYPT_DATA
    check_password "${password}"
    CHECK_RESULT $? "the password is invalid."
    /usr/bin/expect<<-EOF
    set timeout 60
    spawn scp -o StrictHostKeyChecking=no -r "$1" "$2" dpamanager@$3:/home/dpamanager
    expect {
        -nocase "password:" {send "${password}\r"}
    }
    expect {
        -nocase "password:" {exit 1}
    }
    lassign [wait] pid spawnid os_error_flag value
    exit \$value
EOF
}

function sync_certificate()
{
    log_info "begin to sync cert file to ${NODE_LIST[1]}"
    scp_times=0
    while [[ ${scp_times} -lt 360 ]]
    do
        if scp_cmd "${CERT_PATH}" "${LINE_ROOT}/privkey/privkey.conf" "${NODE_LIST[1]}";then
            log_info "sync cert file to ${NODE_LIST[1]} successfully."
            break
        fi
        log_warn "failed to sync cert file to ${NODE_LIST[1]}, retry: ${scp_times}."
        sleep 5
        scp_times=$((scp_times+1))
    done
    if [[ "${scp_times}" == 360 ]];then
        log_error "sync cert file failed."
        exit 1
    fi
}

function main()
{
    if [[ "${LOCAL_IP}" == "${NODE_LIST[0]}" ]]; then
        create_root_ca
        key_pwd=$(awk '{print $NF}' "${LINE_ROOT}"/privkey/privkey.conf)
        export ENCRYPT_DATA=${key_pwd}
        decrypt_pwd=$(proxython -c "import kmc.kmc;A=kmc.kmc.API();print(A.decrypt(0))")
        unset ENCRYPT_DATA
        create_key_and_csr "KarborProxy" "${decrypt_pwd}"
        create_work_cert "${decrypt_pwd}"
        sync_certificate
        CHECK_RESULT $? "sync cert file failed!"
        /usr/bin/cp -f "${LINE_ROOT}"/privkey/privkey.conf "${LINE_ROOT}"/conf
        rm -rf "${LINE_ROOT}"/privkey
    else
        wait_times=0
        while [[ ${wait_times} -lt 360 ]]
        do
            if [[ -f /home/dpamanager/server-cert/ca-cert.pem ]];then
                log_info "cert file is existed."
                break
            fi
            log_info "waiting sync cert file..."
            sleep 5
            wait_times=$((wait_times+1))
        done
        if [[ ${wait_times} == 360 ]];then
            log_error "waiting timeout."
            exit 1
        fi
        /usr/bin/mv -f /home/dpamanager/server-cert "${LINE_ROOT}"
        /usr/bin/mv -f /home/dpamanager/privkey.conf "${LINE_ROOT}"/conf
    fi
    generate_certificate
    rm -rf "${CERT_PATH}"
}

main
