#!/bin/bash
CURRENT_PATH=$(dirname "$(readlink -f "$0")")
LINE_ROOT=$(dirname "${CURRENT_PATH}")
ROOT_PATH=$(dirname "${LINE_ROOT}")
OPT_PATH="/opt"
source "${LINE_ROOT}/bin/utils.sh"
export LOG_TAG="proxy_install"

function check_product_version()
{
    OLD_PRODUCT_VERSION=`cat /opt/huawei-data-protection/agentproxy/conf/version | grep "release_version" | awk -F'=' '{print $2}' | sed 's/ //g'`
    NEW_PRODUCT_VERSION=`cat ${LINE_ROOT}/conf/version | grep "release_version" | awk -F'=' '{print $2}' | sed 's/ //g'`
    if [[ ${NEW_PRODUCT_VERSION} == ${OLD_PRODUCT_VERSION} ]];then
        log_error "the new version is the same as the previous version and does not need to be upgraded."
        exit 1
    fi
}

function backup_old_version_data()
{
    log_info "begin upgrade backup service."
    if [[ -f "${OPT_PATH}/old_version_data.tar.gz" ]];then
        rm -f "${OPT_PATH}/old_version_data.tar.gz"
        rm -rf "${OPT_PATH}/old_version_data"
    fi

    python_ver=`proxython -V | awk '{print $2}' | awk -F '.' '{printf("%s.%s", $1, $2)}'`
    sed -i s/3.8/${python_ver}/g `grep 3.8 -rl "${LINE_ROOT}/cfg/upgrade_files.conf"`
    fail_count=0
    while [ ${fail_count} -lt 3 ]; do
        run_tar_upgrade_files_cmd="tar -P -czf ${OPT_PATH}/old_version_data.tar.gz $(cat "${LINE_ROOT}/cfg/upgrade_files.conf")"
        ${run_tar_upgrade_files_cmd}
        if [[ "$?" -eq 0 ]]; then
            log_info "backup old_version_data success."
            break
        fi
        log_warn "backup old_version_data failed, fail_count=${fail_count}."
        rm -f "${OPT_PATH}/old_version_data.tar.gz"
        let ++fail_count
    done
    [[ "${fail_count}" -eq 3 ]] && CHECK_RESULT 1 "backup old_version_data failed."

    if [[ ! -d "${OPT_PATH}/old_version_data" ]];then
        mkdir -p "${OPT_PATH}/old_version_data"
    fi
    tar_file="${OPT_PATH}/old_version_data.tar.gz"
    check_tar_file_before_unzip ${tar_file}
    ECHO_RETURN $? "Check ${tar_file} failed."
    tar -xvf ${tar_file} -C "${OPT_PATH}/old_version_data" >/dev/null 2>&1
}

function uninstall_services()
{
    chattr -iR /opt/huawei-data-protection/agentproxy/pkg/
    proxython ${LINE_ROOT}/bin/install_services.py "${LINE_ROOT}/services" --uninstall --for_upgrade
    CHECK_RESULT $? "uninstall agentproxy service failed"

    rm -rf /usr/bin/get_info.py
    rm -f /usr/bin/showsys
    rm -rf /etc/sudoers.d/{dpamanager,agentproxy}
    rm -rf /usr/local/bin/{ha,cps-monitor,zookeeper,jre}
    rm -rf /opt/huawei-data-protection
    log_info "uninstall service successfully."
}

function install_services()
{
    sh ${LINE_ROOT}/install.sh -f "${ROOT_PATH}/sys.ini" --for_upgrade
    service cron restart
    service sshd restart
    log_info "install service successfully --for upgrade."
}

function recover_agentproxy_conf()
{
    old_conf_file=$1
    conf_file=$2
    CONF_LIST=("default_user_pwd" "cipher_type" "https_cert_pwd" "tcp_cert_pwd" "rpc_cert_pwd" "weak_pwd"
        "iam_username" "iam_passwd" "svc_user_name" "svc_user_password" "oc_username" "oc_password" "syslog_address")
    for conf_item in "${CONF_LIST[@]}";do
        item_val=$(cat ${old_conf_file} | grep ${conf_item})
        sed -i "s/^${conf_item}.*/${item_val}/g" ${conf_file}
    done
}

function recover_file()
{
    ha_conf_path="/usr/local/bin"
    rm -rf /usr/local/bin/ha/*
    /usr/bin/cp -rf "${OPT_PATH}/old_version_data${ha_conf_path}/ha" "${ha_conf_path}"
    chown -hR agentproxy:agentproxy /usr/local/bin/ha

    conf_root_path="/opt/huawei-data-protection/agentproxy/conf"
    old_conf_root_path="${OPT_PATH}/old_version_data${conf_root_path}"
    recover_agentproxy_conf "${old_conf_root_path}/agentproxy.conf" "${conf_root_path}/agentproxy.conf"
    /usr/bin/cp -f "${old_conf_root_path}/haproxy/privkey.conf" "${conf_root_path}/haproxy/privkey.conf"
    /usr/bin/cp -f "${old_conf_root_path}/omm-ha/privkey.conf" "${conf_root_path}/omm-ha/privkey.conf"
    /usr/bin/cp -rf "${old_conf_root_path}/kmc" "${conf_root_path}"

    cert_path="/opt/huawei-data-protection/agentproxy"
    /usr/bin/cp -rf "${OPT_PATH}/old_version_data${cert_path}/certs" "${cert_path}"
    chown -hR agentproxy:agentproxy "${cert_path}/certs"

    cd ${ROOT_PATH}
    start_service cps-monitor
    start_service omm-ha
    start_service agentproxy
    log_info "recover all conf files success."
}

check_product_version
backup_old_version_data
uninstall_services
install_services
recover_file
sleep 15s
log_info "The upgrade of single node is complete."
