#!/usr/bin/proxython
import os

import requests
import urllib3
from basesdk import utils
from kmc import kmc

urllib3.disable_warnings()
kmc_api = kmc.API()
DEF_CERT = "/opt/huawei-data-protection/agentproxy/certs/agentproxy/arb/ca.crt"

KEY_ENABLE = "arb.enable"
KEY_CERT = "arb.cert"
KEY_IPLIST = "arb.ips"
KEY_USER = "arb.username"
KEY_PASSWORD = "arb.kmc.password"
KEY_LOCAL = "arb.localDC"
KEY_REMOTE = "arb.remoteDC"


class CheckArb(object):
    def __init__(self):
        super(CheckArb, self).__init__()
        self.status = dict()
        self.status['ok'] = 'ok'
        self.status['fail'] = 'fail'
        self.status['unknow'] = 'unknow'
        self.status['public_fail_local'] = 'public_fail_remote'
        self.status['public_fail_remote'] = 'public_fail_local'
        self.status['public_fail_both'] = 'public_fail_both'
        self.status['split_brain_close'] = 'split_brain_open'
        self.status['split_brain_open'] = 'split_brain_close'
        self.status['fail_local'] = 'fail_remote'
        self.status['fail_remote'] = 'fail_local'
        self.status['monitor_fail_local'] = 'monitor_fail_remote'
        self.status['monitor_fail_remote'] = 'monitor_fail_local'

    @staticmethod
    def read_conf(key, default=""):
        arb_conf = utils.get_info("arb_path")
        run_cmds = [['cat', arb_conf], ['grep', key]]
        code, out = utils.run_cmds(run_cmds)
        if code or not out \
                or len(out.split('\n')) != 1 \
                or len(out.split('=')) != 2 \
                or not out.split('=')[1].strip():
            return default
        return out.split('=')[1].strip()

    @staticmethod
    def check_result(result):
        blacklist = ['|', ';', '&', '$', '>', '<', '`', '\\\\', '!', '\\n']
        for black in blacklist:
            if black in result:
                return False
        return True

    def enable(self):
        return self.read_conf(KEY_ENABLE, "false").lower() == 'true'

    def cert(self):
        cert_path = self.read_conf(KEY_CERT, DEF_CERT)
        if not os.path.exists(cert_path):
            # Send Alarm
            return False
        return cert_path

    def get_arb_info(self):
        ips = self.read_conf(KEY_IPLIST)
        user = self.read_conf(KEY_USER)
        pwd = kmc_api.decrypt(0, self.read_conf(KEY_PASSWORD))
        local = self.read_conf(KEY_LOCAL)
        remote = self.read_conf(KEY_REMOTE)
        return ips, user, pwd, local, remote

    def check_health(self, ip, user, pwd):
        health = requests.get("https://%s/health" % ip,
                              auth=(user, pwd),
                              verify=self.cert())
        if health.status_code in [401, 403]:
            raise health.raise_for_status()
        if health.status_code == 200 \
                and "true" == health.json().get('health', "false"):
            return True
        return False

    def check_version(self, ip, user, pwd):
        version = requests.get("https://%s/version" % ip,
                               auth=(user, pwd),
                               verify=self.cert())
        if version.status_code in [401, 403]:
            raise version.raise_for_status()
        if version.status_code == 200 \
                and version.json().get('etcdserver') \
                and version.json().get('etcdcluster'):
            return True
        return False

    def get_dcstatus(self, ip, user, pwd, dc):
        dcstatus = requests.get("https://%s/v2/keys/monitor/status"
                                "/%s/detail_status" % (ip, dc),
                                auth=(user, pwd), verify=self.cert())
        if dcstatus.status_code != 200:
            raise dcstatus.raise_for_status()
        return dcstatus.json().get('node').get('value')

    def query_arb(self, ips, user, pwd, local, remote):
        for ip in ips.split(','):
            if not self.check_health(ip, user, pwd):
                continue
            if not self.check_version(ip, user, pwd):
                continue
            ls = self.get_dcstatus(ip, user, pwd, local)
            rs = self.get_dcstatus(ip, user, pwd, remote)
            if self.check_result(ls) and self.check_result(rs):
                return ls, rs
        raise Exception("Arb server(%s) fault" % ips['ips'])

    def main(self):
        if not self.enable():
            return 0
        ips, user, pwd, local, remote = self.get_arb_info()
        ls, rs = self.query_arb(ips, user, pwd, local, remote)
        print("localDcStatus:%s" % ls)
        print("remoteDcStatus:%s" % rs)
        print("isMatch:%s" % "true" if ls == self.status[rs] else "false")
        return 0


if __name__ == '__main__':
    exit(CheckArb().main())
