#!/usr/bin/proxython

import getpass
import os
import sys
import re
from sys import argv

from basesdk import rpc_client
from basesdk import security_harden
from basesdk import utils
from checkArb import CheckArb
from kmc import kmc

kmc_api = kmc.API()
DEF_PORT = '2379'
DEF_USER = 'arbiterClient'
CUR_DIR = os.path.dirname(os.path.abspath(__file__))
logger = utils.get_logger("operation")
LOCAL_IP = utils.get_info("manage_ip")


class ConfigArb(object):
    def __init__(self):
        super(ConfigArb, self).__init__()

    @staticmethod
    def show_paramer():
        print("usage:")
        print(f"  {argv[0]} disable")
        print(f"  {argv[0]} <ips> <local_dc> <remote_dc>")
        sys.exit(1)

    @staticmethod
    def handel_error(info):
        logger.error(info)
        print("Config ARB failed.")
        sys.exit(1)

    @staticmethod
    def check_ips(ips):
        for _ip in ips.split(","):
            result = re.fullmatch(r'(([01]{0,1}\d{0,1}\d|2[0-4]\d|25[0-5])\.)'
                                  r'{3}([01]{0,1}\d{0,1}\d|2[0-4]\d|25[0-5])', _ip)
            if not result:
                return False
        return True

    def write_conf(self, key, value):
        arb_conf = utils.get_info("arb_path")
        run_cmds = [['cat', arb_conf], ['grep', key]]
        try:
            _, out = utils.run_cmds(run_cmds)
            if out and len(out.split('\n')) != 1:
                raise Exception("Read %s failed" % key)
            if not out:
                run_cmds = ['sed', '-i', '1i %s=%s' % (key, value), arb_conf]
            else:
                run_cmds = ['sed', '-i', 's/%s/%s=%s/g' %
                            (out, key, value), arb_conf]
            res, _ = utils.run_cmd(run_cmds)
            if res:
                raise Exception("write %s failed" % key)
        except Exception as ex:
            err_info = "config arb in node %s failed.%s" % (LOCAL_IP, ex)
            self.handel_error(err_info)

    def write_pwd(self, pwd):
        run_cmd = ['sh', '%s/keytool_encrypt.sh' % CUR_DIR, pwd]
        code, _ = utils.run_cmd(run_cmd, allow_log=False, sensitive_args=[pwd])
        if code:
            self.handel_error("config arb failed. encrypt arb pwd by keytool failed.")
        logger.info("encrypt arb pwd by keytool success.")

    @staticmethod
    def get_url_list(ips):
        url_list = list()
        for ip in ips.split(','):
            url_list.append("%s:%s" % (ip, DEF_PORT))
        return url_list

    def ip_check(self, value):
        code, _ = utils.run_cmd(['proxy_ping', '-c3', value])
        if code:
            self.handel_error(f"config arb failed. check ip({value}) failed.")
        logger.info(f"check ip({value}) success.")

    def arb_service_check(self, ips, local, remote, pwd):
        logger.info("begin check arb service.")
        for ip in ips.split(','):
            self.ip_check(ip)
        logger.info("check arb ips success.")
        url_list = self.get_url_list(ips)
        try:
            CheckArb().query_arb(','.join(url_list), DEF_USER, pwd, local, remote)
            logger.info("service check: query_arb success.")
        except Exception as ex:
            self.handel_error(f"arb_service_check failed. {ex}")

    def disable_arb(self):
        if len(argv) == 2:
            self.config_remote_node(utils.get_info("ha_peer_node"), "disable remote")
        if len(argv) > 2 and argv[2] != "remote":
            self.show_paramer()
        self.write_conf('arb.enable', 'false')
        logger.info(f"Disable ARB success.")
        print("Disable ARB success.")
        sys.exit(0)

    def config_remote_node(self, node_ip, parameters=""):
        logger.info("begin to config ARB in remote node.")
        try:
            client = rpc_client.get_client(node_ip)
            res = client.handle_arb_job(parameters)
            if not res:
                raise Exception("Config arb on remote node failed.")
            logger.info("Config arb on remote node success.")
        except Exception as ex:
            self.handel_error(f"config arb failed. {ex}")

    def config_local_node(self, ips, local, remote, pwd):
        logger.info("begin to config ARB in current node.")
        ip_list = list()
        for ip in ips.split(','):
            ip_list.append("%s:%s" % (ip, DEF_PORT))
        self.write_conf('arb.ips', ','.join(ip_list))
        self.write_conf('arb.localDC', local)
        self.write_conf('arb.remoteDC', remote)
        self.write_conf('arb.username', DEF_USER)
        self.write_conf('arb.enable', 'true')
        self.write_conf('arb.kmc.password', pwd)
        self.write_pwd(pwd)
        logger.info("config ARB success.")
        print("config ARB success.")
        sys.exit(0)

    def enable_arb(self):
        if len(argv) < 4:
            self.show_paramer()
        if len(argv) == 4:
            pwd = getpass.getpass("Please enter the password of user:")
            pwd_confirm = getpass.getpass("Please input the password again to confirm:")
            if pwd != pwd_confirm:
                print("Two different input password.")
                sys.exit(1)
            self.arb_service_check(argv[1], argv[3], argv[2], pwd)
            pwd = kmc_api.encrypt(0, pwd)
            param = "%s %s %s %s" % (argv[1], argv[3], argv[2], pwd)
            self.config_remote_node(utils.get_info("ha_peer_node"), param)
            self.config_local_node(argv[1], argv[2], argv[3], pwd)
        elif len(argv) == 5:
            pwd = argv[4]
            dec_pwd = kmc_api.decrypt(0, pwd)
            self.arb_service_check(argv[1], argv[2], argv[3], dec_pwd)
            self.config_local_node(argv[1], argv[2], argv[3], pwd)
        else:
            logger.info("parameter number error.")
            print("parameter number error.")
            self.show_paramer()

    def main(self):
        if len(argv) < 2:
            self.show_paramer()
        try:
            if argv[1] == "disable":
                self.disable_arb()
            elif self.check_ips(argv[1]):
                self.enable_arb()
            else:
                print("invalid parameters.")
                self.show_paramer()
        except Exception as err:
            self.handel_error(f"config arb failed.{err}")


if __name__ == '__main__':
    security_harden.switch_to_agentproxy_user()
    ConfigArb().main()
