import getpass
import json
import logging
import os
import re
import stat
import subprocess
import sys
from datetime import datetime

import six
from oslo_utils.uuidutils import is_uuid_like

if six.PY2:
    import commands

EBACKUP_SERVER_IP = "ebackup_server_ip"
EBACKUP_SERVER_PORT = "ebackup_server_port"
EBACKUP_SERVER_USERNAME = "ebackup_login_username"
EBACKUP_SERVER_PASSWORD_V2 = "ebackup_login_password_v2"
EBACKUP_LOGIN_CRT = "ebackup_login_crt"
AZS = "azs"
FUSION_STORAGE_IP = "fusion_storage_ip"
PHYSICAL_NETWORK_TYPE = "physical_network_type"
FUSION_STORAGE_AGENT_IP = "fusion_storage_agent_ip"

db_logger = logging.getLogger()


def init_log():
    global db_logger
    current_path = os.path.dirname(os.path.abspath(__file__))
    if not os.path.exists(current_path + os.sep + 'logs'):
        os.mkdir(current_path + os.sep + 'logs')

    formatter = logging.Formatter('%(asctime)s %(levelname)-3s: [%(lineno)s] [%(funcName)s] %(message)s ')
    console_formatter = logging.Formatter('%(message)s ')
    file_handler = logging.FileHandler(current_path + os.sep + 'logs' + os.sep + 'data_recover.log')
    file_handler.setFormatter(formatter)
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.formatter = console_formatter
    db_logger.setLevel(logging.DEBUG)
    file_handler.setLevel(logging.DEBUG)
    console_handler.setLevel(logging.DEBUG)

    db_logger.addHandler(file_handler)
    db_logger.addHandler(console_handler)
    return


def exec_os_cmd(args):
    if six.PY3:
        status, output = subprocess.getstatusoutput(args)
    else:
        status, output = commands.getstatusoutput(args)
    if status != 0:
        raise RuntimeError(output)
    return output


def _check_params(volume_id, start_time, end_time):
    if volume_id and not is_uuid_like(volume_id):
        print('Invalid volume id.')
        exit(1)
    try:
        datetime.strptime(start_time, '%Y-%m-%d %H:%M:%S')
        datetime.strptime(end_time, '%Y-%m-%d %H:%M:%S')
    except ValueError:
        print('Invalid time.')
        exit(1)


class DBExecutor(object):
    def __init__(self):
        exec_os_cmd('export LD_LIBRARY_PATH=/opt/huawei-data-protection/ebackup/db/lib/')

    def exec_sql_cmd(self, sql):
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        with os.fdopen(os.open('tempsql', flags, modes), 'w') as file_handler:
            file_handler.write(sql)
        sql_cmd = 'gsql cinder -f tempsql'
        r = exec_os_cmd(sql_cmd)
        os.remove('tempsql')
        return r


class CloudTool(object):
    def __init__(self):
        self.db = DBExecutor()

    def migrate_backups(self, sourceFile):
        ST_BACKUP = [0, 1]
        ST_COPY = [4, 5]
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        file = os.fdopen(os.open(sourceFile, flags, modes))
        msg = 'opened file: %s' % file
        db_logger.info(msg)
        linenum = 0
        while True:
            line = file.readline().strip('\n')
            if not line:
                break
            try:
                linenum = linenum + 1
                db_logger.info(line.strip())
                (volumeID, types, originTriad, destTriad, originNAS, destNAS) = line.replace(' ', '').split('|')
                msg = 'handle line[%d]: %s,%s,%s,%s,%s,%s' % (
                    linenum, volumeID, types, originTriad, destTriad, originNAS, destNAS)
                db_logger.info(msg)

                if not (types in ('backup', 'copy') and is_uuid_like(volumeID)):
                    db_logger.error('Not support %s' % types)
                    continue

                if "" in (volumeID, originTriad, destTriad, originNAS, destNAS):
                    db_logger.error('Some important variable is empty')
                    continue

                sql = "select ID,SERVICE_METADATA from backups where VOLUME_ID='%s'" % volumeID
                volume_backups = self.db.exec_sql_cmd(sql)
                backups = volume_backups.split('\n')
                for backup in backups[2:-3]:
                    try:
                        (bid, service_metadata) = backup.replace(' ', '').split('|')
                        if not re.match("^[a-fA-F\\d]{8}(-[a-fA-F\\d]{4}){3}-[a-fA-F\\d]{12}$", bid):
                            db_logger.warning('not available UUID')
                            continue
                        metadata = json.loads(service_metadata)
                        if types == 'backup':
                            TYPE_DICT = ST_BACKUP
                        else:
                            TYPE_DICT = ST_COPY
                        if metadata.get('ST', '') not in TYPE_DICT:
                            db_logger.warning('%s is not %s,skip' % (bid, types))
                            continue

                        if metadata.get('SP', '') == originTriad:
                            metadata['SP'] = destTriad
                        if metadata.get('BP', '') == originNAS:
                            metadata['BP'] = destNAS
                        new_metadata = json.dumps(metadata)
                        sql = "update backups set SERVICE_METADATA='%s' where ID='%s'" % (new_metadata, bid)
                        output = self.db.exec_sql_cmd(sql)
                        db_logger.info(sql)
                        db_logger.info(output)
                    except Exception as msg:
                        db_logger.error('not available line (%s) (%s)' % (backup, msg))
                        continue

            except Exception as msg:
                db_logger.error(msg)
                continue
        file.close()
        db_logger.info("[SUCCESS]migrate complete")
        return

    def awaken_backup(self, sourceFile):
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        outputfile = os.fdopen(os.open("/home/gaussdba/backuplist.txt", flags, modes), 'w')
        file = os.fdopen(os.open(sourceFile, flags, modes))
        msg = 'opened file: %s' % file
        db_logger.info(msg)
        backuplist = ''
        while True:
            line = file.readline().strip('\n')
            if not is_uuid_like(line):
                break
            try:
                sql = ("select id, service_metadata from backups where service_metadata LIKE '%%%s%%'" % line)
                vbackups = self.db.exec_sql_cmd(sql)
                backups = vbackups.split('\n')
                for backup in backups[2:-3]:
                    try:
                        (bid, service_metadata) = backup.replace(' ', '').split('|')
                        if not re.match("^[a-fA-F\\d]{8}(-[a-fA-F\\d]{4}){3}-[a-fA-F\\d]{12}$", bid):
                            db_logger.warning('not available UUID')
                            continue
                        metadata = json.loads(service_metadata)
                        if metadata.get('backupurl', '') == line:
                            db_logger.info('[%s]found cinder record[%s] relate with ebackup record' % (line, bid))
                            updatesql = ("update backups set status='available',deleted='f' where ID='%s'" % bid)
                            output = self.db.exec_sql_cmd(updatesql)
                            db_logger.info(updatesql)
                            db_logger.info(output)
                            backuplist = backuplist + bid + '\n'
                    except Exception as msg:
                        db_logger.error('not available line (%s) (%s)' % (backup, msg))
                        continue
            except Exception as msg:
                db_logger.error(msg)
                continue
        outputfile.write(backuplist)
        outputfile.close()
        file.close()
        db_logger.info("[SUCCESS]awaken complete")
        return

    def export_backups(self, file_path=None, start_time=None, end_time=None, volumeid=None):
        flags = os.O_WRONLY | os.O_CREAT | os.O_TRUNC
        modes = stat.S_IWUSR | stat.S_IRUSR
        dst_file = os.fdopen(os.open("/home/gaussdba/export.json", flags, modes), 'w')
        exportbackups = []
        if file_path is None:
            if volumeid:
                sql = (
                    "select ID,SNAPSHOT_ID,VOLUME_ID,PROJECT_ID,OBJECT_COUNT,STATUS,DELETED,SERVICE_METADATA,DISPLAY_NAME,CREATED_AT "
                    "from backups where (status='available' or status='deleted' or "
                    "status='pre_delete' or status='post_delete' or status='restoring') and volume_id='%s' and CREATED_AT > '%s' and CREATED_AT < '%s' and "
                    "service_metadata NOT LIKE '%%backupurl_tmp%%'" % (volumeid, start_time, end_time))
            else:
                sql = (
                    "select ID,SNAPSHOT_ID,VOLUME_ID,PROJECT_ID,OBJECT_COUNT,STATUS,DELETED,SERVICE_METADATA,DISPLAY_NAME,CREATED_AT "
                    "from backups where (status='available' or status='deleted' or "
                    "status='pre_delete' or status='post_delete' or status='restoring') and CREATED_AT > '%s' and CREATED_AT < '%s' and "
                    "service_metadata NOT LIKE '%%backupurl_tmp%%'" % (start_time, end_time))
            vbackups = self.db.exec_sql_cmd(sql)
            db_logger.info(sql)
            db_logger.info(vbackups)
            backups = vbackups.split('\n')
            for backup in backups[2:-3]:
                try:
                    (bid, snapshotid, volumeid, projectid, objectcount, status, deleted, service_metadata, displayname,
                     create_at) = backup.replace(' ', '').split('|')
                    if not re.match("^[a-fA-F\\d]{8}(-[a-fA-F\\d]{4}){3}-[a-fA-F\\d]{12}$", bid):
                        db_logger.warning('not available UUID')
                        continue
                    servicejson = json.loads(service_metadata)
                    if len(displayname.split('backup@')) == 2:
                        (_, displayname) = displayname.split('backup@')
                    if not objectcount:
                        objectcount = '0'
                    backupjsondict = {}
                    backupjsondict['BackupID'] = bid
                    backupjsondict['SnapshotID'] = snapshotid
                    backupjsondict['VolumeID'] = volumeid
                    backupjsondict['ProjectID'] = projectid
                    backupjsondict['ObjectCount'] = int(objectcount)
                    backupjsondict['Status'] = status
                    backupjsondict['Deleted'] = deleted
                    backupjsondict['ServiceMetadata'] = json.loads(service_metadata)
                    backupjsondict['BackupUrl'] = servicejson.get('backupurl', '')
                    backupjsondict['CascadingID'] = displayname
                    backupjsondict['CreateTime'] = create_at[:10] + ' ' + create_at[10:]
                    exportbackups.append(backupjsondict)
                except Exception as msg:
                    db_logger.error('not available line of output (%s) (%s)' % (backup, msg))
                    continue
        else:
            sourcefile = open(file_path)
            msg = 'opened file: %s' % sourcefile
            db_logger.info(msg)
            while True:
                line = sourcefile.readline().strip('\n')
                if not is_uuid_like(line):
                    break
                try:
                    sql = "select ID,SNAPSHOT_ID,VOLUME_ID,PROJECT_ID,OBJECT_COUNT,STATUS,DELETED,SERVICE_METADATA,DISPLAY_NAME,CREATED_AT" \
                          " from backups where (service_metadata like '%%backupurl\\\":\\\"%s%%""' or " \
                          "service_metadata like '%%backupurl\\\": \\\"%s%%')" % (line, line)
                    vbackups = self.db.exec_sql_cmd(sql)
                    db_logger.info(sql)
                    db_logger.info(vbackups)
                    backups = vbackups.split('\n')
                    for backup in backups[2:-3]:
                        try:
                            (bid, snapshotid, volumeid, projectid, objectcount, status, deleted, service_metadata,
                             displayname,
                             create_at) = backup.replace(' ', '').split('|')
                            if not re.match("^[a-fA-F\\d]{8}(-[a-fA-F\\d]{4}){3}-[a-fA-F\\d]{12}$", bid):
                                db_logger.warning('not available UUID')
                                continue
                            servicejson = json.loads(service_metadata)
                            if len(displayname.split('backup@')) == 2:
                                (_, displayname) = displayname.split('backup@')
                            if not objectcount:
                                objectcount = '0'
                            backupjsondict = {}
                            backupjsondict['BackupID'] = bid
                            backupjsondict['SnapshotID'] = snapshotid
                            backupjsondict['VolumeID'] = volumeid
                            backupjsondict['ProjectID'] = projectid
                            backupjsondict['ObjectCount'] = int(objectcount)
                            backupjsondict['Status'] = status
                            backupjsondict['Deleted'] = deleted
                            backupjsondict['ServiceMetadata'] = json.loads(service_metadata)
                            backupjsondict['BackupUrl'] = servicejson.get('backupurl', '')
                            backupjsondict['CascadingID'] = displayname
                            backupjsondict['CreateTime'] = create_at[:10] + ' ' + create_at[10:]
                            exportbackups.append(backupjsondict)
                        except Exception as msg:
                            db_logger.error('not available line of output (%s) (%s)' % (backup, msg))
                            continue
                except Exception as msg:
                    db_logger.error('not available line (%s) (%s)' % (line, msg))
                    continue
            sourcefile.close()
        backupsdict = dict()
        backupsdict['backups'] = exportbackups
        json.dump(backupsdict, dst_file, sort_keys=False, indent=4, separators=(',', ':'))
        dst_file.close()
        db_logger.info("[SUCCESS]export complete")
        return

    @staticmethod
    def update_uds_pwd():
        from cinder.backup.drivers import ebackupagent
        from cinder.backup.lib import cpsrestful as cpscli

        cps_user, cps_password = set_env_var()
        cps_connect = cpscli.CpsConnection('', '', cps_user, cps_password)

        exist_backup_templates = cps_connect.get_cinder_backup_templates()
        backup_parameters = dict()
        for template in exist_backup_templates:
            template_ext_params = cps_connect.get_backup_template_ext_params(template)
            if template_ext_params.get('default.backup_driver') == 'cinder.backup.drivers.hwsebackup':
                backup_parameters = template_ext_params
                break
        if not backup_parameters:
            db_logger.error('No ebackup driver params found')
            exit(0)

        unit_path = backup_parameters.get('default.ebackup_storage_unit_path', '')
        bucket_name = backup_parameters.get('default.ebackup_bucket_name', '')
        if not unit_path or not bucket_name:
            db_logger.error('No uds information found in driver config')
            exit(0)

        uds_info = {}
        uds_info['uds_path'] = unit_path + ':/' + bucket_name
        ak = backup_parameters.get('default.ebackup_storage_username_v2', '')
        uds_info['uds_ak'] = ebackupagent.Utils.decrypt_password(ak)
        sk = backup_parameters.get('default.ebackup_storage_password_v2', '')
        uds_info['uds_sk'] = ebackupagent.Utils.decrypt_password(sk)
        uds_info['uds_proto'] = backup_parameters.get('default.ebackup_storage_s3_protocol', '')

        ebackup_server_ip = backup_parameters.get('default.ebackup_server_ip', '')
        if not ebackup_server_ip:
            cfg_servers_map = get_ebackup_server_list(backup_parameters)
            if not cfg_servers_map:
                db_logger.error("modify uds failed, please configure the eBackup server information firstly.")
                exit(0)
            for name, server_item in cfg_servers_map.items():
                ebackup_conf = ebackup_server_config()
                ebackup_conf.ebackup_server_ip = server_item[EBACKUP_SERVER_IP]
                ebackup_conf.ebackup_server_port = server_item[EBACKUP_SERVER_PORT]
                ebackup_conf.ebackup_login_username = server_item[EBACKUP_SERVER_USERNAME]
                ebk_pwd = ebackupagent.Utils.decrypt_password(server_item[EBACKUP_SERVER_PASSWORD_V2])
                ebackup_conf.ebackup_login_password = ebk_pwd
                ebackup_conf.ebackup_login_crt = server_item[EBACKUP_LOGIN_CRT]
                ebackup_conn = ebackupagent.Connection(ebackup_conf)
                error_code = modi_ebk_storageunit(unit_path, bucket_name, uds_info, ebackup_conn)
                return error_code

        else:
            ebackup_conf = ebackup_server_config()
            ebackup_conf.ebackup_server_ip = backup_parameters.get('default.ebackup_server_ip', '')
            ebackup_conf.ebackup_login_crt = backup_parameters.get('default.ebackup_login_crt', '')
            ebackup_conf.ebackup_login_username = backup_parameters.get('default.ebackup_login_username', '')
            ebk_pwd = backup_parameters.get('default.ebackup_login_password_v2', '')
            ebackup_conf.ebackup_login_password = ebackupagent.Utils.decrypt_password(ebk_pwd)
            ebackup_conn = ebackupagent.Connection(ebackup_conf)
            error_code = modi_ebk_storageunit(unit_path, bucket_name, uds_info, ebackup_conn)
            return error_code

    def repair_dedup_remote_copy(self):
        try:
            sql = "select id,service_metadata from backups where (service_metadata like '%\"IT\":3%' or " \
                  "service_metadata like '%\"IT\": 3%') " \
                  "and (service_metadata like '%\"DL\": 3%' or service_metadata like '%\"DL\":3%' or " \
                  "service_metadata like '%\"DL\": 1%' or service_metadata like '%\"DL\":1%') and status='available';"
            vbackups = self.db.exec_sql_cmd(sql)
            backups = vbackups.split('\n')
            for backup in backups[2:-3]:
                try:
                    (bid, service_metadata) = backup.replace(' ', '').split('|')
                    if not re.match("^[a-fA-F\\d]{8}(-[a-fA-F\\d]{4}){3}-[a-fA-F\\d]{12}$", bid):
                        db_logger.warning('not available UUID')
                        continue
                    metadata = json.loads(service_metadata)
                    if metadata.get('IT', '') == 3 and metadata.get('DL', '') in [1, 3]:
                        if metadata.get('DL', '') == 3:
                            metadata['DL'] = 2
                        else:
                            metadata['DL'] = 0
                        service_metadata = json.dumps(metadata)
                        updatesql = ("update backups set service_metadata='%s' where ID='%s'" % (service_metadata, bid))
                        output = self.db.exec_sql_cmd(updatesql)
                        db_logger.info("update backup %s" % bid)
                        db_logger.info(output)
                except Exception as msg:
                    db_logger.error('not available line (%s) (%s)' % (backup, msg))
                    continue
            db_logger.info("update completed!")
        except Exception as msg:
            db_logger.error(msg)


class ebackup_server_config(object):
    ebackup_server_ip = ''
    ebackup_server_port = ''
    ebackup_login_username = ''
    ebackup_login_password = ''
    ebackup_login_crt = ''


def set_env_var():
    db_logger.info('Please set the cps environment variables first.')
    while True:
        while True:
            cps_username = six.moves.input('Enter CPS_USERNAME=')
            if cps_username.strip() == '':
                db_logger.info("CPS_USERNAME can not be empty,please input again.")
            else:
                break
        os.environ['CPS_USERNAME'] = cps_username

        while True:
            cps_password = getpass.getpass('Enter CPS_PASSWORD=')
            if cps_password.strip() == '':
                db_logger.info("CPS_PASSWORD can not be empty,please input again.")
            else:
                break
        os.environ['CPS_PASSWORD'] = cps_password

        cmd = ["cps", "host-list"]
        pipe = subprocess.Popen(cmd, shell=False, stdout=subprocess.PIPE).stdout
        cps_host_list_result = pipe.read()
        if six.PY3:
            cps_host_list_result = str(cps_host_list_result, encoding='utf-8')
        if cps_host_list_result.find('locked') != -1:
            db_logger.info("CPS user is current locked, please input again later.")
            sys.exit()
        elif cps_host_list_result.find('id') == -1 and cps_host_list_result.find('status') == -1:
            db_logger.info("CPS_USERNAME or CPS_PASSWORD is invalid, please input again.")
        else:
            break
    return cps_username, cps_password


def get_ebackup_server_list(backup_parameters):
    cfg_servers_map = {}
    ebackup_servers = backup_parameters.get('default.ebackup_servers', '')
    server_list = ebackup_servers.split(",")
    for server in server_list:
        cfg_server = {}
        for key in backup_parameters:
            if key == (server + '.' + EBACKUP_SERVER_IP):
                cfg_server[EBACKUP_SERVER_IP] = backup_parameters.get(key, '')
            elif key == (server + '.' + EBACKUP_SERVER_PORT):
                cfg_server[EBACKUP_SERVER_PORT] = backup_parameters.get(key, '')
            elif key == (server + '.' + EBACKUP_SERVER_USERNAME):
                cfg_server[EBACKUP_SERVER_USERNAME] = backup_parameters.get(key, '')
            elif key == (server + '.' + EBACKUP_SERVER_PASSWORD_V2):
                cfg_server[EBACKUP_SERVER_PASSWORD_V2] = backup_parameters.get(key, '')
            elif key == (server + '.' + EBACKUP_LOGIN_CRT):
                cfg_server[EBACKUP_LOGIN_CRT] = backup_parameters.get(key, '')
            elif key == (server + '.' + AZS):
                cfg_server[AZS] = backup_parameters.get(key, '')
            elif key == (server + '.' + FUSION_STORAGE_IP):
                cfg_server[FUSION_STORAGE_IP] = backup_parameters.get(key, '')
            elif key == (server + '.' + FUSION_STORAGE_AGENT_IP):
                cfg_server[FUSION_STORAGE_AGENT_IP] = backup_parameters.get(key, '')
            elif key == (server + '.' + PHYSICAL_NETWORK_TYPE):
                cfg_server[PHYSICAL_NETWORK_TYPE] = backup_parameters.get(key, '')
        cfg_servers_map[server] = cfg_server
    return cfg_servers_map


def modi_ebk_storageunit(uds_path, uds_name, uds_info, ebackup_conn):
    pre_unit_path = uds_path + ":/" + uds_name
    get_unit_count_uri = "/vbackup_storageunit/count?filter=FILESYSTEMTYPE::7%20and%20NAME:openstack%20and%20PATH:" + str(
        uds_path) + "&sortby=ID,d"
    get_count_result = ebackup_conn.get(get_unit_count_uri)
    dict_error = get_count_result['error']
    error_code = dict_error['code']
    if error_code != 0:
        db_logger.info("modify unit failed, eBackup return error num is:" + str(error_code))
        if "description" in dict_error:
            error_msg = dict_error['description']
            db_logger.info("description: " + error_msg)
        if "suggestion" in dict_error:
            error_suggestion = dict_error['suggestion']
            db_logger.info("suggestion: " + error_suggestion)
        ebackup_conn.__del__()
        return -1
    dict_data = get_count_result["data"]
    unit_count = int(dict_data["COUNT"])
    index_sheet = 0
    while index_sheet < unit_count:
        get_unit_uri = "/vbackup_storageunit?filter=FILESYSTEMTYPE::7%20and%20NAME:openstack%20and%20PATH:" + str(
            uds_path) + "&range=[" + str(index_sheet) + "-" + str(index_sheet + 30) + "]&sortby=ID,d"
        index_sheet += 30
        get_unit_result = ebackup_conn.get(get_unit_uri)
        dict_error = get_unit_result['error']
        error_code = dict_error['code']
        if error_code != 0:
            db_logger.info("modify unit failed, eBackup return error num is:" + str(error_code))
            if "description" in dict_error:
                error_msg = dict_error['description']
                db_logger.info("description: " + error_msg)
            if "suggestion" in dict_error:
                error_suggestion = dict_error['suggestion']
                db_logger.info("suggestion: " + error_suggestion)
            ebackup_conn.__del__()
            return -1
        list_uds_unit = get_unit_result["data"]
        for uds_unit in list_uds_unit:
            if str(pre_unit_path) not in uds_unit["PATH"]:
                continue
            try_count = 0

            while True:
                unit_id = uds_unit["ID"]
                modify_unit_uri = "/vbackup_storageunit/" + str(unit_id)
                unit_body = {}
                if uds_info["uds_path"] != "":
                    old_unit_path = uds_unit["PATH"]
                    new_unit_path = str(uds_info["uds_path"]) + old_unit_path[len(pre_unit_path):]
                    unit_body["PATH"] = new_unit_path

                if uds_info["uds_ak"] != "":
                    unit_body["USERNAME"] = str(uds_info["uds_ak"])

                if uds_info["uds_sk"] != "":
                    unit_body["PASSWORD"] = str(uds_info["uds_sk"])

                if uds_info["uds_proto"] != "":
                    unit_body["PROTOCOL"] = str(uds_info["uds_proto"])

                put_result = ebackup_conn.put(modify_unit_uri, unit_body)
                dict_error = put_result['error']
                error_code = dict_error['code']
                if error_code != 0:
                    if try_count >= 3:
                        db_logger.info("modify unit failed, eBackup return error num is:" + str(error_code))
                        if "description" in dict_error:
                            error_msg = dict_error['description']
                            db_logger.info("description: " + error_msg)
                        if "suggestion" in dict_error:
                            error_suggestion = dict_error['suggestion']
                            db_logger.info("suggestion: " + error_suggestion)
                        ebackup_conn.__del__()
                        return -1
                    try_count += 1
                    continue
                break
    ebackup_conn.__del__()
    return 0


if __name__ == '__main__':
    init_log()

    if len(sys.argv) == 2 and sys.argv[1] == '--update_uds_pwd':
        ret = CloudTool.update_uds_pwd()
        exit(ret)

    if len(sys.argv) < 4 or sys.argv[1] not in (
            '--migrate', '--export', '--awaken', '--update_uds_pwd', '--repair_dedup_remote_copy'):
        print('\nUsage: python cloudtools.py --migrate -f [migrate_file]\n'
              '       python cloudtools.py --export [-f export_file|-t \'startime\' \'endtime\'  <volumeid>] \n'
              '       python cloudtools.py  --awaken -f [awaken_file]\n'
              '       python cloudtools.py  --update_uds_pwd\n'
              '       python cloudtools.py  --repair_dedup_remote_copy\n')
        exit(-1)
    tool = CloudTool()
    if sys.argv[1] == '--migrate':
        file = sys.argv[3]
        tool.migrate_backups(file)
    elif sys.argv[1] == '--awaken':
        file = sys.argv[3]
        tool.awaken_backup(file)
    elif sys.argv[1] == '--export':
        if sys.argv[2] == '-f':
            file = sys.argv[3]
            tool.export_backups(file)
        elif sys.argv[2] == '-t':
            start_time = sys.argv[3]
            end_time = sys.argv[4]
            if len(sys.argv) >= 6:
                volume_id = sys.argv[5]
            else:
                volume_id = None
            _check_params(volume_id, start_time, end_time)
            tool.export_backups(None, start_time, end_time, volume_id)
        else:
            print('unknown!')
            exit(-1)
    elif sys.argv[1] == '--repair_dedup_remote_copy':
        tool.repair_dedup_remote_copy()
        exit(0)


