import configparser
import os
import json
from cinder.backup.lib import httprestful as httpclient
import logging
from FSSysconf import netPolicyParse
import six
if six.PY3:
    import urllib.error as url_lib
else:
    import urllib2 as url_lib
logger = logging.getLogger('__EBACKUP_CONFIG__')


class CpsConnection(object):
    """ cps restful"""

    def __init__(self, authurl, baseurl, cps_user, cps_password, retrytimes=5, retryinterval=30):
        self.baseurl = self._get_base_url()
        self.cps_user = cps_user
        self.cps_password = cps_password
        self.retrytimes = retrytimes
        self.retryinterval = retryinterval
        self.baseconnect = httpclient.HttpConnection('', retrytimes, retryinterval)

    @staticmethod
    def _get_base_url():
        try:
            floating_ip = netPolicyParse.getComponentFloatingIp(compName="cps-server")
            floating_port = netPolicyParse.getComponentFloatingPort(compName="cps-server")
            return f'https://{floating_ip}:{floating_port}'
        except Exception as e:
            logger.debug("Not found cps-api maybe it's cascaded %s!" % e)
            cps_ip_file = "/etc/huawei/fusionsphere/cfg/serverIp.ini"
            if os.path.exists(cps_ip_file):
                config = configparser.ConfigParser()
                config.read(cps_ip_file)
                return config['IP']['serverip'].strip()
        return ""

    def __get_token(self):
        body = {
            "auth": {
                "identity": {
                    "methods": ["password"],
                    "password": {
                        "user": {
                            "domain": {
                                "name": "Default"
                            },
                            "name": self.cps_user,
                            "password": self.cps_password
                        }
                    }
                },
                "scope": {
                    "project": {
                        "name": "Default",
                        "domain": {
                            "name": "Default"
                        }
                    }
                }
            }
        }
        url = self.authurl + '/auth/tokens?nocatalog'
        headers = dict()
        headers['Content-Type'] = 'application/json'
        resheader, resbody = self.baseconnect.request('post', url, headers, body)
        self.token = resheader['x-subject-token']

    def send_request(self, method, url, headers, body):
        try:
            headers["X-Auth-Password"] = self.cps_password
            headers["X-Auth-User"] = self.cps_user
            headers['Content-Type'] = 'application/json'
            return self.baseconnect.request(method, url, headers, body)
        except url_lib.HTTPError as e:
            if e.code == 401:
                try:
                    headers["X-Auth-Password"] = self.cps_password
                    headers["X-Auth-User"] = self.cps_user
                    return self.baseconnect.request(method, url, headers, body)
                except Exception as final_msg:
                    logger.error(final_msg)
                    raise
            raise

    def get_cinder_backup_templates(self):
        url = self.baseurl + '/cps/v1/services/cinder/componenttemplates?commit_state=commited'
        headers = dict()
        backup_templates = list()
        try:
            resheaders, body = self.send_request('get', url, headers, None)
            templates_format = json.loads(body)
            for template in templates_format['templates']:
                template_name = template.get("name", "")
                if template_name.startswith("cinder-backup"):
                    backup_templates.append(template_name)
                else:
                    logger.debug("not backup template:" + template_name)
            return backup_templates
        except Exception as msg:
            logger.error(msg)
            raise

    def get_backup_template_ext_params(self, template_name):
        url = self.baseurl + '/cps/v1/template/params/cinder/' + template_name
        headers = dict()
        try:
            logger.debug('start get backup template %s\'s extended parameters' % template_name)
            resheaders, body = self.send_request('get', url, headers, None)
            params_format = json.loads(body)
            template_ext_params = params_format['params']
            return template_ext_params
        except Exception as msg:
            logger.error(msg)
            raise

    def set_backup_template_ext_params(self, template_name, params_list, full_cover):
        try:
            logger.debug('start set backup template %s\'s parameters cover:%s' % (template_name, full_cover))
            update_params = dict()
            addition_params = dict()
            exist_params = self.get_backup_template_ext_params(template_name)
            if full_cover:
                self.delete_backup_template_ext_params(template_name, exist_params)
                addition_params = params_list
            else:
                for k, v in params_list.items():
                    if k in exist_params:
                        update_params[k] = v
                    else:
                        addition_params[k] = v
            if addition_params:
                self.add_backup_template_ext_params(template_name, addition_params)
            if update_params:
                self.update_backup_template_ext_params(template_name, update_params)
        except Exception as msg:
            logger.error(msg)
            raise

    def add_backup_template_ext_params(self, template_name, params_list):
        url = self.baseurl + '/cps/v1/template/params'
        headers = dict()
        body = dict()
        try:
            body['service_name'] = 'cinder'
            body['template_name'] = template_name
            body['params'] = params_list
            self.send_request('post', url, headers, body)
        except Exception as msg:
            logger.error(msg)
            raise

    def delete_backup_template_ext_params(self, template_name, params_list):
        url = self.baseurl + '/cps/v1/template/params/cinder/' + template_name
        headers = dict()
        body = dict()
        try:
            body['params'] = list(params_list.keys())
            self.send_request('delete', url, headers, body)
        except Exception as msg:
            logger.error(msg)
            raise

    def update_backup_template_ext_params(self, template_name, params_list):
        url = self.baseurl + '/cps/v1/template/params/cinder/' + template_name
        headers = dict()
        body = dict()
        try:
            body['params'] = params_list
            self.send_request('post', url, headers, body)
        except Exception as msg:
            logger.error(msg)
            raise

    def transaction_state(self):
        url = self.baseurl + '/cps/v1/commit/state'
        headers = dict()
        try:
            resheader, resbody = self.send_request('get', url, headers, None)
            body_format = json.loads(resbody)
            return body_format.get('state')
        except Exception as msg:
            logger.error(msg)
            raise

    def commit(self):
        url = self.baseurl + '/cps/v1/commit'
        headers = dict()
        body = dict()
        try:
            body['timeout'] = 120
            self.send_request('post', url, headers, body)
        except Exception as msg:
            logger.error(msg)
            raise

    def rollback(self):
        url = self.baseurl + '/cps/v1/rollback'
        headers = dict()
        try:
            self.send_request('post', url, headers, None)
        except Exception as msg:
            logger.error(msg)
            raise

    def get_backup_template_params(self, template_name):
        url = self.baseurl + '/cps/v1/services/cinder/componenttemplates/' + template_name + '/params'
        headers = dict()
        try:
            logger.debug('start get backup template %s\'s parameters' % template_name)
            resheaders, body = self.send_request('get', url, headers, None)
            params_format = json.loads(body)
            template_ext_params = params_format['cfg']
            return template_ext_params
        except Exception as msg:
            logger.error(msg)
            raise
