import json
import os
import shutil
import time
from functools import wraps

from bin.agentmanager.sub_install import SubInstallHandle
from bin.agentmanager.sub_install import clear_useless_pkg
from bin.agentmanager.sub_install import delete_all_temp_file
from bin.agentmanager.sub_uninstall import SubUninstallHandle
from common.common_define import CommonDefine
from common.exceptions import ErrorBackupJobIsRunning
from common.logutils import Logger, SUBAGENT_INSTALL_LOG
from common.utils import Utils
from common.utils import VAR_CDM_LIB
from common.utils import VAR_CDM_LOG
from common.utils import VAR_CDM_LIB_BAK
from common.utils import VAR_CDM_LOG_BAK
from common.utils import log_with_exception
from subagent.sub_common_oper import SubCommonOper, sub_agent_path

sub_agent_root_path = Utils.get_sub_agent_root_path()
assist_root_path = Utils.get_agent_assist_root_path()
backup_path = Utils.get_sub_agent_backup_path()
log = Logger().get_logger(SUBAGENT_INSTALL_LOG)


def clear_bak_path(func):
    @wraps(func)
    def wrapper(*args, **kw):
        ret = func(*args, **kw)
        if ret is not True:
            if not SubUpgradeHandle.rollback_old_client():
                log.error("Failed to rollback.")  # 回滚失败不删除备份目录
                return ret
            log.info("Successfully rollback.")
        if not SubCommonOper.clear_useless_dir(backup_path):
            log.info("Clear backup dir failed.")
        return ret

    return wrapper


class SubUpgradeHandle(object):
    def __init__(self):
        self._install_handler = SubInstallHandle()

    @staticmethod
    def _backup_installed_pac():
        if not SubCommonOper.stop_client(sub_agent_path):
            log.error("stop SubAgent failed")
            return False

        if os.path.exists(backup_path):
            try:
                shutil.rmtree(backup_path)
            except Exception as e:
                log.error(f"Remove backup dir failed, error:{e}.")
                return False
        log.info(f"Remove backup_path success {backup_path}.")

        if not os.path.exists(sub_agent_root_path):
            log.error("The installation directory doesn't exists, please check whether have installed it.")
            return False
        try:
            shutil.copytree(sub_agent_root_path, backup_path)
            if not CommonDefine.IS_WINDOWS:
                shutil.copytree(VAR_CDM_LIB, VAR_CDM_LIB_BAK)
                shutil.copytree(VAR_CDM_LOG, VAR_CDM_LOG_BAK)
        except Exception as e:
            log.error(f"Backup installed package failed, error:{e}.")
            return False

        return True

    @staticmethod
    def _uninstall_old_client(agent_name):
        log.info("Start uninstalling the old SubAgent.")
        uninstall_handler = SubUninstallHandle()
        if not uninstall_handler.uninstall_handle(agent_name, False):
            log.error("The old SubAgent fails to be uninstalled.")
            return False
        log.info("SubAgent has been uninstalled successfully.")
        return True

    @staticmethod
    def rollback_old_dpa():
        if CommonDefine.IS_WINDOWS:
            return True
        service_src_path = os.path.join(backup_path, 'CDMClient/etc/ClientService')
        if not os.path.exists(os.path.join(service_src_path, 'HWClientService.service')):
            service_src_path = os.path.join(backup_path, 'CDMClient/etc/ClientService/BasicRunner')
            if not os.path.exists(os.path.join(service_src_path, 'HWClientService.service')):
                log.error("HWClientService.service is not exist.")
                return False
        service_des_path = '/etc/systemd/system'
        try:
            shutil.copyfile(os.path.join(service_src_path, 'HWClientService.service'),
                            os.path.join(service_des_path, 'HWClientService.service'))
            shutil.copytree(VAR_CDM_LIB_BAK, VAR_CDM_LIB)
            shutil.copytree(VAR_CDM_LOG_BAK, VAR_CDM_LOG)
        except Exception as error:
            log.error(f"Rollback old dpa failed, error:{error}.")
            return False
        return True

    @staticmethod
    @log_with_exception(log)
    def rollback_old_client():
        if not os.path.exists(backup_path):
            log.warning("Backup path not exists.")  # 备份目录不存在, 不需要回滚
            return True
        if not SubCommonOper.stop_client(sub_agent_path):
            log.error("Stop SubAgent process failed.")
            return False
        time.sleep(2)
        try:
            copy_result = SubUpgradeHandle.copy_old_sub_agent()
        except Exception as e:
            log.error(f"Rollback old client failed, error:{e}.")
            return False
        if not copy_result:
            return False
        if not SubCommonOper.start_client(sub_agent_path, True):
            log.error("Start SubAgent process failed.")
            return False
        log.info("Successfully rollback the SubAgent.")
        return True

    @staticmethod
    @log_with_exception(log)
    def copy_old_sub_agent():
        os.chdir(assist_root_path)
        if os.path.exists(sub_agent_root_path):
            shutil.rmtree(sub_agent_root_path)
        shutil.copytree(backup_path, sub_agent_root_path)
        if not SubUpgradeHandle.rollback_old_dpa():
            log.error("Rollback old dpa failed.")
            return False
        return True

    @staticmethod
    def _recover_tmp_file():
        origin_tmp_file = os.path.join(backup_path, "testcfgbak.tmp")
        current_tmp_file = os.path.join(sub_agent_root_path, "testcfgbak.tmp")
        if os.path.isfile(origin_tmp_file) and os.path.isfile(current_tmp_file):
            try:
                os.remove(current_tmp_file)
                shutil.copy(origin_tmp_file, current_tmp_file)
            except Exception as e:
                log.error(f"Recover tmp file failed, error[{e}].")
                return False
        return True

    @staticmethod
    def _backup_old_install_info():
        if os.path.isfile(os.path.join(assist_root_path, 'conf/db_info.json')):
            try:
                shutil.copy(
                    os.path.join(assist_root_path, 'conf/db_info.json'),
                    os.path.join(assist_root_path, 'conf/db_info_bak.json'))
            except Exception as e:
                log.error(f"Backup db file failed, error:{e}.")
                return False
        return True

    def _revert_old_dpa_ip(self):
        conf_file = os.path.join(sub_agent_root_path, "CDMClient/etc/ClientService/BasicRunner/clientsetting.config")
        if not os.path.isfile(conf_file):
            conf_file = os.path.join(sub_agent_root_path, "CDMClient/etc/ClientService/clientsetting.config")
            if not os.path.isfile(conf_file):
                log.error(f"Config file[{os.path.basename(conf_file)}] not exists.")
                return False
        option_name = 'targetIpAddr'
        with open(conf_file, 'r', encoding='utf-8') as fr:
            for line in fr.readlines():
                if option_name in line:
                    ip_info = line.split('=')
                    dpa_server_ip = str(ip_info[1])
                    dpa_server_ip = dpa_server_ip.strip()
        self._install_handler.mod_server_ip(dpa_server_ip)

    @staticmethod
    def _recover_db_file():
        db_info_file = os.path.join(assist_root_path, 'conf/db_info.json')
        db_info_file_bak = os.path.join(assist_root_path, 'conf/db_info_bak.json')
        if os.path.isfile(db_info_file):
            try:
                os.remove(db_info_file)
            except Exception as e:
                log.error(f"Remove db info file failed, error:{e}.")
                return False
        if os.path.isfile(db_info_file_bak):
            try:
                os.rename(db_info_file_bak, db_info_file)
            except Exception as e:
                log.error(f"Rename db file failed, error:{e}.")
                return False
        return True

    @delete_all_temp_file(check_install=False)
    def _install_new_client(self, body_str):
        """
        The decorator needs to be use body_str
        """
        if not self._install_handler.install_client():
            log.error("Failed to install the new SubAgent.")
            return False
        log.info("Successfully install the new SubAgent.")
        return True

    @clear_bak_path
    @clear_useless_pkg
    @log_with_exception(log)
    def upgrade_handle(self, body_str):
        if not Utils.check_backup_job():
            log.error("Running backup jobs(eefproc, datasourceproc) have been detected, please stop them first!")
            raise ErrorBackupJobIsRunning
        self._install_handler.check_space()
        if not self._backup_old_install_info():
            log.error("Back up old install info failed.")
            return False
        self._revert_old_dpa_ip()
        try:
            body_dict = json.loads(body_str)
        except Exception as e:
            log.error(f"Input param invalid，error[{e}].")
            return False
        if not self._install_handler.check_param_valid(body_dict, is_upgade=True):
            log.error("The pkg to be upgraded not match.")
            return False
        file_name = body_dict.get("file_name")
        cur_path = os.path.join(assist_root_path, 'bin/assist/subagent/')
        os.chdir(cur_path)
        if not self._backup_installed_pac():
            log.error("Backup installed pkg failed.")
            return False
        if not self._uninstall_old_client(file_name):
            log.error("Failed to uninstall old agent during upgration, start to rollback.")
            return False
        if not self._recover_db_file():
            log.error("Recover db file failed.")
            return False
        if not self._install_new_client(body_str):
            log.error("Failed to install new SubAgent during upgration, start to rollback.")
            return False
        self._recover_tmp_file()
        return True
