import sys
import time

try:
    import win32serviceutil
except ImportError:
    pass

from agentassisthandle.winutils import is_win_service_exist
from common.logutils import Logger, AGENT_ASSIST_MANAGER_LOG
from common.utils import Utils, log_with_exception
from common.common_define import CommonDefine

log = Logger().get_logger(AGENT_ASSIST_MANAGER_LOG)


class AgentAssistStopProcess(object):
    # this init is used for font color of windows consle

    @log_with_exception(log)
    def stop_process(self, process_file_name):
        status, names, pids = Utils.get_process_name_and_pid(process_file_name)
        log.info(f"process_file_name = {process_file_name}, status = {status}, "
                 f"name = {names}, pid = {pids}")
        if not status:
            return False
        for name, pid in zip(names, pids):
            if CommonDefine.IS_WINDOWS:
                ret, info = Utils.execute_cmd(['taskkill', '/F', '/PID',
                                               f'{pid}'])
                log.info(f"code = {ret}, info = {info}")
            else:
                ret, info = Utils.execute_cmd(['kill', '-9', f'{pid}'])
            if not ret:
                log.info(f"kill specific process: name({name})-pid({pid}) done")
                return True
            else:
                log.error(f'failed to kill process({name})-pid({pid}) {info}')
                return False

    def stop_all_process(self):
        try:
            if is_win_service_exist():
                # monitor服务running，先stop，再remove
                try:
                    win32serviceutil.StopService(Utils.monitor_service_name)
                    time.sleep(5)
                    log.info("Stop monitor service success")
                finally:
                    win32serviceutil.RemoveService(Utils.monitor_service_name)
                    log.info("Remove monitor service success")
        except Exception as err:
            log.error(f'Remove monitor service fail:{err}')

        self.stop_process(Utils.monitor_file)
        self.stop_process(Utils.rdagent_file)
        self.stop_process(Utils.log_package_file)

        if any((Utils.check_process(Utils.monitor_file),
                Utils.check_process(Utils.rdagent_file),
                Utils.check_process(Utils.log_package_file))):
            print("Failed to stop the client assistant.")
            log.info('Failed to stop the client assistant.')
            return False
        else:
            print("Succeeded in stopping the client assistant.")
            log.info('Succeeded in stopping the client assistant.')
            return True


if __name__ == '__main__':
    stop = AgentAssistStopProcess()
    if len(sys.argv) < 2:
        if not stop.stop_all_process():
            sys.exit(1)
        else:
            sys.exit(0)
    elif len(sys.argv) > 2 or sys.argv[1] not in (
            'monitor', 'rdagent', 'log_package'):
        print('illegal parameter: please enter one  parameter from '
              '"monitor, rdagent, log_package"')
    else:
        stop.stop_process(sys.argv[1] + '.py')
