#!/usr/bin/python3
import os
import shlex
import shutil
import threading
import time

from common.logutils import Logger
from common.utils import Utils, log_with_exception
from common.common_define import ArchiveType, CommonDefine, Permission

install_path = Utils.get_install_path()
grade_pkg_path = Utils.get_pkg_root_path()
AGENT_ASSIST_UPGRADE_LOG = 'upgrade.log'
NEW_PKG_NAME = 'AgentAssistImage'
SHELL_SCRIPT_PATH = os.path.realpath(os.path.join(grade_pkg_path,
                                                  NEW_PKG_NAME, 'upgrade.sh'))
BAT_SCRIPT_PATH = os.path.realpath(os.path.join(grade_pkg_path,
                                                NEW_PKG_NAME, 'upgrade.bat'))
logging = Logger().get_logger(log_name=AGENT_ASSIST_UPGRADE_LOG,
                              log_path=grade_pkg_path)


class AgentAssistUpgradeHandle(object):
    def __init__(self):
        self._pkg_name = ''
        self._log_path = os.path.join(grade_pkg_path, AGENT_ASSIST_UPGRADE_LOG)

    def _set_upgrade_pkg_name(self, upgrade_pkg_name):
        self._pkg_name = upgrade_pkg_name

    def _unzip_upgrade_pkg(self):
        pkg_type = ArchiveType.TAR
        if CommonDefine.IS_WINDOWS:
            pkg_type = ArchiveType.ZIP

        Utils.unpack(self._pkg_name, grade_pkg_path, pkg_type)

        try:
            os.chdir(grade_pkg_path)
            if os.path.isfile(self._pkg_name):
                if not CommonDefine.IS_WINDOWS:
                    os.chmod(self._pkg_name, Permission.PERMISSION_700)
                os.remove(self._pkg_name)
                logging.info(f'[{self._pkg_name}] is deleted successfully .')
        except Exception as err:
            logging.error(f"Failed to delete the file [{self._pkg_name}]. {err}")

        return True

    def start_upgrade(self, upgrade_pkg_name):
        os.chdir(grade_pkg_path)
        if CommonDefine.IS_WINDOWS:
            pkg_name = upgrade_pkg_name.split('.zip')
        else:
            pkg_name = upgrade_pkg_name.split('.tar')

        if len(pkg_name) == 0:
            return False
        os.rename(pkg_name[0], NEW_PKG_NAME)

        steps = 'UPGRADE_FILE'
        return self.exec_scripting(steps)

    def mod_scripts_per(self):
        upgrade_file = os.path.join(grade_pkg_path, NEW_PKG_NAME,
                                    'scripts/upgrade.py')
        os.chmod(upgrade_file, Permission.PERMISSION_500)
        os.chmod(SHELL_SCRIPT_PATH, Permission.PERMISSION_500)

    def exec_scripting(self, steps):
        logging.info(f'start exec_scripting. steps:{steps}')
        if CommonDefine.IS_WINDOWS:
            ret, res = Utils.execute_cmd([f'{BAT_SCRIPT_PATH}',
                                          f'{install_path}', f'{steps}'])
        else:
            self.mod_scripts_per()
            ret, res = Utils.execute_cmd(["bash", f"{SHELL_SCRIPT_PATH}", f"{install_path}", f"{steps}"])
        logging.info(f'ret = {ret}, ret_type = {type(ret)}, res = {res}, '
                     f'steps = {steps}')
        if ret != 0:
            logging.error(f'Failed to execute the upgrade script. '
                          f'steps:{steps}')
            return False
        logging.info('Script invoking successfully.')
        return True

    @log_with_exception(logging)
    def upgrade_file(self, upgrade_pkg_name):
        logging.info('start to upgrade')
        os.chdir(grade_pkg_path)
        if not os.path.isfile(upgrade_pkg_name):
            logging.error("The upgrade package does not exist.")
            return False

        self._set_upgrade_pkg_name(upgrade_pkg_name)

        file_path = os.path.join(grade_pkg_path, NEW_PKG_NAME)

        try:
            if os.path.exists(file_path):
                shutil.rmtree(file_path)
        except Exception as err:
            logging.error(f"Failed to delete AgentAssistImage folder. {err}")
            return False

        self._unzip_upgrade_pkg()
        if not self.start_upgrade(upgrade_pkg_name):
            return False

        return True

    def _archive_upgrade_logs(self):
        log_path = os.path.join(install_path, 'AgentAssist/log',
                                AGENT_ASSIST_UPGRADE_LOG)
        if os.path.isfile(self._log_path):
            try:
                shutil.copyfile(self._log_path, log_path)
                if not CommonDefine.IS_WINDOWS:
                    os.chmod(log_path, Permission.PERMISSION_600)
                os.remove(self._log_path)
            except Exception as err:
                logging.error(f"Failed to copy Log. {err}")

    def _delete_residual_file(self):
        logging.info(f"start _delete_residual_file.")
        self._archive_upgrade_logs()
        file_path = os.path.join(grade_pkg_path, NEW_PKG_NAME)
        os.chdir(grade_pkg_path)

        try:
            if os.path.exists(file_path):
                shutil.rmtree(file_path)
        except Exception as err:
            logging.error(f"Failed to delete AgentAssistImage folder. {err}")

    def upgrade_handle(self, upgrade_pkg_name):
        if not self.upgrade_file(upgrade_pkg_name):
            self._delete_residual_file()
            return False

        t = threading.Thread(target=_upgrade_process)
        t.start()
        return True


@log_with_exception(logging)
def _upgrade_process():
    upgrade = AgentAssistUpgradeHandle()
    time.sleep(1)
    steps = 'UPGRADE_PROCESS'
    upgrade.exec_scripting(steps)
