import datetime
import hashlib
import hmac
import ipaddress
import json
import os
import shutil
import threading
from kmc import kmc

import psutil
import requests
from kmc import kmc

from common.common_define import CommonDefine
from common.configreader import g_cfg_agentassist, g_cfg_cert, g_cfg_iam
from common.logutils import Logger
from common.utils import Utils
from message.tcp.funchandle import FuncMap
from message.tcp.messagequeue import MessageQueue

MSG_HEAD_SIZE = 28
MSG_BODY_FILEHEAD_SIZE = 332
MSG_BODY_FILEBODY_SIZE = 2097152

msg_queue = MessageQueue()
func_map = FuncMap()
logger = Logger().get_logger()

try:
    update_kmc_time = g_cfg_cert.get_int_option('kmc', 'update_kmc_time_day')
    if update_kmc_time <= 0 or update_kmc_time > 9999:
        raise ValueError("Update kmc time must be in the range:(0,9999].")
except Exception as err:
    logger.error(f"Read conf error:{err}.")
    update_kmc_time = 365
    g_cfg_cert.set_option("kmc", "update_kmc_time_day", str(update_kmc_time))

SUPPORT_CIPHER_TYPES = ['generalCipher', 'SMCompatible']


def get_cipher_type():
    cipher_type = None

    try:
        cipher_type = g_cfg_agentassist.get_option('proxy', 'cipher_type')
    except Exception as error:
        logger.error(f"Get cipher type failed, err:{error}. Support type {SUPPORT_CIPHER_TYPES}.")
        return cipher_type

    if cipher_type not in SUPPORT_CIPHER_TYPES:
        logger.error(f"The cipher type is {cipher_type}, not in {SUPPORT_CIPHER_TYPES}. Will set None.")
        cipher_type = None

    return cipher_type


def set_cipher_type(cipher_type):
    if cipher_type not in SUPPORT_CIPHER_TYPES:
        logger.error(f"The param cipher_type is not in {SUPPORT_CIPHER_TYPES}.")
        return False

    try:
        g_cfg_agentassist.set_option('proxy', 'cipher_type', str(cipher_type))
        return True
    except Exception as expt:
        logger.error(f"Set param 'cipher_type' to config error:{expt}.")
        return False


class TcpUtils(object):
    cloud_info_empty = {"project_id": "", "environ": "", "az": "", "server_id": ""}

    @staticmethod
    def get_cloud_info(hwcloud_ip):
        """获取服务器的UUID"""
        url = 'http://' + hwcloud_ip + '/openstack/latest/meta_data.json'
        try:
            resp = requests.get(url=url, timeout=5)
        except Exception as exception:
            logger.error(f"Request url failed: {exception}.")
            return TcpUtils.cloud_info_empty
        if resp:
            resp_json = resp.content.decode()
            resp_dict = json.loads(resp_json)

            # 获取project_id
            project_id = resp_dict.get("project_id") if resp_dict.get("project_id") else ""
            # 获取environ：1-ecs，2-bms
            environ = 1
            meta_info = resp_dict.get("meta")
            if meta_info:
                cloud_service_type = meta_info.get("metering.cloudServiceType")
                if cloud_service_type and "ec" in cloud_service_type:
                    environ = 1
                resource_type = meta_info.get("metering.resourcetype")
                if resource_type and resource_type == "__type_baremetal":
                    environ = 2
            # 获取availability_zone
            available_zone = resp_dict.get("availability_zone")
            availability_zone = available_zone if available_zone else ""
            # 获取service_id
            srv_id = resp_dict.get("uuid")
            service_id = srv_id if srv_id else ""

            return {"project_id": project_id, "environ": environ, "az": availability_zone, "server_id": service_id}
        else:
            return TcpUtils.cloud_info_empty

    @staticmethod
    def _item_agent_info_list(agent_info_list, results):
        for agent_info in agent_info_list:
            for file_name, info_dict_list in agent_info.items():
                version = agent_info.get(file_name).get("agent_version")
                client_id = agent_info.get(file_name).get("agent_id")
                results.append((version, client_id))
        return results

    @staticmethod
    def _parser_agent_info_str(agent_info_str):
        if not CheckArgs.check_arg_type(agent_info_str, (str, dict)):
            logger.error(f"{agent_info_str} is not json or dict.")
            return []
        if CheckArgs.check_arg_type(agent_info_str, str):
            try:
                return json.loads(agent_info_str).get("agent_list")
            except Exception as exception:
                logger.error(f"The {agent_info_str} is not json or not in dict form: {exception}.")
                return []
        if CheckArgs.check_arg_type(agent_info_str, dict):
            return agent_info_str.get("agent_list")

    @staticmethod
    def get_subagent_info():
        agent_info_file = FileIOUtil.get_path(Utils.get_agent_assist_root_path(), 'conf/agent_list.json')
        if FileIOUtil.is_path_exists(agent_info_file):
            agent_info_str = str()
            results = []
            with open(agent_info_file, "r") as f:
                agent_info_str += f.read()

            if agent_info_str:
                agent_info_list = TcpUtils._parser_agent_info_str(agent_info_str)
                return TcpUtils._item_agent_info_list(agent_info_list, results)
            else:
                logger.error(f"The {agent_info_file} exists but empty.")
                return
        else:
            logger.error(f"The {agent_info_file} not found.")
            return

    @staticmethod
    def handle_process(p, agents):
        rdagent_pyfile = [cmd for cmd in p.cmdline() if cmd.lower().endswith('rdagent.py')]
        if rdagent_pyfile:
            subagent_info = TcpUtils.get_subagent_info()
            if subagent_info:
                version, client_id = subagent_info[0]
                agents.append(
                    {"type": "0", "version": version, "client_id": client_id,
                     "state": "online" if p.status() == "running" or p.status() == "sleeping" else "offline"}
                )
        if 'esfdaemon' in p.name():
            subagent_info = TcpUtils.get_subagent_info()
            if subagent_info and len(subagent_info) > 1:
                version, client_id = subagent_info[1]
                agents.append(
                    {"type": "1", "version": version, "client_id": client_id,
                     "state": "online" if p.status() == "running" or p.status() == "sleeping" else "offline"}
                )

    @staticmethod
    def make_pipe_cmds(key_word):
        return f'tasklist|findstr "{key_word}"' if CommonDefine.IS_WINDOWS else \
            f"pgrep -f {key_word}"

    @staticmethod
    def call_run_pipe_cmds(cmds):
        try:
            cmd_list = Utils.change_cmds_format(cmds)
            _, procs = Utils.execute_cmds(cmd_list)

            if CommonDefine.IS_WINDOWS:
                proc_li = list()
                for proc in procs:
                    proc_li.append(int(proc.split()[1]))
                return sorted(proc_li)
            else:
                procs = list(map(int, procs))
                return sorted(procs)
        except Exception as exception:
            logger.error(f"Run pipe cmd failed: {exception}.")

    @staticmethod
    def get_host_agents():
        """获取主机上所有的子agents列表"""
        pids = []
        process_list = ['python', 'esfdaemon'] if CommonDefine.IS_WINDOWS else ['AgentAssistPython']
        for process_name in process_list:
            cmds = TcpUtils.make_pipe_cmds(process_name)
            pids_of_process_name = TcpUtils.call_run_pipe_cmds(cmds)
            pids += pids_of_process_name
        agents = []
        for pid in pids:
            try:
                p = psutil.Process(pid)
                TcpUtils.handle_process(p, agents)
            except Exception as exception:
                logger.error(f"Get process failed: {exception}.")
                continue
        return agents

    @staticmethod
    def get_host_info():
        """获取主机信息，每一分钟上报一次。字段定义：
        字段名称            类型        字段含义
        ip                 String     主机IP
        environ            String     主机环境
        name               String     主机名
        os_type            String     主机OS类型
        os_version         String     主机OS版本
        os_architecture    String	  x86_64或aarch64
        server_id          String     服务器的UUID
        agents             列表        子agent列表
        #######################################
        子agent列表：
        [{"agent_name":"",
          "agent_type":"",
          "agent_version":"",
          "agent_id":""，
          "agent_status":""}]

        agent_type:0-agentframework,1-subagent
        agent_status：
          subagent：在线、离线、安装/升级/卸载成功/失败
          agentframework：在线/升级成功/卸载成功
        """
        active_code = Utils.get_host_active_code()
        ip = Utils.get_host_ip()
        name = Utils.get_host_name()
        try:
            os_version = Utils.get_host_os_version()
        except Exception as error:
            logger.error(f"Get host os version failed, {error}.")
            os_version = ''
        os_architecture = Utils.get_host_os_architecture()
        agents = TcpUtils.get_host_agents()
        host_info = {"active_code": active_code, "ip": ip, "name": name, "os_type": CommonDefine.SYS_TYPE,
                     "os_version": os_version, "architecture": os_architecture, "agents": agents}
        cloud_info = TcpUtils.get_cloud_info(g_cfg_agentassist.get_option('proxy', 'hwc_ip'))
        host_info.update(cloud_info)
        return host_info

    @staticmethod
    def get_ak_sk_info():
        """获取配置文件中ak,sk的信息"""
        cipher_type = get_cipher_type()
        ak_info = g_cfg_iam.get_option('iam', 'access_key')
        encrypt_sk_info = g_cfg_iam.get_option('iam', 'secret_key')
        decrypt_sk_info = kmc.API().decrypt(0, encrypt_sk_info, cipher_type)
        cloud_info = TcpUtils.get_cloud_info(g_cfg_agentassist.get_option('proxy', 'hwc_ip'))
        utc = datetime.datetime.utcnow()
        datestamp = utc.strftime('%Y%m%d')
        project_id = cloud_info.get('project_id')
        hash_sign = hmac.new(('HWS' + decrypt_sk_info).encode('utf-8'), datestamp.encode('utf-8'), hashlib.sha256)\
            .hexdigest()
        return {"access_key": ak_info, "hash_sign": hash_sign, "project_id": project_id}

    @staticmethod
    def delete_temp_file(tmp_file):
        file_base_name = os.path.basename(tmp_file)
        if not os.path.isfile(tmp_file):
            logger.error(f"{file_base_name} is not file.")
            return False
        try:
            os.remove(tmp_file)
            logger.info(f"Remove {file_base_name} success.")
            return True
        except Exception as error:
            logger.error(f"Remove {file_base_name} fail:{error}.")
            return False


class FileIOUtil(object):

    @staticmethod
    def read_file(file_path):
        with open(file_path, 'r') as f:
            return f.read().strip()

    @staticmethod
    def write_file(file_path, content):
        with open(file_path, 'w') as f:
            f.write(content)

    @staticmethod
    def copy_file(src, dest):
        if not isinstance(src, str) or not isinstance(dest, str):
            return False
        if FileIOUtil.is_path_exists(src):
            shutil.copyfile(src, dest)
            return True
        else:
            return False

    @staticmethod
    def remove_files(files):
        removed_files_li = []
        not_removed_files_li = []
        if not isinstance(files, (list, tuple)):
            logger.error(f"{type(files)} is not list or tuple.")
            return False
        for file in files:
            base_file_name = os.path.basename(file)
            if FileIOUtil.is_path_exists(file):
                removed_files_li.append(base_file_name)
                os.remove(file)
            else:
                logger.error(f"File path is not exist: {base_file_name}.")
                not_removed_files_li.append(base_file_name)
                continue
        logger.info(f"Remove files:{removed_files_li} success,fail to remove files:{not_removed_files_li}.")
        return True

    @staticmethod
    def rename_file(src, dest):
        if not isinstance(src, str) or not isinstance(dest, str):
            return False
        if FileIOUtil.is_path_exists(src):
            if FileIOUtil.is_path_exists(dest):
                os.remove(dest)
            os.rename(src, dest)
            return True
        else:
            return False

    @staticmethod
    def is_file(file_path):
        return os.path.isfile(file_path)

    @staticmethod
    def get_path(path, target=''):
        return os.path.join(path, target)

    @staticmethod
    def split_path(path):
        return os.path.split(path)

    @staticmethod
    def is_path_exists(path):
        return os.path.exists(path)


class TimerTask(threading.Timer):
    def run(self):
        while not self.finished.is_set():
            self.function(*self.args, **self.kwargs)
            self.finished.wait(self.interval)


class CheckArgs(object):
    @staticmethod
    def check_arg_type(arg, arg_type):
        return isinstance(arg, arg_type)

    @staticmethod
    def check_int_range(arg, arg_range):
        if not isinstance(arg, int) or not isinstance(arg_range, (tuple, list)):
            return False
        return arg in arg_range

    @staticmethod
    def check_ip(ip):
        if not CheckArgs.check_arg_type(ip, (str, int)):
            return False
        try:
            ipaddress.ip_address(ip)
            return True
        except Exception:
            return False

    @staticmethod
    def check_port(port):
        if not CheckArgs.check_arg_type(port, int):
            return False
        return 0 <= port <= 65535
