import json
import os
import shutil

from bin.agentmanager.sub_install import SubInstallHandle
from bin.agentmanager.sub_install import clear_useless_pkg
from bin.agentmanager.sub_install import delete_all_temp_file
from bin.agentmanager.sub_uninstall import SubUninstallHandle
from bin.agentmanager.sub_upgrade import SubUpgradeHandle
from common.logutils import Logger, SUBAGENT_INSTALL_LOG
from common.utils import Utils

ASSIST_PATH = Utils.get_agent_assist_root_path()
AGENT_PATH = Utils.get_sub_agent_root_path()
PKG_PATH = Utils.get_pkg_root_path()
log = Logger().get_logger(SUBAGENT_INSTALL_LOG)


class SubAgentMessage(object):
    def __init__(self):
        self._nul_ret = {"job_status": "failed",  # 执行状态
                         "type": "1",  # agent类型：0-agentframework,1-subagent
                         "version": "",  # agent版本
                         "client_id": "",  # agent id
                         "error_code": "CSBS.9999"
                         }
        self._file_name = ""
        self._machine_code = ""
        self._agent_version = ""

    @clear_useless_pkg
    @delete_all_temp_file(check_install=False)
    def _read_agent_info(self, body_str):
        """
        The decorator needs to be use body_str
        """
        agent_info_file = os.path.join(ASSIST_PATH, 'conf/agent_list.json')
        try:
            with open(agent_info_file, "r", encoding='utf8') as f:
                info_dict = json.load(f)
        except Exception as err:
            log.error(f"Read agent info file failed, error[{err}].")
            return False
        if not isinstance(info_dict, dict):
            log.error("Agent info file type error.")
            return False
        info_list = info_dict.get("agent_list")
        if not isinstance(info_list, list):
            log.error("Agent list type error.")
            return False
        for info in info_list:
            if not isinstance(info, dict):
                log.error("Agent info type error.")
                return False
            if self._file_name in info:
                agent_info_dict = info.get(self._file_name)
                if not isinstance(agent_info_dict, dict):
                    log.error("Inner agent info type error.")
                    return False
                self._agent_version = agent_info_dict.get("agent_version")
                self._machine_code = agent_info_dict.get("agent_id")
                return True
        else:
            log.error(f"No match record.")
            return False

    @staticmethod
    def check_is_installed():
        process_running = Utils.check_process_status('esfdaemon')
        if process_running:
            if os.path.exists(AGENT_PATH):
                log.info(f"The installation directory already exists. Please "
                         f"enter {AGENT_PATH},uninstall it before install.")
                return True, True
            else:
                log.error(f"The installation directory not exists but the SubAgent "
                          f"process exists. please manual uninstall it before install.")
                return True, False
        else:
            if os.path.exists(AGENT_PATH):
                log.warning("Old install catalog remain, will remove it.")
                return False, True
        return False, False

    def get_subagent_version(self, body_str):
        if not self._read_agent_info(body_str):
            log.error("Read agent info failed.")
            return self._nul_ret
        suc_val = {"job_status": "success",  # agent安装状态
                   "type": "1",  # agent类型：0-agentframework,1-subagent
                   "version": self._agent_version,  # agent版本
                   "client_id": self._machine_code}  # agent id
        return suc_val

    def install_subagent(self, body_dict):
        log.info("Begin to install the SubAgent.")
        self._file_name = body_dict.get("file_name")
        body_str = json.dumps(body_dict)
        process_exits, dir_exists = SubAgentMessage.check_is_installed()
        if process_exits:
            if dir_exists:
                log.info(f"Manual installed SubAgent exists. please uninstall it before install.Or please select "
                         f"upgrade.")
                return self.get_subagent_version(body_str)
            log.error(f"Manual installed SubAgent exists. please uninstall it before install.")
            return self._nul_ret
        else:
            if dir_exists:
                shutil.rmtree(AGENT_PATH)
                log.info(f"Remove SubAgent success!")
            install_handler = SubInstallHandle()
            install_handler_result = install_handler.install_handle(body_str)
            if isinstance(install_handler_result, bool):
                if install_handler_result:
                    return self.get_subagent_version(body_str)
            elif isinstance(install_handler_result, str):
                self._nul_ret["error_code"] = install_handler_result
        log.error("Install SubAgent failed.")
        return self._nul_ret

    def uninstall_subagent(self, body_dict):
        log.info("Begin to uninstall the SubAgent.")
        file_name = body_dict.get("file_name")
        if not Utils.check_file_name(file_name):
            log.error(f"Input wrong name, pls check, file_name:{file_name}.")
            return self._nul_ret
        uninstall_handler = SubUninstallHandle()
        uninstall_handler_result = uninstall_handler.uninstall_handle(file_name)
        if isinstance(uninstall_handler_result, bool):
            if uninstall_handler_result:
                ret_val = {"job_status": "success",  # agent卸载状态
                           "type": "1",  # agent类型：0-agentframework,1-subagent
                           "version": "",  # agent版本
                           "client_id": ""}  # agent id
                return ret_val
        elif isinstance(uninstall_handler_result, str):
            self._nul_ret["error_code"] = uninstall_handler_result
        log.error("Uninstall SubAgent failed.")
        return self._nul_ret

    def upgrade_subagent(self, body_dict):
        log.info("Begin to upgrade the SubAgent.")
        self._file_name = body_dict.get("file_name")
        body_str = json.dumps(body_dict)
        upgrade_handler = SubUpgradeHandle()
        upgrade_handler_result = upgrade_handler.upgrade_handle(body_str)
        # 返回值可能是True、False、error_code不能直接使用 if upgrade_handler_result
        if isinstance(upgrade_handler_result, bool):
            if upgrade_handler_result:
                return self.get_subagent_version(body_str)
        elif isinstance(upgrade_handler_result, str):
            self._nul_ret["error_code"] = upgrade_handler_result  # return error_code
        log.error("Upgrade SubAgent failed.")
        return self._nul_ret
