#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
"""提供运行系统调用的命令封装，如果不需要直接使用subprocess，推荐使用此模块的API"""

import os
import subprocess
import typing
from collections import namedtuple

from hw_bcmanager_safe.commands import safe_subprocess as safe_subprocess

ExecuteResult = namedtuple('ExecuteResult', ('exit_code', 'std_out', 'std_error'))


def _build_execute_result(exit_code, std_out, std_error, parse_output_to_string=True):
    if parse_output_to_string:
        std_out = std_out.decode().rstrip(os.linesep)
        std_error = std_error.decode().rstrip(os.linesep)

    return ExecuteResult(exit_code, std_out, std_error)


def run_command(cmd: [typing.List], timeout=300, sensitive_args=None, parse_output=True):
    """以安全方式启动新的进程运行命令，可以避免命令注入，敏感信息以回显方式泄露。

    :param cmd: 需要执行的命令，参数类型是列表。如果执行的是交互式输入（输入密码后需要敲回车），那么需要把换行符（os.linesep）放到
    命令参数列表中，模拟手动敲回车的场景。
    :param timeout: 设置命令超时时间，默认300秒，如果命令执行时间超时，子进程将被杀死，并抛出TimeoutExpired异常。
    :param sensitive_args: 敏感参数列表。当命令中存在密码等敏感参数时，需要从cmd中拆分出来，避免ps回显输出。
    :param parse_output: 是否把标准输出和错误转换为str类型并擦除结尾的最后一个换行符，如果指定为False，则返回原始的byte类型。
    :return: 具名元组，内容为：（进程执行结果，标准输出，错误输出）。
    """
    process = safe_subprocess.Popen(cmd, stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE)
    if isinstance(sensitive_args, (list, tuple)):
        encoded_args = [str(sensitive_arg).encode() for sensitive_arg in sensitive_args]
        process.stdin.writelines(encoded_args)

    std_out, std_error = process.communicate(None, timeout)
    return _build_execute_result(process.returncode, std_out, std_error, parse_output)


def run_commands(commands: [typing.List[typing.List]], timeout=300, parse_output=True):
    """以安全方式启动新的进程运行命令，适用于管道连接的操作，可以避免命令注入，敏感信息以回显方式泄露。

    :param commands: 需要执行的命令，参数类型是列表的列表，例如[['cat', 'run.log'], ['grep', 'error']]。
    :param timeout: 设置命令超时时间，默认300秒，如果命令执行时间超时，子进程将被杀死，并抛出TimeoutExpired异常。
    :param parse_output: 是否把标准输出和错误转换为str类型并擦除结尾的最后一个换行符，如果指定为False，则返回原始的byte类型。
    :return: 具名元组，内容为：（进程执行结果，标准输出，错误输出）。
    """
    command_stdin = subprocess.PIPE
    final_process = None
    for command in commands:
        final_process = safe_subprocess.Popen(command, stdin=command_stdin, stdout=subprocess.PIPE,
                                              stderr=subprocess.PIPE)
        command_stdin = final_process.stdout

    std_out, std_error = final_process.communicate(None, timeout)
    return _build_execute_result(final_process.returncode, std_out, std_error, parse_output)
