"""对Python原生的subprocess进行安全封装，防止参数注入"""
import os
import re
import subprocess
import typing

# 路径分隔符，在linux下为/，在windows下为\\
path_separator = re.escape(os.path.sep)
# 允许传入的命令正则表达式，这里为目前已识别到的最小安全白名单全集
CMD_ALLOW_PATTERN = re.compile(f"^[:a-zA-Z0-9{path_separator}=. ,_-]+$")

# 命令注入符号的黑名单
SYMBOL_BLACK_LIST = ("-c", "--c", "=")


def check_shell_inject_risk(str_arg: str):
    """检查给定的字符串参数是否存在shell命令注入的风险，判定风险的依据来源于版本积累的经验。

    :param str_arg:待检查的命令字符串
    :return :如果该命令存在风险，返回True，如果没有风险返回False。
    """
    if not isinstance(str_arg, str) or not str_arg:
        return False

    if str_arg in SYMBOL_BLACK_LIST or not re.match(CMD_ALLOW_PATTERN, str_arg):
        return True

    return False


def _check_shell_inject_args(cmd_list):
    if not isinstance(cmd_list, (list, tuple)):
        raise ValueError(f"Command {cmd_list} must be list or tuple.")

    for cmd in cmd_list:
        if check_shell_inject_risk(cmd):
            raise ValueError(f"Command {cmd_list} may have shell inject risk, terminate it.")


class Popen(subprocess.Popen):
    """返回原生的subprocess.Popen对象，但对输入的命令参数做了严格的参数校验，并且指定shell=False。"""

    def __init__(self, args: [typing.List, typing.Tuple], stdin=None, stdout=None, stderr=None, **kwargs):
        """以安全方式启动新的进程运行命令，适用于管道连接的操作，可以避免命令注入，敏感信息以回显方式泄露。

        :param args: 列表或者元组形式的参数，例如['ls', '-a']。
        :param stdin: 表示子程序的标准输入，如果是None，不会做任何重定向工作，子进程的文件描述符会继承父进程的，
        如果是PIPE，则表示需要创建一个新的管道，
        :param stdout: 表示子程序的标准输出，如果是None，不会做任何重定向工作，子进程的文件描述符会继承父进程的，
        如果是PIPE，则表示需要创建一个新的管道，
        :param stdout: 表示子程序的标准错误，如果是None，不会做任何重定向工作，子进程的文件描述符会继承父进程的，
        如果是PIPE，则表示需要创建一个新的管道，
        :kwargs:key-value形式的输入，与原生subprocess保持一致
        :return: 原生subprocess的Popen对象
        """
        _check_shell_inject_args(args)
        kwargs.pop('shell', None)
        super().__init__(args, stdin=stdin, stdout=stdout, stderr=stderr, shell=False, **kwargs)
