#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import sys

import mock
import pytest

from hw_bcmanager_safe.commands import command
from hw_bcmanager_safe.commands.safe_subprocess import check_shell_inject_risk


class TestCommand(object):
    """测试command模块的接口。"""

    @pytest.mark.commands
    def test_run_command_success(self):
        """
        测试场景： 测试使用safe_command模块运行Python命令
        测试预期：命令执行成功,std_out返回正确
        """
        with mock.patch('subprocess.Popen.communicate') as f:
            f.return_value = (b'this is python', b'')
            execute_result = command.run_command(['python'])
            assert execute_result.std_out == 'this is python'

    @pytest.mark.commands
    def test_run_command_with_inject_args(self):
        """
        测试场景： 测试使用safe_command模块运行带反引号的命令.
        测试预期：命令不会执行，抛出ValueError异常
        """
        with pytest.raises(ValueError):
            command.run_command(['python', '`rm -rf`'])

    @pytest.mark.commands
    def test_run_commands_success(self):
        """
        测试场景： 测试使用safe_command模块运行带管道符的多个命令
        测试预期：命令执行成功,stdout返回的是命令执行的
        """
        with mock.patch('subprocess.Popen.communicate') as f:
            f.return_value = (b'default error', b'')
            execute_result = command.run_commands([['python', 'a.py'], ['python', 'b.py']])
            assert execute_result.std_out == 'default error'

    @pytest.mark.commands
    def test_run_command_success_with_sensitive_args(self):
        """
        测试场景： 测试使用safe_command模块运行带敏感信息的命令
        测试预期：命令执行成功,stdout返回的是命令执行的
        """
        with mock.patch('subprocess.Popen.communicate') as f:
            f.return_value = (b'success', b'')
            execute_result = command.run_command(["python", "-pass"], sensitive_args=['mima'])
            assert execute_result.std_out == 'success'


class TestShellInject(object):
    """测试检测命令注入的公共方法"""

    @pytest.mark.inject
    @pytest.mark.skipif(sys.platform == 'win32', reason='该用例包含了linux路径分隔符，只在linux场景有效')
    @pytest.mark.parametrize('cmd', ['python3', '/opt/bcm/collect.sh', 'cp -rf /home/dj /opt/ccs', None, '', ' '])
    def test_no_risk_cmd_on_linux(self, cmd):
        """
        测试场景： 测试在linux操作系统上可执行的命令，命令只包含常规的字符
        测试预期： 测试通过，检查这些命令都没有风险，返回False
        """
        assert check_shell_inject_risk(cmd) is False

    @pytest.mark.inject
    @pytest.mark.skipif(sys.platform == 'win32', reason='该用例包含的不满足于白名单的字符')
    @pytest.mark.parametrize('cmd', ['bash -c', 'c:\\windows\\python.exe', '$$dir'])
    def test_risk_cmd_on_linux(self, cmd):
        """
        测试场景： 测试在linux操作系统上可执行的命令，命令只包可能存在注入的字符
        测试预期： 测试通过，检查这些命令都没有风险，返回True
        """
        assert check_shell_inject_risk(cmd) is True

    @pytest.mark.inject
    @pytest.mark.skipif(sys.platform == 'linux', reason='该用例包含了windows路径分隔符，只在windows场景有效')
    @pytest.mark.parametrize('cmd', ['python3', 'c:\\windows\\python.exe', 'c:\\agent_assist\\monitory.py', None, ''])
    def test_no_risk_cmd_on_linux(self, cmd):
        """
        测试场景： 测试在linux操作系统上可执行的命令，命令只包含常规的字符
        测试预期： 测试通过，检查这些命令都没有风险，返回False
        """
        assert check_shell_inject_risk(cmd) is False

    @pytest.mark.inject
    @pytest.mark.skipif(sys.platform == 'linux', reason='该用例包含的不满足于白名单的字符')
    @pytest.mark.parametrize('cmd', ['bash -c' 'c:\\windows/python.exe', '$$dir'])
    def test_risk_cmd_on_linux(self, cmd):
        """
        测试场景： 测试在linux操作系统上可执行的命令，命令只包可能存在注入的字符
        测试预期： 测试通过，检查这些命令都没有风险，返回True
        """
        assert check_shell_inject_risk(cmd) is True
